/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/impl/level_file_xml_writer.tpp
 * \brief Implementation of the template methods of the
 *        bf::level_file_xml_writer class.
 * \author Julien Jorge
 */

#include "bf/stream_conv.hpp"
#include "bf/xml_exception.hpp"
#include "bf/wx_facilities.hpp"

#include <claw/assert.hpp>
#include <claw/logger.hpp>
#include <sstream>

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a node for a simple value.
 * \param os The stream in which we save the value.
 * \param node_name The name of the xml node.
 * \param v The value to write.
 */
template<typename Type>
void bf::level_file_xml_writer::value_to_xml
( std::ostream& os, const std::string& node_name, const Type& v ) const
{
  os << "        <" << node_name << " value='" << v.get_value() << "'/>\n";
} // level_file_xml_writer::value_to_xml()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of simple type.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 * \param node_name The name of the xml node.
 */
template<typename Type>
void bf::level_file_xml_writer::save_value
( std::ostream& os, const std::string& field_name,
  const item_instance& item, const std::string& node_name ) const
{
  Type v;
  item.get_value( field_name, v );
  value_to_xml( os, node_name, v );
} // level_file_xml_writer::save_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of list of simple type.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 * \param node_name The name of the xml node.
 */
template<typename Type>
void bf::level_file_xml_writer::save_value_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item, const std::string& node_name ) const
{
  std::list<Type> v;
  typename std::list<Type>::const_iterator it;

  item.get_value( field_name, v );

  for (it=v.begin(); it!=v.end(); ++it)
    value_to_xml( os, node_name, *it );
} // level_file_xml_writer::save_value_list()
