/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/level_file_xml_reader.hpp
 * \brief A class for reading source level files.
 * \author Julien Jorge
 */
#ifndef __BF_LEVEL_FILE_XML_READER_HPP__
#define __BF_LEVEL_FILE_XML_READER_HPP__

#include "bf/level.hpp"

#include <iostream>
#include <wx/xml/xml.h>

namespace bf
{
  /**
   * \brief A class for reading source level files.
   * \author Julien Jorge
   */
  class level_file_xml_reader
  {
  public:
    level* load( const wxString& file_path ) const;

    static unsigned int read_uint( wxXmlNode* node, const std::string& prop );
    static unsigned int
    read_uint_opt( wxXmlNode* node, const std::string& prop, unsigned int def );
    static double read_real_opt
    ( wxXmlNode* node, const std::string& prop, double def );
    static bool read_bool_opt
    ( wxXmlNode* node, const std::string& prop, bool def );

  private:
    level* load_level( wxXmlNode* node ) const;
    void load_layers( level& lvl, wxXmlNode* node ) const;
    void load_layer( level& lvl, wxXmlNode* node ) const;
    void load_layer_content( layer& lay, wxXmlNode* node ) const;
    void load_item( layer& lay, wxXmlNode* node ) const;
    void load_fields( item_instance& item, wxXmlNode* node ) const;
    void load_field( item_instance& item, wxXmlNode* node ) const;
    void load_field
    ( item_instance& item, const type_field& f, wxXmlNode* node ) const;

    template<typename Type>
    void load_value
    ( item_instance& item, const std::string& field_name,
      const std::string& node_name, wxXmlNode* node ) const;

    template<typename Type>
    void load_value_list
    ( item_instance& item, const std::string& field_name,
      const std::string& node_name, wxXmlNode* node ) const;

  }; // class level_file_xml_reader

  /**
   * \brief A class to convert an XML node into real data.
   */
  template<typename Type>
  class xml_to_value
  {
  public:
    void operator()( Type& v, wxXmlNode* node ) const;
  }; // class xml_to_value

  /**
   * \brief A class to convert an XML node into real data. Specialisation for
   *        the bf::sprite class.
   */
  template<>
  class xml_to_value<sprite>
  {
  public:
    void operator()( sprite& v, wxXmlNode* node ) const;
  }; // struct xml_to_value [sprite]

  /**
   * \brief A class to convert an XML node into real data. Specialisation for
   *        the bf::animation class.
   */
  template<>
  class xml_to_value<animation>
  {
  public:
    void operator()( animation& v, wxXmlNode* node ) const;
  private:
    void load_frames( animation& anim, wxXmlNode* node ) const;
    void load_frame( animation& anim, wxXmlNode* node ) const;
  }; // struct xml_to_value [animation]
} // namespace bf

#include "bf/impl/level_file_xml_reader.tpp"

#endif // __BF_LEVEL_FILE_XML_READER_HPP__
