/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file animation_type.cpp
 * \brief Implementation of the mc::animation_type class.
 * \author Julien Jorge
 */
#include "mc/animation_type.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
mc::animation_type::animation_type()
  : m_flip_x(false), m_flip_y(false), m_alpha(1), m_loops(0), m_loop_back(false)
{

} // animation_type::animation_type()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param frames The frames of the animation.
 */
mc::animation_type::animation_type( const std::list< image_type >& frames )
  : m_frames(frames), m_flip_x(false), m_flip_y(false), m_alpha(1), m_loops(0),
    m_loop_back(false)
{

} // animation_type::animation_type()

/*----------------------------------------------------------------------------*/
/**
 * \brief Write the animation in the compiled file.
 * \param f The file to write in.
 */
void mc::animation_type::output( compiled_file& f ) const
{
  std::list< image_type >::const_iterator it;

  f << m_frames.size();

  for ( it=m_frames.begin(); it!=m_frames.end(); ++it)
    it->output(f);

  f << m_flip_x << m_flip_y << m_alpha << m_loops << m_loop_back;
} // animation_type::output()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the frames of the animation.
 * \param frames The new frames.
 */
void
mc::animation_type::set_frames( const std::list< image_type >& frames )
{
  m_frames = frames;
} // animation_type::set_frames()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the horizontal flip attribute of the image.
 * \param b The new value.
 */
void mc::animation_type::set_flip_x( bool b )
{
  m_flip_x = b;
} // animation_type::set_flip_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the vertical flip attribute of the image.
 * \param b The new value.
 */
void mc::animation_type::set_flip_y( bool b )
{
  m_flip_y = b;
} // animation_type::set_flip_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the alpha transparency.
 * \param alpha The new value.
 */
void mc::animation_type::set_alpha( float alpha )
{
  m_alpha = alpha;
} // animation_type::set_alpha()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of times the animation will be played.
 * \param loops The new value.
 */
void mc::animation_type::set_loops( unsigned int loops )
{
  m_loops = loops;
} // animation_type::set_loops()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the "play reverse at end" attribute of the animation.
 * \param b The new value.
 */
void mc::animation_type::set_loop_back( bool b )
{
  m_loop_back = b;
} // animation_type::set_loop_back()

