/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file application.cpp
 * \brief Implementation of the mc::application class.
 * \author Julien Jorge
 */
#include "mc/application.hpp"

#include <fstream>
#include <iostream>
#include "mc/model_compiler.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
mc::application::application( int& argc, char** &argv )
  : claw::application(argc, argv), m_xml(false),
    m_quit(false)
{
  check_arguments( argc, argv );
  store_files( argc, argv );
} // application::application()

/*----------------------------------------------------------------------------*/
/**
 * \brief Run the application : compile the files.
 */
int mc::application::run()
{
  int result = 0;

  if (!m_quit)
    {
      if ( m_files.empty() )
        {
          std::cerr << "No file to compile." << std::endl;
          result = -1;
        }
      else
        {
          std::list<std::string>::const_iterator it;

          for (it=m_files.begin(); it!=m_files.end(); ++it)
            compile_file(*it);
        }
    }

  return result;
} // application::run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Print program usage.
 */
void mc::application::help() const
{
  m_arguments.help("files...");
} // application::help()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check program arguments.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
void mc::application::check_arguments( int& argc, char** &argv )
{
  m_arguments.add_long("--xml", "Output the tree as xml.", true);
  m_arguments.add("-h", "--help", "Print this message and exit.", true);

  m_arguments.parse( argc, argv );

  if ( m_arguments.get_bool("-h") )
    {
      help();
      m_quit = true;
    }

  m_xml = m_arguments.get_bool("--xml");
} // application::application()

/*----------------------------------------------------------------------------*/
/**
 * \brief Store the names of the valid files to compile.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
void mc::application::store_files( int& argc, char** &argv )
{
  for (int i=0; i!=argc; ++i)
    {
      std::ifstream f(argv[i]);

      if (f)
        {
          m_files.push_back(argv[i]);
          f.close();
        }
      else
        std::cerr << argv[i] << ": Can't open file." << std::endl;
    }
} // application::store_files()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a file.
 * \param filename The name of the file to compile.
 */
void mc::application::compile_file( const std::string& filename ) const
{
  std::string output_name;

  get_output_name( filename, output_name );

  std::ofstream out(output_name.c_str());

  if (out)
    {
      compiled_file out_f(out);
      model_compiler comp;

      comp.run(out_f, filename, m_xml);

      out.close();
    }
  else
    std::cerr << output_name << ": Can't open output file." << std::endl;
} // application::compile_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the output file, from the name of the input file.
 * \param filename The name of the intput file.
 * \param output_name (out) The name of the output file.
 */
void mc::application::get_output_name( const std::string& filename,
                                       std::string& output_name ) const
{
  const std::string extension(".cm");

  unsigned int pos = filename.find_last_of(".");

  if (pos == std::string::npos)
    output_name = filename + extension;
  else if ( filename.substr(pos, filename.length() - pos) == extension )
    output_name = filename + extension;
  else
    output_name = filename.substr(0, pos) + extension;
} // application::get_output_name()
