/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file model.cpp
 * \brief Implementation of the mc::model class.
 * \author Julien Jorge
 */
#include "mc/model.hpp"

#include "model_code_value.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Output the model to the compiled file.
 * \param f The file to write in.
 */
void mc::model::output( compiled_file& f ) const
{
  output_actions(f);

  f << bear::model_code_value::eof;
} // model::output()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an image resource.
 * \param name The name of the resource.
 * \param value The resource to add.
 * \pre !has_image_resource(name)
 */
void mc::model::add_image_resource( const std::string& name, const gfx& value )
{
  CLAW_PRECOND( !has_image_resource(name) );

  m_image_resources[name] = value;
} // model::add_image_resource()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a sound resource.
 * \param name The name of the resource.
 * \param value The resource to add.
 * \pre !has_sound_resource(name)
 */
void
mc::model::add_sound_resource( const std::string& name, const sound& value )
{
  CLAW_PRECOND( !has_sound_resource(name) );

  m_sound_resources[name] = value;
} // model::add_sound_resource()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an action.
 * \param value The action to add.
 * \pre !has_action( value.get_name() )
 */
void mc::model::add_action( const action& value )
{
  CLAW_PRECOND( !has_action(value.get_name()) );

  m_actions[value.get_name()] = value;
} // model::add_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a graphic resource is present.
 * \param name The name of the resource to check.
 */
bool mc::model::has_image_resource( const std::string& name ) const
{
  return m_image_resources.find(name) != m_image_resources.end();
} // model::has_image_resource()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a sound resource is present.
 * \param name The name of the resource to check.
 */
bool mc::model::has_sound_resource( const std::string& name ) const
{
  return (name == "no_sound")
    || (m_sound_resources.find(name) != m_sound_resources.end());
} // model::has_sound_resource()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if an action is present.
 * \param name The name of the resource to check.
 */
bool mc::model::has_action( const std::string& name ) const
{
  return m_actions.find(name) != m_actions.end();
} // model::has_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a graphic resource.
 * \param name The name of the resource.
 */
const mc::gfx& mc::model::get_image_resource( const std::string& name ) const
{
  CLAW_PRECOND( has_image_resource(name) );

  return m_image_resources.find(name)->second;
} // model::get_image_resource()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get sound resource.
 * \param name The name of the resource.
 */
mc::sound mc::model::get_sound_resource( const std::string& name ) const
{
  CLAW_PRECOND( has_sound_resource(name) );

  if (name == "no_sound")
    return sound(name, "");
  else
    return m_sound_resources.find(name)->second;
} // model::get_sound_resource()

/*----------------------------------------------------------------------------*/
/**
 * \brief Output the actions to the compiled file.
 * \param f The file to write in.
 */
void mc::model::output_actions( compiled_file& f ) const
{
  std::map<std::string, action>::const_iterator it;

  for (it=m_actions.begin(); it!=m_actions.end(); ++it)
    it->second.output(f);
} // model::output_actions()
