/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file compiler.cpp
 * \brief Implementation of the mc::compiler class.
 * \author Julien Jorge
 */
#include "mc/model_compiler.hpp"
#include "mc/model_grammar.hpp"
#include "mc/model.hpp"
#include "mc/node_compiler/node_compiler_file.hpp"

#include <fstream>
#include <sstream>
#include <cstring>
#include <map>
#include <claw/assert.hpp>
#include <boost/spirit/tree/tree_to_xml.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a file.
 * \param f The output file.
 * \param in_file The name of the file to compile.
 * \param xml If set to true, the tree is written on standard output as XML.
 * \pre in_file exists.
 */
bool mc::model_compiler::run( compiled_file& f, const std::string& in_file,
                              bool xml )
{
  CLAW_PRECOND( file_exists(in_file) );

  std::ifstream in_f( in_file.c_str(), std::ios_base::binary );
  std::stringstream file_data;

  file_data << in_f.rdbuf();

  in_f.close();

  bool ok = run( f, file_data.str().c_str(), file_data.str().size(), in_file,
                 xml );

  return ok;
} // model_compiler::run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a file in memory.
 * \param f The output file.
 * \param file_data The content of the file to compile.
 * \param file_size Size of the \a file_data buffer.
 * \param file_name The name of the file.
 * \param xml If set to true, the tree is written on standard output as XML.
 */
bool mc::model_compiler::run( compiled_file& f, const char* file_data,
                              unsigned int file_size,
                              const std::string& file_name,
                              bool xml )
{
  CLAW_PRECOND( file_data );

  bool ok;

  boost::spirit::tree_parse_info<iterator, node_factory> info;
  model_grammar grammar;
  iterator begin(file_data, file_data + file_size, file_name), end;

  info = boost::spirit::pt_parse<node_factory>
    ( begin, end, grammar,
      boost::spirit::comment_p("/*", "*/")
      | boost::spirit::space_p );

  ok = info.full;

  if (xml)
    xml_output( info.trees );

  scan_tree( f, info.trees[0] );

  return ok;
} // model_compiler::run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the tree of the file.
 * \param f The file to write in.
 * \param node The root of the tree to compile.
 */
void mc::model_compiler::scan_tree( compiled_file& f, const tree_node& node )
{
  node_compiler_file file;
  model m;

  file.compile_node( m, node );

  m.output( f );
} // mc::model_compiler::parse_tree()

/*----------------------------------------------------------------------------*/
/**
 * \brief Output the tree as XML.
 * \param trees The trees to output.
 */
void mc::model_compiler::xml_output( const tree_match::container_t& trees ) const
{
  std::ofstream f("output.xml");
  std::map<boost::spirit::parser_id, std::string> rule_names;

  rule_names[model_grammar::id_file] = "file";
  rule_names[model_grammar::id_identifier] = "identifier";
  rule_names[model_grammar::id_filename] = "filename";
  rule_names[model_grammar::id_any_type] = "any_type";
  rule_names[model_grammar::id_u_integer_type] = "u_integer_type";
  rule_names[model_grammar::id_real_type] = "real_type";
  rule_names[model_grammar::id_bool_type] = "bool_type";
  rule_names[model_grammar::id_string_type] = "string_type";
  rule_names[model_grammar::id_image_type] = "image_type";
  rule_names[model_grammar::id_animation_type] = "animation_type";
  rule_names[model_grammar::id_image_body_item] = "image_body_item";
  rule_names[model_grammar::id_frame_list] = "frame_list";
  rule_names[model_grammar::id_resources_part] = "resources_part";
  rule_names[model_grammar::id_resource] = "resource";
  rule_names[model_grammar::id_resource_image] = "resource_image";
  rule_names[model_grammar::id_resource_sound] = "resource_sound";
  rule_names[model_grammar::id_model_part] = "model_part";
  rule_names[model_grammar::id_model_description] = "model_description";
  rule_names[model_grammar::id_action] = "action";
  rule_names[model_grammar::id_const_declaration] = "const_declaration";

  boost::spirit::tree_to_xml(f, trees, "", rule_names);

  f.close();
} // model_compiler::xml_output()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if a file exists.
 * \param filename The name of the file to test.
 */
bool mc::model_compiler::file_exists( const std::string& filename ) const
{
  std::ifstream f( filename.c_str() );
  bool result = f.is_open();

  if (result)
    f.close();

  return result;
} // model_compiler::file_exists()
