/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file model_grammar.hpp
 * \brief The grammar of a model file (without comments).
 * \author Julien Jorge
 */
#ifndef __MC_MODEL_GRAMMAR_HPP__
#define __MC_MODEL_GRAMMAR_HPP__

#include <boost/spirit.hpp>

namespace bs = boost::spirit;

namespace mc
{
  /**
   * \brief The grammar of a model file (without comments).
   * \author Julien Jorge
   */
  class model_grammar:
    public boost::spirit::grammar<model_grammar>
  {
  public:
    static const int id_file = 1;

    static const int id_identifier = 10;
    static const int id_filename   = 11;

    static const int id_any_type        = 20;
    static const int id_u_integer_type  = 21;
    static const int id_real_type       = 22;
    static const int id_bool_type       = 23;
    static const int id_string_type     = 24;
    static const int id_image_type      = 25;
    static const int id_animation_type  = 26;
    static const int id_image_body_item = 27;
    static const int id_frame_list      = 28;

    static const int id_resources_part = 40;
    static const int id_resource       = 41;
    static const int id_resource_image = 42;
    static const int id_resource_sound = 43;

    static const int id_model_part        = 50;
    static const int id_model_description = 51;
    static const int id_model_atom        = 52;
    static const int id_const_declaration = 53;
    static const int id_action            = 55;

    /**
     * \brief Grammar definition.
     */
    template<typename ScannerT> class definition
    {
    private:
      /**
       * \brief Report any error.
       */
      class error_report_parser
      {
      public:
        typedef bs::nil_t result_t;

      public:
        error_report_parser();
        error_report_parser( const std::string msg );

        template<typename LocalScanner>
        int operator()(const LocalScanner& scan, result_t& /*result*/) const;

      private:
        /** \brief A text explaining the error. */
        std::string m_msg;
      }; // class error_report_parser

      /**
       * \brief Report a "missing character" error.
       */
      class char_error_report_parser : public error_report_parser
      {
      public:
        char_error_report_parser();
        char_error_report_parser( char c );
      }; // class char_error_report_parser

      /**
       * \brief Report a "missing keyword" error.
       */
      class keyword_error_report_parser : public error_report_parser
      {
      public:
        keyword_error_report_parser();
        keyword_error_report_parser( const std::string& keyword );
      }; // class keyword_error_report_parser

      /**
       * \brief Report a type error.
       */
      class type_error_report_parser : public error_report_parser
      {
      public:
        type_error_report_parser();
        type_error_report_parser( const std::string& type );
      }; // class type_error_report_parser

      typedef bs::functor_parser<error_report_parser>         generic_error;
      typedef bs::functor_parser<char_error_report_parser>    char_error;
      typedef bs::functor_parser<keyword_error_report_parser> keyword_error;
      typedef bs::functor_parser<type_error_report_parser>    type_error;

    public:
      definition( const model_grammar& /*self*/ );

      /**
       * \brief Get the axiom rule.
       */
      const bs::rule<ScannerT, bs::parser_tag<id_file> > & start() const
      {
        return m_file;
      }

    private:
      void define_miscellanious();
      void define_types();
      void define_resources_part();
      void define_model_part();

      void initialize_error_parsers();

    private:
      /** \brief Rule matching the full file. */
      bs::rule<ScannerT, bs::parser_tag<id_file> > m_file;

      /** \brief Rule matching an identifier. */
      bs::rule<ScannerT, bs::parser_tag<id_identifier> > m_identifier;

      /** \brief Rule matching the name of a file. */
      bs::rule<ScannerT, bs::parser_tag<id_filename> > m_filename;

      /** \brief Rule matching any type. */
      bs::rule<ScannerT, bs::parser_tag<id_any_type> > m_any_type;

      /** \brief Rule matching an unsigned integer. */
      bs::rule<ScannerT, bs::parser_tag<id_u_integer_type> > m_u_integer_type;

      /** \brief Rule matching a real. */
      bs::rule<ScannerT, bs::parser_tag<id_real_type> > m_real_type;

      /** \brief Rule matching a boolean. */
      bs::rule<ScannerT, bs::parser_tag<id_bool_type> > m_bool_type;

      /** \brief Rule matching a string. */
      bs::rule<ScannerT, bs::parser_tag<id_string_type> > m_string_type;

      /** \brief Rule matching an image. */
      bs::rule<ScannerT, bs::parser_tag<id_image_type> > m_image_type;

      /** \brief Rule matching an animation. */
      bs::rule<ScannerT, bs::parser_tag<id_animation_type> > m_animation_type;

      /** \brief Rule matching an item in the body of an image declaration. */
      bs::rule<ScannerT, bs::parser_tag<id_image_body_item> >
      m_image_body_item;

      /** \brief Rule matching a list of frame (for animations). */
      bs::rule<ScannerT, bs::parser_tag<id_frame_list> > m_frame_list;

      /** \brief Rule matching the resource part. */
      bs::rule<ScannerT, bs::parser_tag<id_resources_part> > m_resources_part;

      /** \brief Rule matching a resource. */
      bs::rule<ScannerT, bs::parser_tag<id_resource> > m_resource;

      /** \brief Rule matching an image resource. */
      bs::rule<ScannerT, bs::parser_tag<id_resource_image> > m_resource_image;

      /** \brief Rule matching a sound resource. */
      bs::rule<ScannerT, bs::parser_tag<id_resource_sound> > m_resource_sound;

      /** \brief Rule matching the model part. */
      bs::rule<ScannerT, bs::parser_tag<id_model_part> > m_model_part;

      /** \brief Rule matching the body of a model. */
      bs::rule<ScannerT, bs::parser_tag<id_model_description> >
      m_model_description;

      /** \brief Rule matching an item in the body of a model. */
      bs::rule<ScannerT, bs::parser_tag<id_model_atom> > m_model_atom;

      /** \brief Rule matching the declaration of a constant. */
      bs::rule<ScannerT, bs::parser_tag<id_const_declaration> >
      m_const_declaration;

      /** \brief Rule matching the declaration of an action. */
      bs::rule<ScannerT, bs::parser_tag<id_action> > m_action;

      /** \} */

      /**
       * \brief Syntaxic errors.
       * \{
       */

      /** \brief Resources part is missing. */
      keyword_error m_error_resources_part;

      /** \brief Animation expected. */
      generic_error m_error_animation;

      /** \brief Identifier expected. */
      generic_error m_error_identifier;

      /** \brief Comma expected. */
      char_error m_error_comma;

      /** \brief Filename expected. */
      generic_error m_error_filename;

      /** \brief Unsigned integer expected. */
      type_error m_error_u_integer;

      /** \brief Image expected. */
      type_error m_error_image;

      /** \brief Equal symbol expected. */
      char_error m_error_equal;

      /** \brief Type expected. */
      generic_error m_error_type;

      /** \brief Semicolon expected. */
      char_error m_error_semicolon;

      /** \brief Not terminated string. */
      generic_error m_error_not_terminated_string;

      /** \brief Boolean expected. */
      type_error m_error_bool;

      /** \brief Sound or graphic resource expected. */
      generic_error m_error_resource;

      /** \brief Layer atom expected (constant or action). */
      generic_error m_error_model_atom;

      /** \brief Colon expected. */
      char_error m_error_colon;

      /** \brief Right bracket expected. */
      char_error m_error_right_bracket;

      /** \brief Left bracket expected. */
      char_error m_error_left_bracket;

      /** \brief Real number expected. */
      type_error m_error_real;

      /** \brief Right parenthesis expected. */
      char_error m_error_right_parenthesis;

      /** \brief Left parenthesis expected. */
      char_error m_error_left_parenthesis;

      /** \brief Right brace expected. */
      char_error m_error_right_brace;

      /** \brief Left brace expected. */
      char_error m_error_left_brace;

      /** \brief Item definition/creation or constant declaration expected. */
      generic_error m_error_layer_atom;

      /** \brief Missing 'fixed' keyword between brackets. */
      keyword_error m_error_fixed;

      /** \brief String expected. */
      type_error m_error_string;

    }; // class definition
  }; // class model_grammar
} // namespace mc

#include "mc/code/model_grammar.tpp"

#endif // __MC_MODEL_GRAMMAR_HPP__
