/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file node_compiler_const_declaration.cpp
 * \brief Implementation of the mc::node_compiler_const_declaration class.
 * \author Julien Jorge
 */
#include "mc/node_compiler/node_compiler_const_declaration.hpp"

#include <claw/assert.hpp>
#include "mc/node_compiler/node_compiler_identifier.hpp"
#include "mc/node_compiler/node_compiler_u_integer_type.hpp"
#include "mc/node_compiler/node_compiler_real_type.hpp"
#include "mc/node_compiler/node_compiler_bool_type.hpp"
#include "mc/node_compiler/node_compiler_string_type.hpp"
#include "mc/node_compiler/node_compiler_image_type.hpp"
#include "mc/node_compiler/node_compiler_animation_type.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a node of type const_declaration.
 * \param symbols The symbols already read in the current model.
 * \param the_model The model we are compiling.
 * \param node Node to compile.
 */
void mc::node_compiler_const_declaration::compile_node
( symbol_table& symbols,  const model& the_model, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_const_declaration );
  CLAW_PRECOND( node.children.size() == 2 );

  node_compiler_identifier comp_name;

  std::string name;

  comp_name.compile_node( name, node.children[0] );

  const tree_node& child = node.children[1];

  if ( symbols.constant_exists(name) )
    error(node.value.begin(), node.value.end(),
          "name '" + name + "' is already used.");
  else
    switch( child.children[0].value.id().to_long() )
      {
      case model_grammar::id_u_integer_type :
        add_u_integer( name, symbols, child.children[0] );
        break;
      case model_grammar::id_real_type :
        add_real( name, symbols, child.children[0] );
        break;
      case model_grammar::id_string_type :
        add_string( name, symbols, child.children[0] );
        break;
      case model_grammar::id_bool_type :
        add_bool( name, symbols, child.children[0] );
        break;
      case model_grammar::id_image_type :
        add_image( name, symbols, the_model, child.children[0] );
        break;
      case model_grammar::id_animation_type :
        add_animation( name, symbols, the_model, child.children[0] );
        break;
      case model_grammar::id_identifier :
        add_identifier( name, symbols, child.children[0] );
        break;
      default:
        CLAW_ASSERT(false, "Invalid node id.");
      }
} // node_compiler_const_declaration::compile_node()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type unsigned integer.
 * \param name The name of the constant.
 * \param symbols The symbols already read in the current model.
 * \param node Node to compile.
 */
void mc::node_compiler_const_declaration::add_u_integer
( const std::string& name, symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_u_integer_type );

  node_compiler_u_integer_type comp;
  u_integer_type val;
  comp.compile_node( val, node );

  symbols.add_u_integer( name, val );
} // node_compiler_const_declaration::add_u_integer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type real.
 * \param name The name of the constant.
 * \param symbols The symbols already read in the current model.
 * \param node Node to compile.
 */
void mc::node_compiler_const_declaration::add_real
( const std::string& name, symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_real_type );

  node_compiler_real_type comp;
  real_type val;
  comp.compile_node( val, node );

  symbols.add_real( name, val );
} // node_compiler_const_declaration::add_real()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type string.
 * \param name The name of the constant.
 * \param symbols The symbols already read in the current model.
 * \param node Node to compile.
 */
void mc::node_compiler_const_declaration::add_string
( const std::string& name, symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_string_type );

  node_compiler_string_type comp;
  string_type val;
  comp.compile_node( val, node );

  symbols.add_string( name, val );
} // node_compiler_const_declaration::add_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type string.
 * \param name The name of the constant.
 * \param symbols The symbols already read in the current model.
 * \param node Node to compile.
 */
void mc::node_compiler_const_declaration::add_bool
( const std::string& name, symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_bool_type );

  node_compiler_bool_type comp;
  bool_type val;
  comp.compile_node( val, node );

  symbols.add_bool( name, val );
} // node_compiler_const_declaration::add_bool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type image.
 * \param name The name of the constant.
 * \param symbols The symbols already read in the current model.
 * \param the_model The model we are compiling.
 * \param node Node to compile.
 */
void mc::node_compiler_const_declaration::add_image
( const std::string& name, symbol_table& symbols,  const model& the_model,
  const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_image_type );

  node_compiler_image_type comp;
  image_type val;
  comp.compile_node( val, symbols, the_model, node );

  symbols.add_image( name, val );
} // node_compiler_const_declaration::add_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type animation.
 * \param name The name of the constant.
 * \param symbols The symbols already read in the current model.
 * \param the_model The model we are compiling.
 * \param node Node to compile.
 */
void mc::node_compiler_const_declaration::add_animation
( const std::string& name, symbol_table& symbols,  const model& the_model,
  const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_animation_type );

  node_compiler_animation_type comp;
  animation_type val;
  comp.compile_node( val, symbols, the_model, node );

  symbols.add_animation( name, val );
} // node_compiler_const_declaration::add_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a constant from an already defined constant.
 * \param name The name of the constant.
 * \param symbols The symbols already read in the current model.
 * \param node Node to compile.
 */
void mc::node_compiler_const_declaration::add_identifier
( const std::string& name, symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_identifier );

  node_compiler_identifier comp;
  std::string original;
  comp.compile_node( original, node );

  if ( symbols.u_integer_exists(original) )
    symbols.add_u_integer( name, symbols.get_u_integer(original) );
  else if ( symbols.real_exists(original) )
    symbols.add_real( name, symbols.get_real(original) );
  else if ( symbols.bool_exists(original) )
    symbols.add_bool( name, symbols.get_bool(original) );
  else if ( symbols.string_exists(original) )
    symbols.add_string( name, symbols.get_string(original) );
  else if ( symbols.image_exists(original) )
    symbols.add_image( name, symbols.get_image(original) );
  else if ( symbols.animation_exists(original) )
    symbols.add_animation( name, symbols.get_animation(original) );
  else
    error(node.value.begin(), node.value.end(), "Unknow constant.");
} // node_compiler_const_declaration::add_identifier()
