/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file headstone.cpp
 * \brief Implementation of the ptb::headstone class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/headstone.hpp"

#include "ptb/item/soul.hpp"
#include "engine/game.hpp"
#include "engine/world.hpp"
#include "universe/forced_tracking.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( headstone, ptb )

const bear::universe::time_type ptb::headstone::s_time_to_appear = 2;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::headstone::headstone()
  : m_soul_has_appeared(false), m_soul_energy(0), 
    m_soul_animation(NULL), m_time_start(0)
{
  set_size( 22, 45 );
  
  set_can_move_items(false);
  set_phantom(true);
} // headstone::headstone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::headstone::~headstone()
{

} // headstone::headstone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void ptb::headstone::start()
{
  super::start();

  set_model("model/headstone.cm");
  execute_action("appear");
} // headstone::start()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::headstone::progress( bear::universe::time_type elapsed_time )
{
  super::progress( elapsed_time );
  
  if ( ! m_soul_has_appeared )
    {
      m_time_start += elapsed_time;
      
      if ( m_time_start >= s_time_to_appear )
	{
	  m_soul_has_appeared = true;
	  create_soul();
	}
    }
} // headstone::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool ptb::headstone::is_valid() const
{
  return (m_soul_animation != NULL) && super::is_valid();
} // headstone::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c unsigned \c integer.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::headstone::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool ok = true;

  if (name == "energy")
    m_soul_energy = value;
  else
    ok = super::set_u_integer_field(name,value);

  return ok;
} // headstone::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c float.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::headstone::set_real_field( const std::string& name, double value )
{
  bool ok = true;
  
  if ( name == "pos_bottom" )
     {
      bear::universe::position_type pos(get_position());

      pos.y = value - get_height();
      set_position(pos);
    }
  else
    ok = super::set_real_field(name, value);

  return ok;
} // headstone::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c bear::visual::animation.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::headstone::set_animation_field
( const std::string& name, bear::visual::animation* value )
{
  bool ok = true;
  
  if ( name == "soul_animation" )
    m_soul_animation = value;
  else
    ok = super::set_animation_field(name, value);

  return ok;
} // headstone::set_animation_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the dependent item and create the forced movement.
 */
void ptb::headstone::set_dependent_item(physical_item* item)
{
  const bear::universe::position_type distance(get_center_of_mass() - 
					 item->get_center_of_mass());

  bear::universe::forced_tracking* movement;
  movement = new bear::universe::forced_tracking
    (*((bear::universe::physical_item*)this),*item,distance);

  set_forced_movement( *movement );
  get_forced_movement().start();
} // headstone::set_dependent_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialization of this state.
 */
void ptb::headstone::create_soul() const
{
  bear::engine::world* w = get_owner();

  soul* new_soul = new soul;

  new_soul->set_animation_field("animation", m_soul_animation);
  new_soul->set_real_field("pos_x", get_center_of_mass().x - 
                           new_soul->get_width()/2);
  new_soul->set_real_field("pos_y", get_position().y);
  new_soul->set_u_integer_field("energy", m_soul_energy);
  new_soul->set_integer_field("pos_z", get_z_position()+1 );

  w->register_item( new_soul );

  CLAW_ASSERT( new_soul->is_valid(),
               "The soul of wasp isn't correctly initialized" );
  new_soul->start();
} // headstone::create_soul()
