/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file player_speaker_zone.cpp
 * \brief Implementation of the ptb::player_speaker_zone class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/player_speaker_zone.hpp"

#include "engine/controller_layout.hpp"
#include "engine/game.hpp"
#include "engine/player.hpp"
#include "engine/string_base.hpp"
#include "ptb/speaker_item.hpp"
#include "ptb/item/plee/plee.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( player_speaker_zone, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::player_speaker_zone::player_speaker_zone()
  : m_first_player_spoke(false), m_second_player_spoke(false),
    m_first_player_in_zone(false), m_second_player_in_zone(false)
{
  set_phantom(true);
  set_can_move_items(false);
} // player_speaker_zone::player_speaker_zone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void ptb::player_speaker_zone::start()
{
  m_first_player =
    bear::engine::pointer_to_player( bear::engine::player::player_name(1) );
  m_second_player =
    bear::engine::pointer_to_player( bear::engine::player::player_name(2) );
} // player_speaker_zone::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void
ptb::player_speaker_zone::progress( bear::universe::time_type elapsed_time )
{
  test_player_enter(m_first_player,m_first_player_in_zone);
  test_player_enter(m_second_player,m_second_player_in_zone);
} // player_speaker_zone::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if a player enter in the zone.
 */
void ptb::player_speaker_zone::test_player_enter
( bear::engine::pointer_to_player player, bool& player_in_zone )
{
  if ( player )
    {
      claw::math::rectangle<bear::universe::coordinate_type> bounding_box_1
        ( player->get_bounding_box() );

      if ( bounding_box_1.intersects(get_bounding_box()) )
        {
          if ( ! player_in_zone )
            {
              ptb::speaker_item* speaker = 
                dynamic_cast<ptb::speaker_item*>(&(*player));
              
              speaker->reset_speeches();
              
              const bear::engine::controller_layout& layout =
                bear::engine::game::get_instance().get_rules().get_layout
                ( player->get_index() );
              
              std::vector< std::string>::iterator it;
              for ( it = m_speeches.begin(); it != m_speeches.end(); ++it)
                {
                  std::string speech(*it);
                  bear::engine::string_base::get_instance().get_string(speech);
                  layout.escape_action_sequence(speech);
                  speaker->add_speech(speech);
                }
              
              player_in_zone = speaker->speak();
            }
        }
      else 
        player_in_zone = false;
    }
} // player_speaker_zone::test_player_enter()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type list of <std::string>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::player_speaker_zone::set_string_list_field
( const std::string& name, const std::vector<std::string>& value )
{
  bool result = false;

  if ( name == "speeches" )
    {
      m_speeches = std::vector<std::string>(value);
      result = true;
    }
  else
    result = super::set_string_list_field( name, value );

  return result;
} // player_speaker_zone::set_string_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool ptb::player_speaker_zone::is_valid() const
{
  return !m_speeches.empty() && super::is_valid();
} // player_speaker_zone::is_valid()
