/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file wasp.cpp
 * \brief Implementation of the ptb::wasp class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/wasp/wasp.hpp"
#include "ptb/item/wasp/sting.hpp"
#include "engine/export.hpp"
#include <claw/assert.hpp>
#include "visual/sprite.hpp"
#include "ptb/item/wasp/state_wasp_fly.hpp"
#include "ptb/item/wasp/state_wasp_attack.hpp"
#include "ptb/item/wasp/state_wasp_dead.hpp"

BASE_ITEM_EXPORT( wasp, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::wasp::wasp()
  : m_have_attacked(false),
    m_first_target(0,0), m_second_target(0,0),
    m_translation(0,0), m_current_state(fly_state), 
    m_positive_orientation(true),
    m_number_of_steps(0)
{
  set_size( 70, 60 );
  m_monster_type = monster::ennemy_monster;
  m_energy = 0;
  m_offensive_force = s_wasp_offensive_force;
  m_offensive_coefficients[normal_attack] = 1;

  m_states.resize(states_cardinality);
  m_states[fly_state] = new state_wasp_fly(this);
  m_states[attack_state] = new state_wasp_attack(this);
  m_states[dead_state] = new state_wasp_dead(this);
} // wasp::wasp()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::wasp::~wasp()
{
  unsigned int i;

  for (i=0; i!=states_cardinality; ++i)
    delete m_states[i];
} // wasp::~wasp()

/*---------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void ptb::wasp::start()
{
  super::start();

  set_model("model/wasp.cm");

  m_first_target = get_center_of_mass();
  m_second_target = m_translation + m_first_target;

  start_action_parent("fly");
  set_state(fly_state);
} // wasp::progress()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::wasp::progress( bear::universe::time_type elapsed_time )
{
  super::progress( elapsed_time );

  m_states[m_current_state]->progress( elapsed_time );
} // wasp::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the state of wasp.
 */
void ptb::wasp::set_state(wasp_state_name state)
{
  m_current_state = state;
  m_states[m_current_state]->start();
  reset_animation();
} // wasp::set_state()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the execution of action to the parent in order to
 *        start the animation.
 * \param action The action to realize. 
 */
void ptb::wasp::start_action_parent(const std::string action)
{
  super::execute_action(action);

  if ( m_positive_orientation )
    mirror( false );
  else
    mirror( true );
} // wasp::start_action_parent()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <real>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::wasp::set_real_field
( const std::string& name, double value )
{
  bool ok = true;

  if (name == "translation_x")
    m_translation.x = value;
  else if (name == "translation_y")
    m_translation.y = value;
  else
    ok = super::set_real_field(name,value);

  return ok;
} // wasp::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type unsigned int.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::wasp::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool ok = true;

  if (name == "steps")
    m_number_of_steps = value;
  else
    ok = super::set_u_integer_field(name,value);

  return ok;
} // wasp::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool ptb::wasp::is_valid() const
{
  bool ok = true;
  
  if ( m_number_of_steps == 0 )
    ok = false;
  else
    ok = super::is_valid();
  
  return ok;
} //wasp::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the first target.
 */
bear::universe::position_type ptb::wasp::get_first_target() const
{
  return m_first_target;
} // wasp::get_first_target()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the second target.
 */
bear::universe::position_type ptb::wasp::get_second_target() const
{
  return m_second_target;
} // wasp::get_second_target()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the number of steps.
 */
unsigned int ptb::wasp::get_number_of_steps() const
{
  return m_number_of_steps;
} // wasp::get_number_of_steps()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the orientation.
 */
bool ptb::wasp::get_positive_orientation()
{
  return m_positive_orientation;
} // wasp::get_positive_orientation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the orientation.
 */
void ptb::wasp::set_positive_orientation(bool orientation)
{
  m_positive_orientation = orientation;

  if ( m_positive_orientation )
    mirror( false );
  else
    mirror( true );
} // wasp::set_positive_orientation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the item that he have no energy now.
 */
void ptb::wasp::inform_no_energy()
{
  m_is_injured = false;
  set_state(dead_state);
} // wasp::inform_no_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that the item has a contact on its bottom.
 */
void ptb::wasp::set_bottom_contact(bear::universe::physical_item_state& item)
{
  super::set_bottom_contact(item);

  m_states[m_current_state]->set_bottom_contact(item);
} // wasp::set_bottom_contact()
