/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file player_arrows_layer.cpp
 * \brief Implementation of the ptb::player_arrows_layer class.
 * \author Sebastien Angibaud
 */
#include "ptb/layer/player_arrows_layer.hpp"
#include "ptb/export.hpp"
#include "engine/font_factory.hpp"
#include "engine/game.hpp"
#include "engine/camera.hpp"
#include "universe/zone.hpp"
#include "text/font.hpp"

#include <algorithm>
#include <claw/algorithm.hpp>
#include <claw/functional.hpp>
#include <claw/assert.hpp>

GUI_LAYER_EXPORT( player_arrows_layer, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
  ptb::player_arrows_layer::player_arrows_layer()
    : m_first_player( bear::engine::player::player_name(1) ),
      m_second_player( bear::engine::player::player_name(2) ),
      m_text(NULL,
             bear::engine::font_factory::create("font/fixed_yellow-19x39.tga"))
{
  m_text.set_auto_size(true);
  m_text.set_text("0");
  m_text.set_position( 0,0 );

  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& image_resource = glob.get_image("gfx/status.tga");

  claw::math::rectangle<unsigned int> box1( 101, 0, 47, 27 );
  m_player_1_sprite = new bear::visual::sprite( image_resource, box1 );
  claw::math::rectangle<unsigned int> box2( 101, 27, 47, 27 );
  m_player_2_sprite = new bear::visual::sprite( image_resource, box2 );
} // player_arrows_layer::player_arrows_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::player_arrows_layer::~player_arrows_layer()
{
  if ( m_player_1_sprite != NULL ) 
    delete m_player_1_sprite;

  if ( m_player_2_sprite != NULL ) 
    delete m_player_2_sprite;
} // player_arrows_layer::~player_arrows_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the angle of the arrow.
 * \param position_player The position of the player.
 * \param position_arrow The position of the arrow.
 */
double ptb::player_arrows_layer::adjust_angle
( const bear::universe::position_type& position_player,
  const bear::universe::position_type& position_arrow ) const
{
  bear::universe::speed_type v;
  v.x = position_player.x - position_arrow.x;
  v.y = position_player.y - position_arrow.y;
  bear::universe::speed_type vect(v);
  double angle = 0;
  v.normalize();
  
  if ( v.y >= 0 )
    angle = -acos(v.x);
  else
    angle = acos(v.x);

  return angle;
} // player_arrows_layer::adjust_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the position of an arrow.
 * \param zone_position The zone position.
 * \param position_player The position of the player.
 * \param visible_area The visible part of the layer.
 */
claw::math::coordinate_2d<int>
ptb::player_arrows_layer::adjust_position
( const bear::universe::position_type& position_player,
  const bear::universe::rectangle_type& visible_area ) const
{
  bear::universe::position_type result;
  
  bear::universe::position_type center(visible_area.position);
  center.x += visible_area.width/2;
  center.y += visible_area.height/2;
  bear::universe::position_type pos_player(position_player);
  
  claw::math::vector_2d<double> vect(center,pos_player);
  vect.normalize();
  vect.x *= ( visible_area.width / 2 ) - 50;
  vect.y *= ( visible_area.height / 2 ) - 50;

  result =
    center + vect - visible_area.position - m_player_1_sprite->get_size() / 2;

  return result.cast_value_type_to<int>();
} // player_arrows_layer::adjust_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the distance between the player and the arrow.
 * \param index type of arrow.
 * \param arrow_position The position of the arrow.
 * \param position_player The position of the player.
 * \param visible_area The visible part of the layer.
 * \param box_text The box of the text.
 */
void ptb::player_arrows_layer::adjust_distance
( double angle,
  const bear::universe::position_type& arrow_position,
  const bear::universe::position_type& position_player,
  const bear::universe::rectangle_type& visible_area )
{
  bear::universe::position_type relative_position;
  relative_position = arrow_position + visible_area.position;

  bear::universe::speed_type vect(position_player, relative_position);
  unsigned int dist = (unsigned int)vect.length() / 100;

  std::ostringstream oss;
  oss << dist;
  m_text.set_text(oss.str());

  bear::universe::position_type position_text(arrow_position);

  bear::universe::position_type gap;
  gap.x = cos(angle) * 50;
  gap.y = sin(angle) * 50;

  position_text.x = position_text.x + m_player_1_sprite->width() / 2 - 
    gap.x - m_text.width()/2;
  position_text.y = position_text.y + m_player_1_sprite->height() / 2 + 
    gap.y - m_text.height()/2 + 4;

  m_text.set_position( position_text.cast_value_type_to<unsigned int>() );
} // player_arrows_layer::adjust_distance()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the arrow of a player.
 * \param screen The screen on which we'll draw.
 * \param visible_area The visible part of the layer.
 * \param p The player for which we want the arrow.
 * \param a The set of arrows for this player.
 */
void ptb::player_arrows_layer::render_player
( bear::visual::screen& screen,
  const bear::universe::rectangle_type& visible_area,
  const bear::engine::pointer_to_player& p, bear::visual::sprite* arrow )
{
  const claw::math::rectangle<bear::universe::coordinate_type> box =
    p->get_bounding_box();

  bear::universe::zone::position zone_position =
    bear::universe::zone::find( box, visible_area );

  if ( zone_position != bear::universe::zone::middle_zone )
    {
      double angle = 0;
      
      bear::universe::position_type pos =
        adjust_position( p->get_center_of_mass(), visible_area );
      const bear::universe::position_type pos_center =
        pos + visible_area.position + arrow->get_size() / 2;

      angle = adjust_angle(p->get_center_of_mass(), pos_center);
      
      adjust_distance( angle, pos, p->get_center_of_mass(), visible_area );
      
      screen.render( pos.cast_value_type_to<int>(), *arrow, angle );

      m_text.render( screen );
    }
} // player_arrows_layer::render_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the layer.
 * \param screen The screen on which we'll draw.
 */
void ptb::player_arrows_layer::render( bear::visual::screen& screen )
{
  bear::engine::camera::msg_get_focus msg;

  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  glob.send_message( bear::engine::camera::default_name(), msg );

  if (m_first_player)
    render_player( screen, msg.focus, m_first_player, m_player_1_sprite );

  if (m_second_player)
    render_player( screen, msg.focus, m_second_player, m_player_2_sprite );
} // player_arrows_layer::render()
