/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file recent_path_layer_layer.cpp
 * \brief Implementation of the ptb::recent_path_layer_layer class.
 * \author Julien Jorge
 */
#include <algorithm>

#include "ptb/layer/recent_path_layer.hpp"

#include "ptb/export.hpp"

GUI_LAYER_EXPORT( recent_path_layer, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::recent_path_layer::item_positions::item_positions
( const bear::engine::base_item* item )
  : super(9)
{
  this->operator[](0) = item->get_position();
  this->operator[](1) = item->get_top_middle();
  this->operator[](2) = item->get_top_right();
  this->operator[](3) = item->get_left_middle();
  this->operator[](4) = item->get_center_of_mass();
  this->operator[](5) = item->get_right_middle();
  this->operator[](6) = item->get_bottom_left();
  this->operator[](7) = item->get_bottom_middle();
  this->operator[](8) = item->get_bottom_right();
} // recent_path_layer::item_positions::item_positions()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the position of an item has changed.
 * \param item The item from which we tak the position.
 */
bool ptb::recent_path_layer::item_positions::has_moved
( const bear::engine::base_item* item ) const
{
  return this->operator[](0) != item->get_position();
} // recent_path_layer::item_positions::has_moved()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::recent_path_layer::recent_path_layer()
  : base_debugging_layer(bear::input::keyboard::kc_F7)
{

} // recent_path_layer::recent_path_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the boxes of the items.
 * \param items The items to draw.
 * \param delta The delta to apply to the position of the items.
 * \param screen The screen on which we draw.
 */
void ptb::recent_path_layer::render
( const std::list<bear::engine::base_item*>& items,
  const bear::universe::position_type& delta, bear::visual::screen& screen )
{
  item_map::iterator itm;

  // remove lost items.
  for ( itm=m_items.begin(); itm!=m_items.end(); )
    if ( std::find(items.begin(), items.end(), itm->first) == items.end() )
      {
        item_map::iterator tmp(itm);
        ++itm;
        m_items.erase(tmp);
      }
    else
      ++itm;

  // add new items and update the positions of the known items.
  std::list<bear::engine::base_item*>::const_iterator it;
  for (it=items.begin(); it!=items.end(); ++it)
    {
      itm = m_items.find(*it);

      if ( itm == m_items.end() )
        m_items[*it].push_front(*it);
      else if ( itm->second.back().has_moved(*it) )
        {
          if ( itm->second.size() == 10 )
            itm->second.pop_front();

          itm->second.push_back( item_positions(*it) );
        }
    }

  // one color for each line.
  claw::graphic::pixel32 colors[9] =
    {
      claw::graphic::pixel32( 255, 255, 255, 255 ),
      claw::graphic::pixel32( 255,   0,   0, 255 ),
      claw::graphic::pixel32(   0, 255,   0, 255 ),
      claw::graphic::pixel32(   0,   0, 255, 255 ),
      claw::graphic::pixel32( 255, 255,   0, 255 ),
      claw::graphic::pixel32( 255,   0, 255, 255 ),
      claw::graphic::pixel32(   0, 255, 255, 255 ),
      claw::graphic::pixel32( 190, 190, 190, 255 ),
      claw::graphic::pixel32( 190, 190,   0, 255 )
    };

  // Draw the lines.
  for (itm=m_items.begin(); itm!=m_items.end(); ++itm)
    for (unsigned int i=0; i!=9; ++i)
      {
        std::list< claw::math::coordinate_2d<int> > points;
        std::list<item_positions>::const_iterator itl;

        for ( itl=itm->second.begin(); itl!=itm->second.end(); ++itl )
          points.push_back( ((*itl)[i] - delta).cast_value_type_to<int>() );

        screen.draw_line( colors[i], points.begin(), points.end() );
      }
} // recent_path_layer::render()
