/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file game.cpp
 * \brief Implementation of the bear::engine::game class.
 * \author Julien Jorge
 */
#include "engine/game.hpp"

#include "input/system.hpp"
#include "engine/level_loader.hpp"
#include "engine/resource_pool.hpp"
#include "engine/version.hpp"

#include "debug/timing_log.hpp"

#include <boost/filesystem/convenience.hpp>
#include <claw/exception.hpp>
#include <claw/logger.hpp>
#include <claw/string_algorithm.hpp>
#include <claw/system_info.hpp>
#include <sstream>

namespace bear
{
  namespace engine
  {
    /**
     * \brief Set a game variable from a command line argument string formated
     *        "name<sep>value".
     * \param args The values assigned to the command line arguments.
     * \param sep The separator between the name and the value. Must not be
     *        present in the variable's name.
     */
    template<typename T>
    static bool game_set_game_variable_from_arg
    ( const std::list<std::string>& args, const char sep )
    {
      bool result(true);
      std::list<std::string>::const_iterator it;

      for (it=args.begin(); it!=args.end(); ++it)
        {
          const std::size_t pos( it->find_first_of(sep) );

          if ( pos == std::string::npos )
            result = false;
          else
            {
              const std::string name( it->substr(0, pos) );
              const std::string value( it->substr(pos+1) );

              if ( !claw::text::is_of_type<T>(value) )
                result = false;
              else
                {
                  std::istringstream iss(value);
                  T v;
                  iss >> v;

                  game::get_instance().get_game_variables().template set<T>(name, v);
                  result = true;
                }
            }
        }

      return result;
    } // game_set_game_variable_from_arg()

  } // namespace engine
} // namespace bear

/*----------------------------------------------------------------------------*/
bear::engine::game* bear::engine::game::s_instance(NULL);
std::string bear::engine::game::s_init_game_function_prefix("init_");
std::string bear::engine::game::s_end_game_function_prefix("end_");

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the instance of the game.
 * \pre s_instance != NULL
 */
bear::engine::game& bear::engine::game::get_instance()
{
  CLAW_PRECOND( s_instance != NULL );

  return *s_instance;
} // game::get_instance()

/*----------------------------------------------------------------------------*/
/**
 * \brief Print the options of the program.
 */
void bear::engine::game::print_help()
{
  get_arguments_table().help();
} // game::print_help()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
bear::engine::game::game( int& argc, char** &argv )
  : m_status(status_init), m_screen(NULL), m_fullscreen(false),
    m_current_level(NULL), m_level_in_abeyance(NULL), m_time_step(15)
{
  CLAW_PRECOND( s_instance == NULL );
  s_instance = this;

  if ( !check_arguments(argc, argv) )
    m_status = status_quit;
  else
    {
      init_environment();

      try
        {
          m_screen = new visual::screen
            ( m_game_description.screen_size(),
              m_game_description.game_name(), m_fullscreen );
        }
      catch(...)
        {
          clear();
          close_environment();
          s_instance = NULL;
          throw;
        }
    }
} // game::game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::engine::game::~game()
{
  clear();
  close_environment();

  base_item::print_allocated();
  s_instance = NULL;
} // game::~game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Run the game.
 */
void bear::engine::game::run()
{
  if ( m_status != status_quit )
    {
      init_game();

      load_level( m_game_description.start_level() );

      run_level();

      end_game();

      clear();
      close_environment();
    }
} // game::run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn on/off the full screen mode.
 * \param b Tell if we must activate the fullscreen mode.
 */
void bear::engine::game::set_fullscreen( bool full )
{
  if ( m_fullscreen != full )
    {
      m_fullscreen = full;
      m_screen->fullscreen(m_fullscreen);
    }
} // game::set_fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the full screen mode.
 */
bool bear::engine::game::get_fullscreen() const
{
  return m_fullscreen;
} // game::get_fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the status of the fullscreen mode.
 */
void bear::engine::game::toggle_fullscreen()
{
  set_fullscreen( !m_fullscreen );
} // game::toggle_fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Mute/unmute the sounds.
 * \param m The mute status.
 */
void bear::engine::game::set_sound_muted( bool m )
{
  if ( m_current_level == NULL )
    level_globals::global_set_sound_muted(m);
  else
    m_current_level->get_globals().mute_sound(m);
} // game::set_sound_muted()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the sound is muted.
 */
bool bear::engine::game::get_sound_muted() const
{
  return level_globals::global_get_sound_muted();
} // game::get_sound_muted()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the "mute" status of the sounds.
 */
void bear::engine::game::toggle_sound_muted()
{
  set_sound_muted( !get_sound_muted() );
} // game::toggle_sound_muted()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the volume of the sounds.
 * \param v The volume.
 */
void bear::engine::game::set_sound_volume( double v )
{
  if ( m_current_level == NULL )
    level_globals::global_set_sound_volume(v);
  else
    m_current_level->get_globals().set_sound_volume(v);
} // game::set_sound_volume()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the volume of the sounds.
 */
double bear::engine::game::get_sound_volume() const
{
  return level_globals::global_get_sound_volume();
} // game::get_sound_colume()

/*----------------------------------------------------------------------------*/
/**
 * \brief Mute/unmute the music.
 * \param m The mute status.
 */
void bear::engine::game::set_music_muted( bool m )
{
  if ( m_current_level == NULL )
    level_globals::global_set_music_muted(m);
  else
    m_current_level->get_globals().mute_music(m);
} // game::set_music_muted()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the music is muted.
 */
bool bear::engine::game::get_music_muted() const
{
  return level_globals::global_get_music_muted();
} // game::get_music_muted()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the "mute" status of the music.
 */
void bear::engine::game::toggle_music_muted()
{
  set_music_muted( !get_music_muted() );
} // game::toggle_music_muted()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the volume of the musics.
 * \param v The volume.
 */
void bear::engine::game::set_music_volume( double v )
{
  if ( m_current_level == NULL )
    level_globals::global_set_music_volume(v);
  else
    m_current_level->get_globals().set_music_volume(v);
} // game::set_music_volume()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the volume of the musics.
 */
double bear::engine::game::get_music_volume() const
{
  return level_globals::global_get_music_volume();
} // game::get_music_volume()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the status of the fullscreen mode.
 */
void bear::engine::game::toggle_time_step()
{
  if ( m_time_step == 0 )
    m_time_step = 15;
  else
    m_time_step = 0;
} // game::toggle_time_step()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the current time step.
 */
bear::systime::milliseconds_type bear::engine::game::get_time_step() const
{
  return m_time_step;
} // game::get_time_step()

/*----------------------------------------------------------------------------*/
/**
 * \brief Take a shot of the screen.
 * \param img The image in which we save the screen.
 */
void bear::engine::game::screenshot( claw::graphic::image& img ) const
{
  m_screen->shot( img );
} // game::screenshot()

/*----------------------------------------------------------------------------*/
/**
 * \brief End the game.
 */
void bear::engine::game::end()
{
  m_status = status_quit;
} // game::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the name of a level to load as soon as possible.
 * \param level_name The name of the level to load.
 */
void bear::engine::game::set_waiting_level( const std::string& level_name )
{
  m_post_actions.push( new game_action_load_level(level_name) );
} // game::set_waiting_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the level to run as soon as possible.
 * \param the_level The level to run as soon as possible.
 */
void bear::engine::game::set_waiting_level( level* the_level )
{
  CLAW_PRECOND( the_level != NULL );

  m_post_actions.push( new game_action_set_current_level(the_level) );
} // game::set_waiting_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the name of a level to load as soon as possible but keep the
 *        current level in memory for future restoration.
 * \param level_name The name of the level to load.
 */
void bear::engine::game::push_level( const std::string& level_name )
{
  m_post_actions.push( new game_action_push_level(level_name) );
} // game::push_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Restore the level at the top of the stack.
 */
void bear::engine::game::pop_level()
{
  m_post_actions.push( new game_action_pop_level );
} // game::pop_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the screen.
 */
const claw::math::coordinate_2d<unsigned int>&
bear::engine::game::get_screen_size() const
{
  return m_game_description.screen_size();
} // game::get_screen_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the margin of the active area around the camera.
 */
double bear::engine::game::get_active_area_margin() const
{
  return m_game_description.active_area_margin();
} // game::get_active_area_margin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the path of a file in the user's game directory.
 * \param name The name of the file.
 */
std::string
bear::engine::game::get_custom_game_file( const std::string& name ) const
{
  std::string result = get_game_directory();

  if ( !result.empty() )
    {
      boost::filesystem::path path( result, boost::filesystem::native );
      path /= name;
      result = path.string();
    }
  else
    result = name;

  return result;
} // game::get_full_user_path()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the global variables of the game.
 */
bear::engine::game::var_map& bear::engine::game::get_game_variables()
{
  return m_game_variables;
} // game::get_game_variables()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the path of the file of a given level.
 * \param level_name The name of the level for which we want the file.
 */
const std::string&
bear::engine::game::get_level_file( const std::string& level_name ) const
{
  CLAW_PRECOND( level_exists(level_name) );

  return m_game_description.level_files().find(level_name)->second;
} // game::get_level_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is a level with a given name.
 * \param level_name The name of the level to check.
 */
bool bear::engine::game::level_exists( const std::string& level_name ) const
{
  bool result = false;

  if ( m_game_description.level_files().find(level_name)
       != m_game_description.level_files().end() )
    {
      const std::string level_path =
        m_game_description.level_files().find(level_name)->second;

      if ( resource_pool::get_instance().exists( level_path ) )
        {
          std::stringstream f;
          resource_pool::get_instance().get_file( level_path, f );

          result = !!f;
        }
    }

  return result;
} // game::level_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the symbols of the libraries linked to the game.
 */
bear::engine::libraries_pool& bear::engine::game::get_symbols()
{
  return m_symbols;
} // game::get_symbols()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the game.
 */
const std::string& bear::engine::game::get_name() const
{
  return m_game_description.game_name();
} // game::get_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the game specific function for initialisation.
 */
void bear::engine::game::init_game() const
{
  std::string game_proc
    ( s_init_game_function_prefix + get_game_name_as_filename() );

  claw::logger << claw::log_verbose << "Initialising game: '" << game_proc
               << "()'" << std::endl;

  if ( m_symbols.have_symbol( game_proc ) )
    {
      init_game_function_type func =
        m_symbols.get_symbol<init_game_function_type>( game_proc );
      func();
    }
} // game::init_game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the game specific function for ending the game.
 */
void bear::engine::game::end_game() const
{
  std::string game_proc
    ( s_end_game_function_prefix + get_game_name_as_filename() );

  claw::logger << claw::log_verbose << "Ending game: '" << game_proc
               << "()'" << std::endl;

  if ( m_symbols.have_symbol( game_proc ) )
    {
      end_game_function_type func =
        m_symbols.get_symbol<end_game_function_type>( game_proc );
      func();
    }
} // game::end_game()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the game in a string usable for a file name.
 */
std::string bear::engine::game::get_game_name_as_filename() const
{
  std::string result( m_game_description.game_name() );

  std::transform( result.begin(), result.end(), result.begin(), tolower );

  for (unsigned int i=0; i!=result.size(); ++i)
    if (result[i] == ' ')
      result[i] = '_';
    else if (result[i] == '\t')
      result[i] = '_';

  claw::text::squeeze( result, "_" );

  return result;
} // game::get_game_name_as_filename()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get, and create, the personal game directory of the user.
 */
std::string bear::engine::game::get_game_directory() const
{
  boost::filesystem::path dir
    (claw::system_info::get_user_directory(), boost::filesystem::native);

  std::string result;
  std::string subdir = '.' + get_game_name_as_filename();

  dir /= boost::filesystem::path(subdir, boost::filesystem::native);

  if ( create_game_directory(dir.string()) )
    result = dir.string();
  else
    claw::logger << claw::log_error << "Can't create game directory '"
                 << dir.string() << "'." << std::endl;

  return result;
} // game::get_game_directory()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the personal game directory of the user.
 * \param dir The path of the directory.
 */
bool bear::engine::game::create_game_directory( const std::string& dir ) const
{
  bool result = false;
  boost::filesystem::path path( dir, boost::filesystem::native );

  if ( boost::filesystem::exists( path ) )
    result = boost::filesystem::is_directory( path );
  else
    result = boost::filesystem::create_directory( path );

  return result;
} // game::create_game_directory()

/*----------------------------------------------------------------------------*/
/**
 * \brief Run the current_level.
 */
void bear::engine::game::run_level()
{
  m_status = status_run;

  while (m_status != status_quit)
    {
      m_last_progress = systime::get_date_ms();

      do
        {
          one_step_beyond();
        }
      while ( !do_post_actions() && (m_status != status_quit) );
    }
} // game::run_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one progress()/render() iteration.
 */
void bear::engine::game::one_step_beyond()
{
  systime::milliseconds_type current_time;

  current_time = systime::get_date_ms();
  universe::time_type dt( current_time - m_last_progress );

  if ( dt >= m_time_step )
    {
      m_last_progress = current_time;
      
      do
        {
          progress( (universe::time_type)m_time_step / 1000 ); // seconds
          dt -= m_time_step;
        }
      while ( (dt >= m_time_step) && (m_time_step > 0) );

      m_last_progress -= dt;

      render();
    }

  if ( m_time_step > 0 )
    systime::sleep( m_last_progress + m_time_step - current_time );
} // game::one_step_beyond()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the game.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::engine::game::progress( universe::time_type elapsed_time )
{
  // effective procedure
  input::system::get_instance().refresh();

  m_current_level->progress( elapsed_time );
} // game::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the current level.
 */
void bear::engine::game::render()
{
  if ( m_screen->need_restoration() )
    {
      m_current_level->get_globals().restore_images();

      if ( m_level_in_abeyance != NULL )
        m_level_in_abeyance->get_globals().restore_images();

      m_screen->set_restored();
    }

  // effective procedure
  m_screen->begin_render();
  m_current_level->render( *m_screen );
  m_screen->end_render();
} // game::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the environment (screen, inputs, sounds).
 */
void bear::engine::game::init_environment() const
{
  claw::logger << claw::log_verbose << "Initializing screen environment."
               << std::endl;

  visual::screen::initialize( visual::screen::screen_gl );

  claw::logger << claw::log_verbose << "Initializing input environment."
               << std::endl;

  input::system::initialize();

  claw::logger << claw::log_verbose << input::joystick::number_of_joysticks()
               << " joysticks found." << std::endl;

  claw::logger << claw::log_verbose << "Initializing sound environment."
               << std::endl;

  audio::sound_manager::initialize();
} // game::init_environment()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close the environment (screen, inputs, sounds).
 */
void bear::engine::game::close_environment() const
{
  claw::logger << claw::log_verbose << "Closing screen environment."
               << std::endl;

  visual::screen::release();

  claw::logger << claw::log_verbose << "Closing input environment."
               << std::endl;

  input::system::release();

  claw::logger << claw::log_verbose << "Closing sound environment."
               << std::endl;

  audio::sound_manager::release();
} // game::close_environment()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the libraries containing the items.
 * \param p A list of paths to libraries.
 */
void bear::engine::game::load_libraries( const std::list<std::string>& p )
{
  std::list<std::string>::const_iterator it;

  for ( it=p.begin(); it!=p.end(); ++it )
    {
      claw::logger << claw::log_verbose << "Add library '" << *it << "'."
                   << std::endl;
      m_symbols.add_library(*it);
    }
} // game::load_libraries()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add paths in the resource_pool.
 * \param p A list of paths.
 */
void
bear::engine::game::init_resource_pool( const std::list<std::string>& p ) const
{
  std::list<std::string>::const_iterator it;

  for ( it=p.begin(); it!=p.end(); ++it )
    {
      claw::logger << claw::log_verbose
                   << "Adding resource path '" << *it << "'."
                   << std::endl;
      resource_pool::get_instance().add_path(*it);
    }
} // game::init_resource_pool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the names of the levels and their paths.
 */
void bear::engine::game::load_level_files()
{
  std::stringstream f;
  const std::string filename("levels.txt");

  resource_pool::get_instance().get_file(filename, f);

  if ( !f )
    claw::logger << claw::log_error 
                 << "Can't find the level list in file '" << filename << "'."
                 << std::endl;
  else
    {
      std::string line;
      std::size_t line_number(1);

      for ( ; claw::text::getline(f, line); ++line_number )
        {
          claw::text::trim( line, " \t" );

          if ( !line.empty() )
            if ( line[0] != '#' )
              {
                claw::logger << claw::log_verbose << "Add level '" << line
                             << "'" << std::endl;

                if ( !m_game_description.add_level(line) )
                  claw::logger << claw::log_error
                               << filename << ':' << line_number
                               << ": Bad line format. "
                               << "Must be \'level_name:file_path'"
                               << std::endl;
              }
        }
    }
} // game::load_level_files()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do the pending actions.
 */
bool bear::engine::game::do_post_actions()
{
  bool result = false;

  while ( !m_post_actions.empty() )
    {
      game_action* a=m_post_actions.front();
      m_post_actions.pop();

      result = a->apply(*this);

      delete a;
    }

  return result;
} // game::do_post_actions()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the current level (and delete the level currently running).
 * \param the_level The new current level.
 */
void bear::engine::game::set_current_level( level* the_level )
{
  CLAW_PRECOND( the_level != NULL );

  if ( m_current_level != NULL )
    close_level();

  m_current_level = the_level;

  start_current_level();
} // game::set_current_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a level.
 * \param name The name of the level to load.
 */
void bear::engine::game::load_level( const std::string& name )
{
  claw::logger << claw::log_verbose << "------------ Loading level '"
               << name << "'... ------------" << std::endl;

  if ( !level_exists(name) )
    throw CLAW_EXCEPTION
      ( "Can't open level file '" + get_level_file(name) + "'." );
  else
    {
      std::stringstream f;
      resource_pool::get_instance().get_file( get_level_file(name), f );

      /// \todo test the file to see if it's text or binary
      compiled_file level_file( f, true );

      level_loader loader(level_file, name);
      loader.complete_run();

      set_current_level( loader.drop_level() );
    }
} // game::load_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close the current level.
 * \pre m_current_level != NULL
 * \post m_current_level == NULL
 */
void bear::engine::game::close_level()
{
  CLAW_PRECOND( m_current_level != NULL );

  delete m_current_level;
  m_current_level = NULL;

  CLAW_POSTCOND( m_current_level == NULL );
} // game::close_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a level on the stack.
 * \param name The name of the level to load.
 */
void bear::engine::game::do_push_level( const std::string& name )
{
  claw::logger << claw::log_verbose
               << "------------ Pushing '" << name << "'. ------------"
               << std::endl;

  CLAW_PRECOND( m_level_in_abeyance == NULL );

  m_level_in_abeyance = m_current_level;
  m_level_in_abeyance->set_pause();
  m_current_level = NULL;

  load_level( name );

  CLAW_POSTCOND( m_level_in_abeyance != NULL );
} // game::do_push_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Restore the level on the top of the stack.
 */
void bear::engine::game::do_pop_level()
{
  claw::logger << claw::log_verbose
               << "------------ Popping. ------------" << std::endl;
  CLAW_PRECOND( m_level_in_abeyance != NULL );
  CLAW_PRECOND( m_current_level != NULL );

  close_level();

  m_current_level = m_level_in_abeyance;
  m_level_in_abeyance = NULL;

  m_current_level->unset_pause();
} // game::do_pop_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the current level.
 */
void bear::engine::game::start_current_level()
{
  claw::logger << claw::log_verbose
               << "------------ Starting level. ------------" << std::endl;

  CLAW_PRECOND( m_current_level != NULL );

  m_current_level->start();
} // game::start_current_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all layers, controllers and the screen.
 */
void bear::engine::game::clear()
{
  if ( m_current_level != NULL )
    {
      if ( m_level_in_abeyance )
        pop_level();

      close_level();
    }

  if (m_screen != NULL)
    {
      delete m_screen;
      m_screen = NULL;
    }

  // deleting pending actions
  while ( !m_post_actions.empty() )
    {
      delete m_post_actions.front();
      m_post_actions.pop();
    }
} // game::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check the arguments to initialise the game.
 * \param argc Number of program arguments.
 * \param argv Program arguments.
 */
bool bear::engine::game::check_arguments( int& argc, char** &argv )
{
  bool result = false;
  std::string help;
  bool auto_exit(false);
  claw::arguments_table arg( get_arguments_table() );
  char game_var_assignment('=');

  arg.parse( argc, argv );

  if ( arg.get_bool("--version") )
    {
      std::cout << BEAR_VERSION_STRING << std::endl;
      auto_exit = true;
    }

  if ( arg.has_value("--game-name") )
    m_game_description.set_game_name( arg.get_string("--game-name") );

  if ( arg.has_value("--active-area") )
    {
      if ( arg.only_integer_values("--active-area") )
        m_game_description.set_active_area_margin
          ( arg.get_integer("--active-area") );
      else
        help = "--active-area=" + arg.get_string("--active-area");
    }

  m_fullscreen = arg.get_bool("--fullscreen") && !arg.get_bool("--windowed");

  if ( arg.has_value("--screen-height") )
    {
      if ( arg.only_integer_values("--screen-height") )
        m_game_description.set_screen_height
          ( arg.get_integer("--screen-height") );
      else
        help = "--screen-height=" + arg.get_string("--screen-height");
    }

  if ( arg.has_value("--screen-width") )
    {
      if ( arg.only_integer_values("--screen-width") )
        m_game_description.set_screen_width
          ( arg.get_integer("--screen-width") );
      else
        help = "--screen-width=" + arg.get_string("--screen-width");
    }

  if ( arg.has_value("--game-var-assignment") )
    {
      const std::string v( arg.get_string("--game-var-assignment") );

      if ( v.length() == 1 )
        game_var_assignment = v[0];
      else
        help = "--game-var-assignment: not a character.";
    }

  if ( arg.has_value("--set-game-var-int") )
    if ( !game_set_game_variable_from_arg<int>
         ( arg.get_all_of_string("--set-game-var-int"), game_var_assignment ) )
      help = "--set-game-var-int: not an integer";

  if ( arg.has_value("--set-game-var-uint") )
    if ( !game_set_game_variable_from_arg<unsigned int>
         ( arg.get_all_of_string("--set-game-var-uint"), game_var_assignment ) )
      help = "--set-game-var-uint: not an unsigned integer";

  if ( arg.has_value("--set-game-var-bool") )
    if ( !game_set_game_variable_from_arg<bool>
         ( arg.get_all_of_string("--set-game-var-bool"), game_var_assignment ) )
      help = "--set-game-var-bool: not a boolean";

  if ( arg.has_value("--set-game-var-real") )
    if ( !game_set_game_variable_from_arg<double>
         ( arg.get_all_of_string("--set-game-var-real"), game_var_assignment ) )
      help = "--set-game-var-real: not a real number";

  if ( arg.has_value("--set-game-var-string") )
    game_set_game_variable_from_arg<std::string>
      ( arg.get_all_of_string("--set-game-var-string"), game_var_assignment );

  if ( arg.has_value("--start-level") )
    m_game_description.set_start_level( arg.get_string("--start-level") );
  else
    help = "--start-level";

  if ( !help.empty() )
    {
      std::cout << "Bad argument value: '" << help << "'\n";
      arg.help();
    }
  else if ( !auto_exit )
    {
      load_libraries( arg.get_all_of_string("--item-library") );
      init_resource_pool( arg.get_all_of_string("--data-path") );
      load_level_files();
      result = true;
    }

  return result;
} // game::check_arguments()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the arguments table with the arguments of the engine.
 */
claw::arguments_table bear::engine::game::get_arguments_table()
{
  claw::arguments_table arg("Engine's options:");

  arg.add_long("--game-name", "The name of the game.", true, "string");
  arg.add_long
    ("--active-area",
     "The margin around the camera in which we check for activity.", true,
     "integer");
  arg.add_long("--screen-width", "The width of the screen.", true, "integer");
  arg.add_long
    ("--screen-height", "The height of the screen.", true, "integer");
  arg.add_long("--fullscreen", "Run the game in fullscreen mode.", true);
  arg.add_long("--windowed", "Run the game in a window.", true);
  arg.add_long
    ("--data-path", "Path to the directory containing the data of the game.",
     false, "path");
  arg.add_long
    ("--item-library", "Path to a library containing items for the game.",
     false, "path");
  arg.add_long
    ("--start-level", "Name of the first level to run.", false, "string");
  arg.add_long
    ("--set-game-var-int",
     "Set the value of an integer game variable.", true, "name=value");
  arg.add_long
    ("--set-game-var-uint",
     "Set the value of a non negative integer game variable.", true,
     "name=value");
  arg.add_long
    ("--set-game-var-bool", "Set the value of a boolean game variable.", true,
     "name=value");
  arg.add_long
    ("--set-game-var-real", "Set the value of a real number game variable.",
     true, "name=value");
  arg.add_long
    ("--set-game-var-string", "Set the value of a string game variable.", true,
     "name=value");
  arg.add_long
    ("--game-var-assignment",
     "Change the delimiter used in --set-game-var-<type> to separate the name"
     " and the value of the variable.", true, "character");
  arg.add("-v", "--version", "Print the version of the engine and exit.", true);

  return arg;
} // game::get_arguments_table()
