/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file block_align_top.cpp
 * \brief Implementation of the bear::block_align_top class.
 * \author Sebastien Angibaud
 */
#include "generic_items/block_align_top.hpp"

#include "universe/collision_info.hpp"
#include "universe/collision_repair.hpp"
#include "engine/export.hpp"

BASE_ITEM_EXPORT( block_align_top, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 */
bear::block_align_top::block_align_top()
  : m_top_friction(1)
{

} // block_align_top::block_align_top()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the collision is on a solid side and align the other item.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void bear::block_align_top::collision_check_and_align
( engine::base_item& that, universe::collision_info& info )
{
  if ( satisfy_collision_condition(that) )
    if ( ( that.get_left() + that.get_size().x/2 >= get_left() ) &&
	 ( that.get_left() + that.get_size().x/2 <= get_right() ) )
    {
      bear::universe::position_type pos(info.get_bottom_left_on_contact());
      pos.y = get_top()+0.1;

      if ( ! info.other_item().is_phantom() )
	{
	  info.other_item().set_bottom_left( pos );
	  info.other_item().set_bottom_contact();
	  set_top_contact();
	  
	  info.get_collision_repair().set_contact_normal
	    (info.other_item(), bear::universe::vector_type(0, 1));
	  
	  that.set_contact_friction(m_top_friction);
	  z_shift(that);
	}
    }
} // block_align_top::collision_check_and_align()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call collision_check_and_align().
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void bear::block_align_top::collision
( engine::base_item& that, universe::collision_info& info )
{
  collision_check_and_align(that, info);
} // block_align_top::collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c real.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::block_align_top::set_real_field( const std::string& name, double value )
{
  bool result = true;

  if (name == "top_friction")
    m_top_friction = value;
  else
    result = super::set_real_field(name,value);

  return result;
} // block_aligne_top::set_real_field()
