/*
  Bear Engine - Editor library

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/impl/item_instance.tpp
 * \brief Implementation of the template methods of the bf::item_instance class.
 * \author Julien Jorge
 */
#include "bf/item_class_pool.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Extract the names of defined fields.
 * \param m The map in which we take the fields names.
 * \param fields The set in which the fields are saved.
 */
template<typename MapType>
void bf::item_instance::copy_field_names
( const MapType& m, std::set<std::string>& fields ) const
{
  typename MapType::const_iterator it;

  for (it=m.begin(); it!=m.end(); ++it)
    fields.insert(it->first);
} // item_instance::copy_field_names()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a list of values.
 * \param f The file in which we compile.
 * \param v The values to save.
 */
template<typename T>
void bf::item_instance::compile_list
( compiled_file& f, const std::list<T>& v ) const
{
  f << v.size();

  typename std::list<T>::const_iterator it = v.begin();
  const typename std::list<T>::const_iterator eit = v.end();

  for ( ; it!=eit; ++it)
    it->compile(f);
} // item_instance::compile_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove the field values that are not available for the class of this
 *        item.
 * \param m The map of values.
 * \param ml The map of list of values.
 * \param t The type of the fields explored.
 */
template<typename T>
void bf::item_instance::remove_invalid_values
( std::map<std::string, T>& m, std::map< std::string, std::list<T> >& ml,
  type_field::field_type t ) const
{
  std::list<std::string> to_remove;
  typename std::map<std::string, T>::const_iterator it;

  for (it=m.begin(); it!=m.end(); ++it)
    if ( !m_class->has_field( it->first, t ) )
      to_remove.push_front(it->first);
    else
      {
        const type_field& f = m_class->get_field(it->first);
        if ( f.is_list() )
          to_remove.push_front(it->first);
      }

  for ( ; !to_remove.empty(); to_remove.pop_front() )
    m.erase( to_remove.front() );

  typename std::map< std::string, std::list<T> >::const_iterator itl;

  for (itl=ml.begin(); itl!=ml.end(); ++itl)
    if ( !m_class->has_field( itl->first, t ) )
      to_remove.push_front(itl->first);
    else
      {
        const type_field& f = m_class->get_field(it->first);
        if ( !f.is_list() )
          to_remove.push_front(it->first);
      }

  for ( ; !to_remove.empty(); to_remove.pop_front() )
    ml.erase( to_remove.front() );
} // item_instance::remove_invalid_values()
