/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame_main_menu.cpp
 * \brief Implementation of the ptb::frame_main_menu class.
 * \author Julien Jorge
 */
#include "ptb/frame/frame_main_menu.hpp"

#include "engine/game.hpp"
#include "ptb/game_variables.hpp"
#include "ptb/frame/frame_start_menu.hpp"
#include "ptb/frame/frame_configure.hpp"

#include <libintl.h>

/*----------------------------------------------------------------------------*/
const unsigned int ptb::frame_main_menu::s_pos_start = 0;
const unsigned int ptb::frame_main_menu::s_pos_start_tutorial = 1;
const unsigned int ptb::frame_main_menu::s_pos_configure = 2;
const unsigned int ptb::frame_main_menu::s_pos_quit = 3;

const bear::input::key_code ptb::frame_main_menu::s_keyboard_cheat[] =
  {
    bear::input::keyboard::kc_up, bear::input::keyboard::kc_up,
    bear::input::keyboard::kc_down, bear::input::keyboard::kc_left,
    bear::input::keyboard::kc_right, bear::input::keyboard::kc_down,
    bear::input::keyboard::kc_not_a_key
  };

const bear::input::joystick::joy_code ptb::frame_main_menu::s_joystick_cheat[] =
  {
    bear::input::joystick::jc_axis_up, bear::input::joystick::jc_axis_up,
    bear::input::joystick::jc_axis_down, bear::input::joystick::jc_axis_left,
    bear::input::joystick::jc_axis_right, bear::input::joystick::jc_axis_down,
    bear::input::joystick::jc_invalid
  };

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owning_layer The layer onwing the window.
 */
ptb::frame_main_menu::frame_main_menu( windows_layer* owning_layer )
  : menu_frame(owning_layer), m_cheat_index(0)
{
  create_controls();
  set_input_priority(true);
} // frame_main_menu::frame_main_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
bool ptb::frame_main_menu::on_key_press( const bear::input::key_info& key )
{
  if (key.get_code() == s_keyboard_cheat[m_cheat_index])
    {
      ++m_cheat_index;

      if ( s_keyboard_cheat[m_cheat_index]
           == bear::input::keyboard::kc_not_a_key )
        game_variables::set_password_menu_visibility(true);
    }
  else
    m_cheat_index = 0;

  return menu_frame::on_key_press(key);
} // frame_main_menu::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool ptb::frame_main_menu::on_button_press
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  if (button == s_joystick_cheat[m_cheat_index])
    {
      ++m_cheat_index;

      if ( s_joystick_cheat[m_cheat_index]
           == bear::input::joystick::jc_invalid )
        game_variables::set_password_menu_visibility(true);
    }
  else
    m_cheat_index = 0;

  return menu_frame::on_button_press(button, joy_index);
} // frame_main_menu::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Validate the selected action.
 */
void ptb::frame_main_menu::validate() const
{
  switch ( get_menu().cursor_position() )
    {
    case s_pos_start:
      {
        game_variables::set_next_level_name( "intro" );
        show_window( new frame_start_menu(m_owning_layer, false) );
        break;
      }
    case s_pos_start_tutorial:
      {
        game_variables::set_next_level_name( "tutorial" );
        show_window( new frame_start_menu(m_owning_layer, true) );
        break;
      }
    case s_pos_configure:
      show_window( new frame_configure(m_owning_layer) ); break;
    case s_pos_quit:
      bear::engine::game::get_instance().end();
      break;
    }
} // frame_main_menu::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the static text components.
 */
void ptb::frame_main_menu::create_controls()
{
  std::vector<std::string> strings;

  strings.reserve(4);

  strings.push_back( gettext("Start game") );
  strings.push_back( gettext("Start tutorial") );
  strings.push_back( gettext("Configuration") );
  strings.push_back( gettext("Quit") );

  get_menu().make(strings);
  set_size( get_menu().get_size() );
} // frame_main_menu::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called before closing the window.
 * \return True if the window can be closed.
 */
bool ptb::frame_main_menu::on_close()
{
  return false;
} // frame_main_menu::on_close()
