/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file engine/code/world.cpp
 * \brief Implementation of the bear::engine::world class.
 * \author Julien Jorge
 */
#include "engine/world.hpp"
#include "engine/base_item.hpp"
#include "engine/game.hpp"
#include "engine/population.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param pos The position where the items are searched.
 */
bear::engine::world::msg_pick_items::msg_pick_items
( const universe::position_type& pos )
  : m_pos(pos)
{

} // msg_pick_items::msg_pick_items()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to the world.
 * \param that The world to apply the message to.
 */
bool bear::engine::world::msg_pick_items::apply_to( world& that )
{
  that.pick_items_by_position( items, m_pos );
  return true;
} // msg_pick_items::apply_to()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param r The region where the items are searched.
 */
bear::engine::world::msg_pick_items_in_region::msg_pick_items_in_region
( const universe::rectangle_type& r )
  : m_region(r)
{

} // msg_pick_items_in_region::msg_pick_items_in_region()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to the world.
 * \param that The world to apply the message to.
 */
bool bear::engine::world::msg_pick_items_in_region::apply_to( world& that )
{
  that.pick_items_in_rectangle( items, m_region );
  return true;
} // msg_pick_items_in_region::apply_to()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor
 * \param size Size of the world.
 */
bear::engine::world::world( const universe::size_box_type& size )
  : universe::world(size), communication::messageable("world")
{

} // world::world()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor
 * \param size Size of the world.
 */
bear::engine::world::~world()
{
  for ( ; !m_static_items.empty(); m_static_items.pop_front() )
    delete m_static_items.front();
} // world::~world()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the "start()" method on all items.
 */
void bear::engine::world::start()
{
  print_stats();
} // world::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the progress() method on entities in the active region, then
 *        apply physic rules.
 * \param regions The active regions.
 * \param elapsed_time Elapsed time since the last call of this method.
 */
void bear::engine::world::progress_entities
( const region_type& regions, universe::time_type elapsed_time )
{
  universe::world::progress_entities(regions, elapsed_time);
  m_population.remove_dead_items();
} // world::progress_entities()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprites of the visible items in the active region.
 * \param visuals (out) The sprites in the visible area, and their positions.
 * \param camera_box The part of the world visible through the camera.
 */
void bear::engine::world::get_visual
( std::list<scene_visual>& visuals,
  const universe::rectangle_type& camera_box ) const
{
  item_list items;
  item_list::const_iterator it;

  pick_items_in_rectangle(items, camera_box);

  for (it=items.begin(); it!=items.end(); ++it)
    {
      base_item* i( dynamic_cast<base_item*>(*it) );

      if ( i==NULL )
        claw::logger << claw::log_warning
                     << "bear::engine::world::get_visual(): "
                     << "item is not a base_item. Not rendered: "
                     << **it << std::endl;
      else
        i->insert_visual( visuals );
    }
} // world::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a permanent and fixed item in the world.
 */
void bear::engine::world::add_static( base_item* const& who )
{
  universe::world::add_static(who);
  m_static_items.push_front( who );
} // world::add_static()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item in the world.
 */
void bear::engine::world::register_item( base_item* const& who )
{
  /* !!Julien. January 2010, 6th.

     The base_item::kill() method may have been called before adding the item
     (in base_item::build(), for example). In such case, the item is not added
     to the world, but we still insert him in the population to kill him
     immediately. */

  m_population.insert( who );

  if ( !who->is_dead() )
    universe::world::register_item(who);
  else
    m_population.kill( who );
} // world::register_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call universe::world::release_item(who)
 */
void bear::engine::world::release_item( base_item* const& who )
{
  m_population.kill(who);
  universe::world::release_item(who);
} // world::release_item()
