/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file script/code/string_to_arg.cpp
 * \brief Implementation of the specialisations of the
 *        text_interface::string_to_arg class.
 * \author Julien Jorge
 */
#include "engine/script/string_to_arg.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string naming a base_item to a reference to this base_item,
 *        to pass as an argument for a character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
const bear::engine::base_item&
bear::text_interface::string_to_arg
<bear::engine::script_context, const bear::engine::base_item&>::convert
( const engine::script_context& c, const std::string& arg )
{
  const engine::base_item* item( c.get_actor_item(arg) );

  if ( item == NULL )
    {
      claw::logger << claw::log_warning << "No item named '" << arg << "'"
                   << std::endl;
      throw std::invalid_argument("No item named '" + arg + "'");
    }
  else
    return *item;
} // string_to_arg::convert()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string naming a base_item to a reference to this base_item,
 *        to pass as an argument for a character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
bear::engine::base_item&
bear::text_interface::string_to_arg
<bear::engine::script_context, bear::engine::base_item&>::convert
( const engine::script_context& c, const std::string& arg )
{
  engine::base_item* item( c.get_actor_item(arg) );

  if ( item == NULL )
    {
      claw::logger << claw::log_warning << "No item named '" << arg << "'"
                   << std::endl;
      throw std::invalid_argument("No item named '" + arg + "'");
    }
  else
    return *item;
} // string_to_arg::convert()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string naming a base_item to a reference to this base_item,
 *        to pass as an argument for a character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
const bear::engine::script_runner&
bear::text_interface::string_to_arg
<bear::engine::script_context, const bear::engine::script_runner&>::convert
( const engine::script_context& c, const std::string& arg )
{
  const text_interface::base_exportable<engine::script_context>*
    item( c.get_actor(arg) );

  if ( item == NULL )
    {
      claw::logger << claw::log_warning << "No item named '"
                   << arg << "'" << std::endl;
      throw std::invalid_argument("No item named '" + arg + "'");
    }
  else
    {
      const bear::engine::script_runner* script =
        dynamic_cast<const bear::engine::script_runner*>(item);

      if ( script != NULL )
        return *script;
      else
        {
          claw::logger << claw::log_warning << "No script runner named '"
                       << arg << "'" << std::endl;
          throw std::invalid_argument("No script runner named '" +arg +"'");
        }
    }
} // string_to_arg::convert()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string naming a base_item to a reference to this base_item,
 *        to pass as an argument for a character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
bear::engine::script_runner&
bear::text_interface::string_to_arg
<bear::engine::script_context, bear::engine::script_runner&>::convert
( const engine::script_context& c, const std::string& arg )
{
  text_interface::base_exportable<engine::script_context>*
    item( c.get_actor(arg) );

  if ( item == NULL )
    {
      claw::logger << claw::log_warning << "No item named '"
                   << arg << "'" << std::endl;
      throw std::invalid_argument("No item named '" + arg + "'");
    }
  else
    {
      bear::engine::script_runner* script =
        dynamic_cast<bear::engine::script_runner*>(item);

      if ( script != NULL )
        return *script;
      else
        {
          claw::logger << claw::log_warning << "No script runner named '"
                       << arg << "'" << std::endl;
          throw std::invalid_argument("No script runner named '" +arg +"'");
        }
    }
} // string_to_arg::convert()
