/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file string_to_arg.hpp
 * \brief Convert a string to the equivalent value to pass as an argument for a
 *        character method.
 * \author Julien Jorge.
 */
#ifndef __TEXT_INTERFACE_STRING_TO_ARG_HPP__
#define __TEXT_INTERFACE_STRING_TO_ARG_HPP__

#include "text_interface/get_inner_type.hpp"

#include <string>
#include <vector>
#include <list>

namespace bear
{
  namespace text_interface
  {
    /**
     * \brief Convert a string to the equivalent value to pass as an argument
     *        for a character method.
     * \author Julien Jorge.
     */
    template<typename Context, typename T>
    class string_to_arg
    {
    public:
      typedef typename get_inner_type<T>::type result_type;

      static result_type convert( const Context& c, const std::string& arg );

    }; // class string_to_arg

    /**
     * \brief Specialisation for std::string. There is no conversion to do.
     * \author Sbastien Angibaud
     */
    template<typename Context>
    class string_to_arg<Context, std::string>
    {
    public:
      typedef typename get_inner_type<std::string>::type result_type;

      static result_type convert( const Context& c, const std::string& arg );
    }; // class string_to_arg [std::string]

    /**
     * \brief Specialisation for std::string. There is no conversion to do.
     * \author Sbastien Angibaud
     */
    template<typename Context>
    class string_to_arg<Context, const std::string>
    {
    public:
      typedef typename get_inner_type<const std::string>::type result_type;

      static result_type convert( const Context& c, const std::string& arg );
    }; // class string_to_arg [const std::string]

    /**
     * \brief Specialisation for std::string. There is no conversion to do.
     * \author Sbastien Angibaud
     */
    template<typename Context>
    class string_to_arg<Context, const std::string&>
    {
    public:
      typedef typename get_inner_type<const std::string&>::type result_type;

      static result_type convert( const Context& c, const std::string& arg );
    }; // class string_to_arg [const std::string&]

    /**
     * \brief Convert a string to the equivalent sequence of value to pass as an
     *        argument for a character method.
     * \author Julien Jorge
     */
    template<typename Context, typename Sequence>
    class string_to_sequence_arg
    {
    public:
      typedef typename get_inner_type<Sequence>::type result_type;

      static result_type convert( const Context& c, const std::string& arg );
    }; // class string_to_sequence_arg

    /**
     * \brief Specialisation for std::vector.
     * \author Julien Jorge
     */
    template<typename Context, typename T>
    class string_to_arg< Context, std::vector<T> >:
      public string_to_sequence_arg< Context, std::vector<T> >
    { };

    /**
     * \brief Specialisation for const std::vector&.
     * \author Julien Jorge
     */
    template<typename Context, typename T>
    class string_to_arg< Context, const std::vector<T>& >:
      public string_to_sequence_arg< Context, const std::vector<T>& >
    { };

    /**
     * \brief Specialisation for const std::vector.
     * \author Julien Jorge
     */
    template<typename Context, typename T>
    class string_to_arg< Context, const std::vector<T> >:
      public string_to_sequence_arg< Context, const std::vector<T> >
    { };

    /**
     * \brief Specialisation for std::list.
     * \author Julien Jorge
     */
    template<typename Context, typename T>
    class string_to_arg< Context, std::list<T> >:
      public string_to_sequence_arg< Context, std::list<T> >
    { };

    /**
     * \brief Specialisation for const std::list&.
     * \author Julien Jorge
     */
    template<typename Context, typename T>
    class string_to_arg< Context, const std::list<T>& >:
      public string_to_sequence_arg< Context, const std::list<T>& >
    { };

    /**
     * \brief Specialisation for const std::list.
     * \author Julien Jorge
     */
    template<typename Context, typename T>
    class string_to_arg< Context, const std::list<T> >:
      public string_to_sequence_arg< Context, const std::list<T> >
    { };

  } // namespace text_interface
} // namespace bear

#include "text_interface/impl/string_to_arg.tpp"

#endif // __TEXT_INTERFACE_STRING_TO_ARG_HPP__
