/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file delayed_kill_item.cpp
 * \brief Implementation of the bear::delayed_kill_item class.
 * \author Sebastien Angibaud
 */
#include "generic_items/delayed_kill_item.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( delayed_kill_item, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::delayed_kill_item::delayed_kill_item()
  : m_duration(0), m_start_duration(0)
{
  set_phantom(true);
  set_can_move_items(false);
  set_global(true);
} // delayed_kill_item::delayed_kill_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void bear::delayed_kill_item::progress( universe::time_type elapsed_time )
{
  super::progress(elapsed_time);

  m_start_duration += elapsed_time;

  if ( m_start_duration >= m_duration )
    {
      std::vector<handle_type>::iterator it;
      for (it = m_items.begin(); it != m_items.end(); ++it)
        if ( *it != (base_item*)NULL )
          (*it)->kill();

      kill();
    }
} // delayed_kill_item::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type list of <*base_item>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::delayed_kill_item::set_item_list_field
( const std::string& name, const std::vector<base_item*>& value )
{
  bool result = false;

  if ( name == "delayed_kill_item.killing_items" )
    {
      m_items.resize(value.size());

      for ( std::size_t i=0; i!=value.size(); ++i )
        m_items[i] = value[i];

      result = true;
    }
  else
    result = super::set_item_list_field( name, value );

  return result;
} // delayed_kill_item::set_item_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <real>.
 * \param value The value of the field.
 */
bool bear::delayed_kill_item::set_real_field
( const std::string& name, double value )
{
  bool result = true;

  if ( name == "delayed_kill_item.duration" )
    m_duration = value;
  else
    result = super::set_real_field(name, value);

  return result;
} // delayed_kill_item::set_real_field()
