/*
    Bear Engine - Editor library

    Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/impl/base_file_edit.tpp
 * \brief Implementation of the bf::base_file_edit class.
 * \author Julien Jorge
 */

#include "bf/path_configuration.hpp"

#include <wx/filedlg.h>
#include <wx/sizer.h>
#include <wx/filename.h>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The window owning this one.
 * \param filter The filter of the file selection dialog.
 * \param v The initial value.
 */
template<typename T>
bf::base_file_edit<T>::base_file_edit
( wxWindow& parent, const wxString& filter, const file_type& v )
  : super(v), wxPanel(&parent), m_filter(filter)
{
  create_controls();
  Fit();
  value_updated();
} // base_file_edit::base_file_edit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the displayed value is correct and, if it is, set the
 *        value according to the display.
 */
template<typename T>
bool bf::base_file_edit<T>::validate()
{
  this->value_from_string( m_path->GetValue() );
  return true;
} // base_file_edit::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the controls with the new value.
 */
template<typename T>
void bf::base_file_edit<T>::value_updated()
{
  m_path->SetValue( std_to_wx_string(this->get_value().get_value()) );
} // base_file_edit::value_updated()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the edit.
 */
template<typename T>
void bf::base_file_edit<T>::create_controls()
{
  m_path = new wxTextCtrl(this, wxID_ANY);
  m_browse = new wxButton
    ( this, IDC_BROWSE, wxT("..."), wxDefaultPosition, wxSize(30, -1) );

  this->Connect( m_browse->GetId(), wxEVT_COMMAND_BUTTON_CLICKED,
                 wxCommandEventHandler(self_type::on_browse) );

  wxBoxSizer* sizer = new wxBoxSizer( wxHORIZONTAL );
  sizer->Add( m_path, 1, wxEXPAND );
  sizer->Add( m_browse, 0, wxEXPAND );

  this->SetSizer( sizer );
} // base_file_edit::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user clicked on the "browse" button.
 * \param event The event generated by the user's action.
 */
template<typename T>
void bf::base_file_edit<T>::on_browse( wxCommandEvent& WXUNUSED(event) )
{
  std::string p = wx_to_std_string(m_path->GetValue());
  path_configuration::get_instance().get_full_path(p);

  wxFileDialog dlg
    ( this, _("Choose a file"), wxEmptyString, std_to_wx_string(p), m_filter,
      wxFD_DEFAULT_STYLE | wxFD_FILE_MUST_EXIST );

  if (dlg.ShowModal() == wxID_OK)
    {
      std::string new_p = wx_to_std_string( dlg.GetPath() );
      path_configuration::get_instance().get_relative_path(new_p);

      m_path->SetValue( std_to_wx_string(new_p) );
    }
} // base_file_edit::on_browse()
