/*
    Bear Engine - Model editor

    Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/snapshot_frame.cpp
 * \brief Implementation of the bf::snapshot_frame class.
 * \author Sbastien Angibaud
 */
#include "bf/snapshot_frame.hpp"

#include "bf/gui_model.hpp"
#include "bf/mark.hpp"
#include "bf/model_frame.hpp"
#include "bf/snapshot.hpp"
#include "bf/snapshot_date_frame.hpp"
#include "bf/wx_facilities.hpp"

#include "bf/history/action_modify_placement.hpp"
#include "bf/history/action_add_snapshot.hpp"
#include "bf/history/action_move_snapshot.hpp"
#include "bf/history/action_remove_snapshot.hpp"
#include "bf/history/action_modify_function.hpp"
#include "bf/history/action_modify_snapshot_alignment.hpp"
#include "bf/history/action_modify_snapshot_size.hpp"
#include "bf/history/action_modify_snapshot_sound.hpp"
#include "bf/history/action_group.hpp"

#include <limits>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The frame owning this one.
 */
bf::snapshot_frame::snapshot_frame( wxWindow* parent )
  : wxPanel( parent ), m_model_frame(NULL), m_snapshot(NULL)
{
  create_controls();
} // snapshot_frame::snapshot_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the model on which we work.
 * \param f The model frame in which we take the actions.
 */
void bf::snapshot_frame::set_model_frame( model_frame* f )
{
  m_model_frame = f;
  fill();
} // snapshot_frame::set_model_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update when the current action is modified.
 */
void bf::snapshot_frame::update_action( )
{
  fill();
} // snapshot_frame::update_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update when the current snapshot is modified.
 */
void bf::snapshot_frame::update_snapshot( )
{
  fill();
} // snapshot_frame::update_snapshot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update when the current mark is modified.
 */
void bf::snapshot_frame::update_mark( )
{
  fill();
} // snapshot_frame::update_mark()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 */
void bf::snapshot_frame::create_controls()
{
  create_member_controls();
  create_sizer_controls();
} // snapshot_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls for which we keep a reference.
 */
void bf::snapshot_frame::create_member_controls()
{
  m_date_button = new wxButton( this, IDC_DATE_CTRL, wxT("0"));

  m_function_button = new wxButton( this, IDC_FUNCTION, wxEmptyString );

  m_mark_choice = new wxChoice( this, IDC_MARK_CHOICE );

  m_mark_left_spin = new spin_ctrl<double>
    ( this, IDC_MARK_PLACEMENT, wxDefaultPosition,
      wxDefaultSize, 0, - std::numeric_limits<double>::infinity(),
      std::numeric_limits<double>::infinity(), 0, 1 );
  m_mark_bottom_spin = new spin_ctrl<double>
    ( this, IDC_MARK_PLACEMENT, wxDefaultPosition,
      wxDefaultSize, 0, - std::numeric_limits<double>::infinity(),
      std::numeric_limits<double>::infinity(), 0, 1  );
  m_mark_depth_spin = new wxSpinCtrl( this, IDC_MARK_PLACEMENT );
  m_mark_angle_spin = new spin_ctrl<double>
    ( this, IDC_MARK_PLACEMENT, wxDefaultPosition,
      wxDefaultSize, 0, - std::numeric_limits<double>::infinity(),
      std::numeric_limits<double>::infinity(), 0, 0.01 );

  m_mark_depth_spin->SetRange
    ( std::numeric_limits<int>::min(), std::numeric_limits<int>::max() );
  m_mark_visible_box = new wxCheckBox( this, IDC_VISIBILITY,_("Visible") );

  m_sound_button = new wxButton(this, IDC_SOUND_FILE, wxT("..."));
  m_sound_path = new wxTextCtrl(this, IDC_SOUND_TEXT);
  m_globally_sound_box =
    new wxCheckBox( this, IDC_GLOBALLY_SOUND,_("Play globally") );

  m_width_spin = new spin_ctrl<double>
    ( this, IDC_SIZE, wxDefaultPosition,
      wxDefaultSize, 0, 0, std::numeric_limits<double>::infinity(), 0, 1 );
  m_height_spin = new spin_ctrl<double>
    ( this, IDC_SIZE, wxDefaultPosition,
      wxDefaultSize, 0, 0, std::numeric_limits<double>::infinity(), 0, 1 );

   m_x_alignment_value_spin = new spin_ctrl<double>
    ( this, IDC_ALIGNMENT, wxDefaultPosition,
      wxDefaultSize, 0, -std::numeric_limits<double>::infinity(),
      std::numeric_limits<double>::infinity(), 0, 1 );

  m_y_alignment_value_spin = new spin_ctrl<double>
    ( this, IDC_ALIGNMENT, wxDefaultPosition,
      wxDefaultSize, 0, -std::numeric_limits<double>::infinity(),
      std::numeric_limits<double>::infinity(), 0, 1 );

  wxArrayString array;
  array.Add(wxT("align_bottom"));
  array.Add(wxT("align_center"));
  array.Add(wxT("align_top"));
  m_y_alignment_choice = new wxChoice
    ( this, IDC_ALIGNMENT, wxDefaultPosition, wxDefaultSize, array );

  array.Clear();
  array.Add(wxT("align_left"));
  array.Add(wxT("align_center"));
  array.Add(wxT("align_right"));
  m_x_alignment_choice = new wxChoice
    ( this, IDC_ALIGNMENT, wxDefaultPosition, wxDefaultSize, array );

  enable_controls(false);
} // snapshot_frame::create_member_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and add them in sizers.
 */
void bf::snapshot_frame::create_sizer_controls()
{
   wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );

  // width
  wxBoxSizer* s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Date:")),
                0, wxEXPAND | wxALL, 5 );
  s_sizer->Add( m_date_button, 0, wxEXPAND | wxALIGN_LEFT );
  sizer->Add( s_sizer, 0, wxEXPAND );

  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Function:")),
                0, wxEXPAND | wxALL, 5 );
  s_sizer->Add( m_function_button, 1, wxEXPAND | wxALIGN_LEFT );
  sizer->Add( s_sizer, 0, wxEXPAND );

  sizer->Add( create_marks_sizer(), 0, wxEXPAND );

  sizer->Add( create_sound_sizer(), 0, wxEXPAND );

  wxStaticBoxSizer* static_sizer;
  static_sizer = new wxStaticBoxSizer( wxHORIZONTAL, this, _("Size") );
  static_sizer->Add( new wxStaticText(this, wxID_ANY, _("Width:")), 0,
                     wxALIGN_CENTER_VERTICAL );
  static_sizer->Add( m_width_spin, 1 );
  static_sizer->AddSpacer(5);
  static_sizer->Add( new wxStaticText(this, wxID_ANY, _("Height:")), 0,
                     wxALIGN_CENTER_VERTICAL );
  static_sizer->Add( m_height_spin, 1 );
  sizer->Add( static_sizer, 0, wxEXPAND );

  static_sizer =
    new wxStaticBoxSizer( wxVERTICAL, this, _("Alignments") );
  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("X:")), 0,
                     wxALIGN_CENTER_VERTICAL );
  s_sizer->Add( m_x_alignment_choice, 1, wxALL );
  s_sizer->AddSpacer(5);
  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Gap:")), 0,
                     wxALIGN_CENTER_VERTICAL );
  s_sizer->Add( m_x_alignment_value_spin, 1 );
  static_sizer->Add( s_sizer, 0, wxEXPAND );
  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Y:")), 0,
                     wxALIGN_CENTER_VERTICAL );
  s_sizer->Add( m_y_alignment_choice, 1, wxALL );
  s_sizer->AddSpacer(5);
  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Gap:")), 0,
                     wxALIGN_CENTER_VERTICAL );
  s_sizer->Add( m_y_alignment_value_spin, 1 );

  static_sizer->Add( s_sizer, 0, wxEXPAND );
  sizer->Add( static_sizer, 0, wxEXPAND );

  SetSizer(sizer);
} // snapshot_frame::create_sizer_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for marks.
 */
wxSizer* bf::snapshot_frame::create_marks_sizer()
{
  wxStaticBoxSizer* result =
    new wxStaticBoxSizer( wxVERTICAL, this, _("Marks") );

  wxBoxSizer* s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Mark:")),
      0, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_mark_choice, 1, wxALL, 5 );
  result->Add( s_sizer, 0, wxEXPAND, 0 );

  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("X:")),
      0, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_mark_left_spin, 1, wxALL, 5 );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Y:")),
      0, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_mark_bottom_spin, 1, wxALL, 5 );
  result->Add( s_sizer, 0, wxEXPAND, 0 );

  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Depth:")),
      0, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_mark_depth_spin, 1, wxALL, 5 );
  s_sizer->Add
    ( new wxStaticText(this, wxID_ANY, _("Angle:")),
      0, wxALIGN_CENTRE_VERTICAL | wxALL, 5 );
  s_sizer->Add( m_mark_angle_spin, 1, wxALL, 5 );
  result->Add( s_sizer, 0, wxEXPAND, 0 );

  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add( m_mark_visible_box, 0, wxALL, 5 );
  result->Add( s_sizer, 0, wxEXPAND, 0 );

  return result;
} // snapshot_frame::create_marks_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and the sizer for sound.
 */
wxSizer* bf::snapshot_frame::create_sound_sizer()
{
  wxStaticBoxSizer* result =
    new wxStaticBoxSizer( wxVERTICAL, this, _("Sound") );

  wxBoxSizer* s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add( m_sound_path, 1, wxEXPAND );
  s_sizer->Add( m_sound_button , 0, wxEXPAND );

  result->Add( m_globally_sound_box, 0, wxEXPAND );
  result->Add( s_sizer, 0, wxEXPAND );

  return result;
} // snapshot_frame::create_sound_sizer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the list with the marks of the model.
 */
void bf::snapshot_frame::fill()
{
  bool enable = false;
  m_snapshot = NULL;

  if ( m_model_frame != NULL )
    if ( m_model_frame->get_model().has_active_action() )
      {
        m_snapshot = m_model_frame->get_model().get_selected_snapshot();

        if ( m_snapshot != NULL )
          {
            enable = true;

            m_date_button->SetLabel
              ( wxString::Format(wxT("%.3f"), m_snapshot->get_date()) );

            m_function_button->SetLabel
              (std_to_wx_string(m_snapshot->get_function()));

            m_function_button->SetLabel
              (std_to_wx_string(m_snapshot->get_function()));

            m_sound_path->SetValue
              ( std_to_wx_string(m_snapshot->get_sound().get_file()) );
            m_globally_sound_box->SetValue
              ( m_snapshot->get_sound().is_played_globally() );

            m_width_spin->SetValue(m_snapshot->get_width());
            m_height_spin->SetValue(m_snapshot->get_height());
            m_x_alignment_choice->SetStringSelection
              (std_to_wx_string(m_snapshot->get_x_alignment()));
            m_y_alignment_choice->SetStringSelection
              (std_to_wx_string(m_snapshot->get_y_alignment()));
            m_x_alignment_value_spin->SetValue
              (m_snapshot->get_x_alignment_value());
            m_y_alignment_value_spin->SetValue
              (m_snapshot->get_y_alignment_value());
          }
      }

  enable_controls(enable);

  fill_mark();
} // snapshot_frame::fill()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill controls of current mark.
 */
void bf::snapshot_frame::fill_mark()
{
  bool enable = false;

  m_mark_choice->Clear();

  if ( m_model_frame != NULL )
    if ( m_model_frame->get_model().has_active_action() )
      if ( m_snapshot != NULL )
        {
          snapshot::const_mark_placement_iterator it;

          for ( it = m_snapshot->mark_placement_begin();
                it != m_snapshot->mark_placement_end(); ++it )
            m_mark_choice->Append
              ( std_to_wx_string(it->get_mark()->get_label()) );

          if ( m_model_frame->get_model().has_mark_selection() )
            {
              enable = true;
              const mark* m
                ( m_model_frame->get_model().get_main_mark_selection() );

              m_mark_choice->SetStringSelection
                ( std_to_wx_string(m->get_label()) );

              mark_placement p = m_snapshot->get_placement(m->get_label());

              m_mark_left_spin->SetValue(p.get_position_x());
              m_mark_bottom_spin->SetValue(p.get_position_y());
              m_mark_depth_spin->SetValue(p.get_depth_position());
              m_mark_angle_spin->SetValue(p.get_angle());
              m_mark_visible_box->SetValue(p.is_visible());
            }
        }

  m_mark_left_spin->Enable(enable);
  m_mark_bottom_spin->Enable(enable);
  m_mark_depth_spin->Enable(enable);
  m_mark_angle_spin->Enable(enable);
  m_mark_visible_box->Enable(enable);
} // snapshot_frame::fill_mark()

/*----------------------------------------------------------------------------*/
/**
 * \brief The mark placement has changed.
 */
void bf::snapshot_frame::placement_change()
{
  mark* m( m_model_frame->get_model().get_main_mark_selection() );

  m_model_frame->do_action
    ( new action_modify_placement
      (m_snapshot, m, m_mark_left_spin->GetValue(),
       m_mark_bottom_spin->GetValue(), m_mark_depth_spin->GetValue(),
       m_mark_angle_spin->GetValue(), m_mark_visible_box->GetValue()) );
} // snapshot_frame::placement_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The alignment has changed.
 */
void bf::snapshot_frame::alignment_change()
{
  m_model_frame->do_action
    ( new action_modify_snapshot_alignment
      (m_snapshot,
       wx_to_std_string(m_x_alignment_choice->GetStringSelection()),
       m_x_alignment_value_spin->GetValue(),
       wx_to_std_string(m_y_alignment_choice->GetStringSelection()),
       m_y_alignment_value_spin->GetValue() ) );
} // snapshot_frame::alignment_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The size has changed.
 */
void bf::snapshot_frame::size_change()
{
  m_model_frame->do_action
    ( new action_modify_snapshot_size
      (m_snapshot, m_width_spin->GetValue(), m_height_spin->GetValue() ) );
} // snapshot_frame::size_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The sound has changed.
 */
void  bf::snapshot_frame::sound_change()
{
  sound_description sound;

  sound.set_file( wx_to_std_string( m_sound_path->GetValue() ) );
  sound.set_played_globally(m_globally_sound_box->IsChecked());
  m_model_frame->do_action
    ( new action_modify_snapshot_sound(m_snapshot, sound) );
} // snapshot_frame::sound_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief Enable or disable the controls.
 * \param enable Tell if the controls have to be enabled.
 */
void  bf::snapshot_frame::enable_controls(bool enable)
{
  m_date_button->Enable(enable);
  m_mark_choice->Enable(enable);
  m_function_button->Enable(enable);
  m_mark_visible_box->Enable(enable);
  m_sound_button->Enable(enable);
  m_sound_path->Enable(enable);
  m_globally_sound_box->Enable(enable);
  m_width_spin->Enable(enable);
  m_height_spin->Enable(enable);
  m_x_alignment_choice->Enable(enable);
  m_y_alignment_choice->Enable(enable);
  m_x_alignment_value_spin->Enable(enable);
  m_y_alignment_value_spin->Enable(enable);
} // snapshot_frame::enable_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::snapshot_frame::on_close(wxCloseEvent& event)
{
  if ( event.CanVeto() )
    {
      Hide();
      event.Veto();
    }
} // snapshot_frame::on_close()

/*----------------------------------------------------------------------------*/
/**
 * \brief The date button is pressed.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_date_change( wxCommandEvent& WXUNUSED(event) )
{
  action& act = m_model_frame->get_model().get_active_action();
  snapshot_date_frame dlg(this, &act);

  dlg.fill_from( m_snapshot );

  if ( dlg.ShowModal() == wxID_OK )
    {
      double new_date = dlg.get_date();
      double initial_date = m_snapshot->get_date();

      if ( new_date != initial_date )
        {
          action_group* group = new action_group();
          group->set_description(_("Move snapshot"));

          if ( act.has_snapshot_date( new_date ) )
            group->add_action
              ( new action_remove_snapshot
                ( act.get_name(),
                  act.get_snapshot_before_or_at_date(new_date)));

          group->add_action
            ( new action_move_snapshot(m_snapshot, new_date) );

          if ( initial_date == 0 )
            {
              snapshot* s =
                new snapshot
                ( *act.get_snapshot_before_or_at_date(initial_date));
              s->set_date(0);
              group->add_action
                ( new action_add_snapshot( act.get_name(), s));
            }

          m_model_frame->do_action(group);
          m_model_frame->set_date(new_date);
          fill();
        }
    }
} // snapshot_frame::on_date_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The selected mark has been modified.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_mark_change( wxCommandEvent& WXUNUSED(event) )
{
  m_model_frame->set_mark_selection
    (wx_to_std_string(m_mark_choice->GetStringSelection()));
} // snapshot_frame::on_mark_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The alignment has been modified.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_alignment_choice_change
( wxCommandEvent& WXUNUSED(event) )
{
  alignment_change();
} // snapshot_frame::on_alignment_choice_change()


/*----------------------------------------------------------------------------*/
/**
 * \brief The placement has been changed.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_placement_change
( spin_event<double>& WXUNUSED(event) )
{
  placement_change();
} // snapshot_frame::on_placement_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The alignment has been changed.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_alignment_change
( spin_event<double>& WXUNUSED(event) )
{
  alignment_change();
} // snapshot_frame::on_alignment_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The size has been changed.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_size_change( spin_event<double>& WXUNUSED(event) )
{
  size_change();
} // snapshot_frame::on_size_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The depth control has been changed.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_depth_change( wxSpinEvent& WXUNUSED(event) )
{
  placement_change();
} // snapshot_frame::on_depth_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The visibility statut of the mark has been modified.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_visibility_change( wxCommandEvent& WXUNUSED(event) )
{
  placement_change();
} // snapshot_frame::on_visibility_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The function button is pressed.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_function_change( wxCommandEvent& WXUNUSED(event) )
{
  wxTextEntryDialog dlg
    (this, _("Function:"), _("Set function"),
     std_to_wx_string(m_snapshot->get_function()) );

  if ( dlg.ShowModal() == wxID_OK )
    {
      m_model_frame->do_action
              ( new action_modify_function
          (m_snapshot, wx_to_std_string(dlg.GetValue())) );

      fill();
    }
} // snapshot_frame::on_function_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The globally_sound button is pressed.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_globally_sound_change
( wxCommandEvent& WXUNUSED(event) )
{
  sound_change();
} // snapshot_frame::on_globally_sound_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The sound_file button is pressed.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_sound_file_edit( wxCommandEvent& WXUNUSED(event) )
{
  std::string p = wx_to_std_string(m_sound_path->GetValue());
  path_configuration::get_instance().get_full_path(p);

  wxFileDialog dlg(this,_("Choose a sound"), wxEmptyString,
                   std_to_wx_string(p),
                   _("Sound files (*.ogg;*.wav)|*.wav;*.ogg"),
                   wxFD_DEFAULT_STYLE | wxFD_FILE_MUST_EXIST );

  if ( dlg.ShowModal() == wxID_OK )
    {
      std::string new_p = wx_to_std_string( dlg.GetPath() );
      path_configuration::get_instance().get_relative_path(new_p);

      m_sound_path->SetValue(std_to_wx_string(new_p) ) ;

      sound_change();
    }
} // snapshot_frame::on_sound_file_edit()

/*----------------------------------------------------------------------------*/
/**
 * \brief The text sound_file is edited.
 * \param event The event that occured.
 */
void bf::snapshot_frame::on_sound_text_edit( wxCommandEvent& WXUNUSED(event) )
{
  sound_change();
} // snapshot_frame::on_sound_text_edit()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::snapshot_frame, wxPanel)
  EVT_CLOSE( bf::snapshot_frame::on_close )
  EVT_BUTTON( bf::snapshot_frame::IDC_DATE_CTRL,
              bf::snapshot_frame::on_date_change )
  EVT_CHOICE(bf::snapshot_frame::IDC_MARK_CHOICE,
             bf::snapshot_frame::on_mark_change)
  EVT_CHOICE(bf::snapshot_frame::IDC_ALIGNMENT,
             bf::snapshot_frame::on_alignment_choice_change)
  EVT_SPIN_VALUE_CHANGE(bf::snapshot_frame::IDC_MARK_PLACEMENT,
                        bf::snapshot_frame::on_placement_change,
                        double)
  EVT_SPIN_VALUE_CHANGE(bf::snapshot_frame::IDC_ALIGNMENT,
                        bf::snapshot_frame::on_alignment_change,
                        double)
  EVT_SPIN_VALUE_CHANGE(bf::snapshot_frame::IDC_SIZE,
                        bf::snapshot_frame::on_size_change,
                        double)
  EVT_SPINCTRL( bf::snapshot_frame::IDC_MARK_PLACEMENT,
                bf::snapshot_frame::on_depth_change )
  EVT_CHECKBOX(bf::snapshot_frame::IDC_VISIBILITY,
               bf::snapshot_frame::on_visibility_change )
  EVT_BUTTON( bf::snapshot_frame::IDC_FUNCTION,
              bf::snapshot_frame::on_function_change )
  EVT_CHECKBOX( bf::snapshot_frame::IDC_GLOBALLY_SOUND,
              bf::snapshot_frame::on_globally_sound_change )
  EVT_BUTTON( bf::snapshot_frame::IDC_SOUND_FILE,
              bf::snapshot_frame::on_sound_file_edit )
  EVT_TEXT( bf::snapshot_frame::IDC_SOUND_TEXT,
            bf::snapshot_frame::on_sound_text_edit )
END_EVENT_TABLE()
