/*
    Bear Engine - Model editor

    Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/mark_selection.hpp
 * \brief A selection of marks.
 * \author Julien Jorge
 */
#ifndef __BF_MARK_SELECTION_HPP__
#define __BF_MARK_SELECTION_HPP__

#include <set>

namespace bf
{
  class mark;

  /**
   * \brief A selection of marks.
   * \author Julien Jorge
   */
  class mark_selection
  {
  public:
    /** \brief Iterator on the selected marks. */
    typedef std::set<mark*>::const_iterator const_iterator;

  public:
    /** \brief Constructor. */
    mark_selection();

    /**
     * \brief Insert an mark in the selection.
     * \param mark The mark to insert.
     * \param main_selection Tell if the main selection is set on the mark.
     */
    void insert( mark* mark, bool main_selection = false );

    /**
     * \brief Insert a selection in the selection.
     * \param s The marks to insert.
     */
    void insert( const mark_selection& s );

    /**
     * \brief Remove an mark from the selection.
     * \param mark The mark to insert.
     */
    void remove( mark* mark );

    /**
     * \brief Remove a selection from the selection.
     * \param s The marks to remove.
     */
    void remove( const mark_selection& s );

    /**
     * \brief Tell if an mark is selected.
     * \param mark The mark to check.
     */
    bool is_selected( mark* mark ) const;

    /**
     * \brief Tell if an mark is the main selection.
     * \param mark The mark to check.
     */
    bool is_main_selection( mark* mark ) const;

    /** \brief Make the selection empty. */
    void clear();

    /** \brief Tell if the selection is empty. */
    bool empty() const;

    /** \brief Get the main selection. */
    mark* get_main_selection() const;

    /** \brief Get an iterator on the first mark in the selection. */
    const_iterator begin() const;

    /** \brief Get an iterator after the last mark in the selection. */
    const_iterator end() const;

  private:
    /** \brief The main selection. */
    mark* m_mark;

    /** \brief All the selected marks. */
    std::set<mark*> m_group;

  }; // class mark_selection
} // namespace bf

#endif // __BF_MARK_SELECTION_HPP__
