/*
  Plee the Bear

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file code/level_variables.hpp
 * \brief Implementation of the ptb::level_variables class.
 * \author Julien Jorge
 */
#include "ptb/level_variables.hpp"

#include "engine/variable/variable.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief A function that gets a level variable.
 * \param lvl The level.
 * \param n The name of the variable.
 * \param def The default value to return if the variable is not set.
 */
template<typename T>
static T ptb_level_variables_get_value
( const bear::engine::level& lvl, const std::string& n, const T& def )
{
  bear::engine::variable<T> var(n);

  if ( lvl.level_variable_exists(var) )
    {
      lvl.get_level_variable(var);
      return var.get_value();
    }
  else
    return def;
} // ptb_level_variables_get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of players on an exit.
 * \param lvl The level in which we take the variable.
 */
unsigned int
ptb::level_variables::get_players_on_exit( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "players_on_exit", (unsigned int)0 );
} // level_variables::get_players_on_exit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of players on an exit.
 * \param lvl The level in which we take the variable.
 * \param c The number of players.
 */
void ptb::level_variables::set_players_on_exit
( bear::engine::level& lvl, unsigned int c )
{
  lvl.set_level_variable
    ( bear::engine::variable<unsigned int>( "players_on_exit", c ) );
} // level_variables::set_players_on_exit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of killed enemies.
 * \param lvl The level in which we take the variable.
 */
unsigned int
ptb::level_variables::get_killed_enemies( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "killed_ennemies", (unsigned int)0 );
} // level_variables::get_killed_enemies()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of killed enemies.
 * \param lvl The level in which we take the variable.
 * \param c The number of killed enemies.
 */
void ptb::level_variables::set_killed_enemies
( bear::engine::level& lvl, unsigned int nb )
{
  lvl.set_level_variable
    ( bear::engine::variable<unsigned int>( "killed_enemies", nb ) );
} // level_variables::set_killed_enemies()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of enemies in the level.
 * \param lvl The level in which we take the variable.
 */
unsigned int
ptb::level_variables::get_enemy_count( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "enemy_count", (unsigned int)0 );
} // level_variables::get_enemy_count()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of enemies in the level.
 * \param lvl The level in which we take the variable.
 * \param c The number of killed enemies.
 */
void ptb::level_variables::set_enemy_count
( bear::engine::level& lvl, unsigned int nb )
{
  lvl.set_level_variable
    ( bear::engine::variable<unsigned int>( "enemy_count", nb ) );
} // level_variables::set_enemy_count()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the hazelnut has been found.
 * \param lvl The level in which we take the variable.
 */
bool ptb::level_variables::get_hazelnut( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value( lvl, "hazelnut", false );
} // level_variables::get_hazelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set if the hazelnut has been found.
 * \param lvl The level in which we take the variable.
 * \param v Found or not.
 */
void ptb::level_variables::set_hazelnut( bear::engine::level& lvl, bool v )
{
  lvl.set_level_variable
    ( bear::engine::variable<bool>( "hazelnut", v ) );
} // level_variables::set_hazelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of honeypot found.
 * \param lvl The level in which we take the variable.
 */
unsigned int
ptb::level_variables::get_honeypots_found( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "honeypots_found", (unsigned int)0 );
} // level_variables::get_honeypots_found()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of honeypots found.
 * \param lvl The level in which we take the variable.
 * \param c The number of found honeypots.
 */
void ptb::level_variables::set_honeypots_found
( bear::engine::level& lvl, unsigned int nb )
{
  lvl.set_level_variable
    ( bear::engine::variable<unsigned int>( "honeypots_found", nb ) );
} // level_variables::set_honeypots_found()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if we are leaving the level.
 * \param lvl The level in which we take the variable.
 */
bool ptb::level_variables::is_exiting( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value( lvl, "exiting", false );
} // level_variables::is_exiting()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set that we are leaving the level.
 * \param lvl The level in which we take the variable.
 */
void ptb::level_variables::set_exiting( bear::engine::level& lvl )
{
  lvl.set_level_variable
    ( bear::engine::variable<bool>( "exiting", true ) );
} // level_variables::set_exiting()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the owl was met.
 * \param lvl The level in which we take the variable.
 */
bool ptb::level_variables::owl_is_met( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value( lvl, "met_the_owl", false );
} // level_variables::owl_is_met()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set that the owl is met.
 * \param lvl The level in which we take the variable.
 */
void ptb::level_variables::meet_the_owl( bear::engine::level& lvl )
{
  lvl.set_level_variable
    ( bear::engine::variable<bool>( "met_the_owl", true ) );
} // level_variables::meet_the_owl()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the time record for the level.
 * \param lvl The level in which we take the variable.
 */
bear::universe::time_type
ptb::level_variables::get_time_record( const bear::engine::level& lvl )
{
  return ptb_level_variables_get_value
    ( lvl, "time_record", (bear::universe::time_type)0 );
} // level_variables::get_time_record()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the time record for the level.
 * \param lvl The level in which we take the variable.
 * \param t The record.
 */
void ptb::level_variables::set_time_record
( bear::engine::level& lvl, bear::universe::time_type t )
{
  lvl.set_level_variable
    ( bear::engine::variable<bear::universe::time_type>( "time_record", t ) );
} // level_variables::set_time_record()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the players can harm each other.
 * \param lvl The level in which we take the variable.
 */
bool ptb::level_variables::get_friendly_fire(bear::engine::level& lvl)
{
  return ptb_level_variables_get_value( lvl, "friendly_fire", false );
} // game_variables::get_friendly_fire()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the players can harm each other.
 * \param lvl The level in which we take the variable.
 * \param v Harm or don't.
 */
void ptb::level_variables::set_friendly_fire
( bear::engine::level& lvl, bool v )
{
  lvl.set_level_variable
    ( bear::engine::variable<bool>( "friendly_fire", v ) );
} // game_variables::set_friendly_fire()
