/*
  Plee the Bear

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file monster.cpp
 * \brief Implementation of the ptb::monster class.
 * \author Sebastien Angibaud
 */
#include "ptb/monster.hpp"
#include "ptb/game_variables.hpp"
#include "ptb/level_variables.hpp"
#include "engine/base_item.hpp"
#include "engine/variable/variable.hpp"

#include <iostream>
#include <sstream>
#include <string>

/*----------------------------------------------------------------------------*/
const unsigned int ptb::monster::s_attack_count = 4;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::monster::monster()
  : m_offensive_force(0), m_energy(0), m_initial_energy(0),
    m_offensive_phase(true),
    m_defensive_powers(s_attack_count+1, false),
    m_offensive_coefficients(s_attack_count+1, 0), m_invincible(false),
    m_is_injured(false)
{
  m_offensive_coefficients[normal_attack] = 1;
} // base_monster::base_monster()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::monster::~monster()
{

} // base_monster::~base_monster()

/*----------------------------------------------------------------------------*/
/**
 * \brief The item receive an attack.
 *
 * \param attacker The monster attacking me.
 * \param info Some information about the collision.
 */
bool ptb::monster::receive_an_attack
( monster& attacker, const bear::universe::collision_info& info)
{
  bool result = false;

  if( (!m_invincible) && (!m_is_injured) )
    {
      if ( attacker.is_invincible() &&
           ( attacker.get_monster_type() != stone_1_monster ) &&
           ( attacker.get_monster_type() != stone_2_monster ) &&
           ( attacker.get_monster_type() != stone_3_monster ) )
        {
          if ( ( m_energy > 0 ) && is_vulnerable( attacker,info ) )
            {
              result = true;
              injure(attacker,info);
              remove_energy(attacker,m_energy);
              attacker.has_attacked(*this);
            }
        }
      else if( is_vulnerable( attacker,info )  )
        {
          double energy = 0;

          for( unsigned int i=0; i!=m_defensive_powers.size(); ++i )
            if ( !get_defensive_power(i,attacker,info) )
              energy += attacker.get_offensive_coefficient(i,*this,info);

          energy *= attacker.m_offensive_force;

          if ( energy > 0 )
            {
              result = true;
              injure(attacker,info);
              remove_energy(attacker,energy);
              attacker.has_attacked(*this);
            }
        }
    }

  return result;
} // monster::receive_an_attack()

/*----------------------------------------------------------------------------*/
/**
 * \brief The item has attacked.
 * \param other The monster that is attacked.
 */
void ptb::monster::has_attacked(const monster& other)
{

} // monster::has_attacked()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell is the item has a defensive power.
 */
bool ptb::monster::get_defensive_power
( unsigned int index,
  const monster& attacker,
  const bear::universe::collision_info& info ) const
{
  return m_defensive_powers[index];
} // monster::get_defensive_power()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a given offensive coefficient.
 */
unsigned int ptb::monster::get_offensive_coefficient
( unsigned int index,
  const monster& other,
  const bear::universe::collision_info& info ) const
{
  return m_offensive_coefficients[index];
} // monster::get_offensive_coefficient()

/*----------------------------------------------------------------------------*/
/**
 * \brief We remove some energy of the item..
 *
 * \param attacker The attacker monster.
 * \param energy The quantity of energy removed.
 */
void ptb::monster::remove_energy(const monster& attacker, double energy )
{
  if ( energy >= m_energy )
    m_energy = 0;
  else
    m_energy -= energy;
} // ptb::monster::remove_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief The monster is injure.
 * \param attacker The monster attacking me.
 */
void ptb::monster::injure
(const monster& attacker,const bear::universe::collision_info& info)
{
  m_is_injured = true;
  m_injured_time = 0;
} // ptb::monster::injure()

/*----------------------------------------------------------------------------*/
/**
 * \brief The monster isn't injure any more.
 */
void ptb::monster::finish_injure()
{
  m_is_injured = false;
} // ptb::monster::finish_injure()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
void ptb::monster::to_string( std::string& str ) const
{
  std::ostringstream oss;

  oss << "\ntype: ";
  if ( m_monster_type == player_1_monster )
    oss << "player_1";
  else if ( m_monster_type == player_2_monster )
    oss << "player_2";
  else if ( m_monster_type == player_3_monster )
    oss << "player_3";
  else if ( m_monster_type == enemy_monster )
    oss << "enemy";
  else if ( m_monster_type == stone_1_monster )
    oss << "stone_1";
  else if ( m_monster_type == stone_2_monster )
    oss << "stone_2";
  else if ( m_monster_type == nature_monster )
    oss << "nature";

  oss << "\noffensive_force: " << m_offensive_force;
  oss << "\nenergy: " << m_energy;
  oss << "\noffensive_phase: ";
  if ( m_offensive_phase )
    oss << "true";
  else
    oss << "false";

  oss << "\ndefensive: ";
  std::vector<bool>::const_iterator it;
  for(it = m_defensive_powers.begin(); it != m_defensive_powers.end(); ++it )
    oss << *it << " ";

  oss << "\noffensive_coef: ";
  std::vector<unsigned int>::const_iterator it2;
  for(it2 = m_offensive_coefficients.begin();
      it2 != m_offensive_coefficients.end(); ++it2 )
    oss << *it2 << " ";

  oss << "\ninvincible: ";
  if ( m_invincible )
    oss << "true";
  else
    oss << "false";

  oss << "\ninjured: ";
  if ( m_is_injured )
    oss << "true";
  else
    oss << "false";

  oss << "\n";

  str += oss.str();
} // monster::to_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the current item is vulnerable to an other item.
 *
 * \param attacker Type of the attacker.
 * \param info Some informations about the collision.
 */
bool ptb::monster::is_vulnerable
( monster& attacker, const bear::universe::collision_info& info ) const
{
  bool result = false;

  bool friendly(game_variables::get_friendly_fire());
  bear::engine::variable<bool> var("friendly_fire");
  bear::engine::level& lvl =
    dynamic_cast<const bear::engine::base_item*>(this)->get_level();

  if ( lvl.level_variable_exists(var) )
    friendly = level_variables::get_friendly_fire(lvl);

  switch( m_monster_type )
    {
    case nature_monster:
      result = false;
      break;

    case player_1_monster:
      if ( ( attacker.get_monster_type() == stone_2_monster ) ||
           ( attacker.get_monster_type() == player_2_monster ) )
        result = friendly;
      else
        result = ( attacker.get_monster_type() != stone_1_monster );
      break;

    case player_2_monster:
      if ( ( attacker.get_monster_type() == stone_1_monster ) ||
           ( attacker.get_monster_type() == player_1_monster ) )
        result = friendly;
      else
        result = ( attacker.get_monster_type() != stone_2_monster );
      break;

    case player_3_monster:
      result = ( attacker.get_monster_type() != stone_3_monster );
      break;

    case enemy_monster:
      result = ( attacker.get_monster_type() != enemy_monster );
      break;

    case stone_1_monster:
       if ( ( attacker.get_monster_type() == stone_2_monster ) ||
           ( attacker.get_monster_type() == player_2_monster ) )
        result = friendly;
      else
        result = ( ( attacker.get_monster_type() != player_1_monster ) &&
                   ( attacker.get_monster_type() != stone_1_monster ) );
      break;

    case stone_2_monster:
       if ( ( attacker.get_monster_type() == stone_1_monster ) ||
           ( attacker.get_monster_type() == player_1_monster ) )
        result = friendly;
       else
         result = ( ( attacker.get_monster_type() != player_2_monster ) &&
                 ( attacker.get_monster_type() != stone_2_monster ) );
      break;

    case stone_3_monster:
       if ( ( attacker.get_monster_type() == stone_3_monster ) ||
           ( attacker.get_monster_type() == player_3_monster ) )
        result = friendly;
       else
         result = ( ( attacker.get_monster_type() != player_3_monster ) &&
                 ( attacker.get_monster_type() != stone_3_monster ) );
      break;

    default:
      break;

    }


  return result;
} // monster::is_vulnerable()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the invincible status.
 * \param invincible The new invincible status.
 */
void ptb::monster::set_invincible(const bool invincible)
{
  m_invincible = invincible;
} // ptb::monster::set_invincible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the monster is invincible.
 */
bool ptb::monster::is_invincible() const
{
  return m_invincible;
} // ptb::monster::is_invincible()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the offensive phase status.
 * \param offensive_phase The new offensive phase status.
 */
void ptb::monster::set_offensive_phase(const bool offensive_phase)
{
  m_offensive_phase = offensive_phase;
} // ptb::monster::set_offensive_phase()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the offensive phase status.
 */
bool ptb::monster::get_offensive_phase() const
{
  return m_offensive_phase;
} // ptb::monster::get_offensive_phase()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the defensive power status.
 * \param index Type of the considered attack.
 * \param status The new defensive status.
 */
void
ptb::monster::set_defensive_power(const attack_type index, const bool status)
{
  m_defensive_powers[index] = status;
} // ptb::monster::set_defensive_power()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a given defensive power status.
 * \param index Type of the considered attack.
 */
bool
ptb::monster::get_defensive_power(const attack_type index) const
{
  return m_defensive_powers[index];
} // ptb::monster::get_defensive_power()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set an offensive coefficient.
 * \param index Type of the considered attack.
 * \param coef The new offensive coefficient.
 */
void
ptb::monster::set_offensive_coefficient
(const attack_type index, const unsigned int coef)
{
  m_offensive_coefficients[index] = coef;
} // ptb::monster::set_offensive_coefficient()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the energy of the monster.
 */
double ptb::monster::get_energy() const
{
  return m_energy;
} // monster::get_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the energy of the monster.
 * \param energy The new energy.
 */
void ptb::monster::set_energy( double energy )
{
  m_energy = energy;
} // monster::set_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the initial energy of the monster.
 */
double ptb::monster::get_initial_energy() const
{
  return m_initial_energy;
} // monster::get_initial_energy()


/*----------------------------------------------------------------------------*/
/**
 * \brief Return if te monster is injured.
 */
bool ptb::monster::is_injured() const
{
  return m_is_injured;
} // monster::is_injured()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if the monster is in a offensive phase.
 */
bool ptb::monster::is_in_offensive_phase() const
{
  return m_offensive_phase;
} // monster::is_in_offensive_phase()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the type of monster.
 */
ptb::monster::monster_type ptb::monster::get_monster_type() const
{
  return m_monster_type;
} // monster::get_monster_type()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the type of monster.
 */
void ptb::monster::set_monster_type(ptb::monster::monster_type m)
{
  m_monster_type = m;
} // monster::set_monster_type()
