/*
  Plee the Bear

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file hazelnut.cpp
 * \brief Implementation of the ptb::hazelnut class.
 * \author Sebastien Angibaud
 */
#include "ptb/item/hazelnut.hpp"

#include "ptb/player.hpp"
#include "ptb/level_variables.hpp"

BASE_ITEM_EXPORT( hazelnut, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 */
ptb::hazelnut::hazelnut()
{
  set_can_move_items(false);
  set_elasticity(1);
  set_friction(0.98);
  set_mass(20);
  set_density(2);
} // hazelnut::hazelnut()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the media required by this class.
 */
void ptb::hazelnut::pre_cache()
{
  super::pre_cache();

  get_level_globals().load_animation("animation/owl/hazelnut.canim");
} // hazelnut::pre_cache()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void ptb::hazelnut::build()
{
  super::build();

  set_animation
    ( get_level_globals().get_animation("animation/owl/hazelnut.canim") );

  set_size( get_animation().get_size() );
} // hazelnut::build()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::hazelnut::progress( bear::universe::time_type elapsed_time )
{
  super::progress(elapsed_time);

  if ( has_bottom_contact() )
    add_internal_force( bear::universe::force_type(0, 200000) );
} // hazelnut::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the collision is with a player.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void ptb::hazelnut::collision_check_and_apply
( bear::engine::base_item& that, bear::universe::collision_info& info )
{
  player* p = dynamic_cast<player*>(&that);

  if ( p != NULL )
    {
      if ( !level_variables::get_hazelnut(get_level()) )
        {
          if ( info.get_collision_side() != bear::universe::zone::middle_zone)
            {
              level_variables::set_hazelnut(get_level(), true);
              kill();
            }
          else
            default_collision(info);
        }
      else
        default_collision(info);
    }
  else
    default_collision(info);
} // hazelnut::collision_check_and_apply()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call collision_check_and_apply().
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void ptb::hazelnut::collision
( bear::engine::base_item& that, bear::universe::collision_info& info )
{
  collision_check_and_apply(that, info);
} // hazelnut::collision()
