/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001  Kevin P. Lawton
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include "dt.h"

/* Area for storing handler values */
Bit32u r3h_DS;
Bit32u r3h_ESP;
Bit32u r3h_target_EIP;
Bit32u r3h_ESP_empty;
Bit32u globalID;

/* Area for storing guest values */
Bit32u guest_SS;
Bit32u guest_ESP;
Bit32u guest_EIP;

unsigned r3h_request;
Bit32u r3h_data;
Bit32u idPatchDispl;
Bit32u jmpPatchDispl;

unsigned char r3h_stack[R3H_STACK_SIZE];


/*
 *  %edx = $dtG2THash + ((l>>5) & 0x00001fff)*32;
 *  %edx = $dtG2THash + ((l>>5) & 0x00001fff)<<5;
 *  %edx = $dtG2THash + (l & 0x0003ffe0);
 *
 *  %edx = $dtG2THash + ((l) & 0x00001fff)*32;
 *  %edx = $dtG2THash + ((l) & 0x00001fff)<<5;
 */

/* +++ Perhaps merge r3h_branch static/dynamic handlers?  Would have
 * to pass a backpatch parameter if so.
 */

asm (
  ".text                       \n\t"
  ".globl __r3h_branch_dynamic \n\t"
  "__r3h_branch_dynamic:       \n\t"
  "  pushfl                    \n\t"   /* Save all guest state */
  "  pushl %ecx                \n\t"   /* which will be modified */
  "  pushl %edx                \n\t"
  "  movl  16(%esp), %ecx      \n\t"   /* Get target addr from stack */
  "  movl  %ecx, %edx          \n\t"
#if DT_G2THashMethod == 0
  "  andl  $0x0003ffe0, %edx   \n\t"
#else
  "  andl  $0x00001fff, %edx   \n\t"
  "  shll  $5, %edx            \n\t"
#endif
  "  addl  $dtG2THash, %edx    \n\t"
  "0:                          \n\t"
  "  cmpl %ecx, 0(%edx)        \n\t"
  "  jne 1f                    \n\t"
  "  movl 4(%edx), %ecx        \n\t"
  "done:                       \n\t"
  "  movl %ecx, r3h_target_EIP \n\t"
  "  popl   %edx               \n\t"
  "  popl   %ecx               \n\t"
  "  popfl                     \n\t"
#if DT_UseR3hStack
  "  cs; mov guest_ESP, %esp   \n\t"
#endif
  "  cs; jmp *(r3h_target_EIP) \n\t"
  "1:                          \n\t"
  "  cmpl %ecx, 8(%edx)        \n\t"
  "  jne 2f                    \n\t"
  "  movl 12(%edx), %ecx       \n\t"
  "  jmp  done                 \n\t"
  "2:                          \n\t"
  "  cmpl %ecx, 16(%edx)       \n\t"
  "  jne 3f                    \n\t"
  "  movl 20(%edx), %ecx       \n\t"
  "  jmp  done                 \n\t"
  "3:                          \n\t"
  "  cmpl %ecx, 24(%edx)       \n\t"
  "  jne 4f                    \n\t"
  "  movl 28(%edx), %ecx       \n\t"
  "  jmp  done                 \n\t"
  "4:                          \n\t"
  "  popl   %edx               \n\t"
  "  popl   %ecx               \n\t"
  "  popfl                     \n\t"
  /* Save guest context modified by the handler */
  "  pushl %ds                 \n\t"   /* Save guest DS/ES since handler needs to use these */
  "  pushl %es                 \n\t"   /* and sets them to its own segment. */
  "  pushal                    \n\t"   /* Save guest general registers. */
  "  pushfl                    \n\t"   /* Save guest eflags */
  "  mov  %ss, %ax             \n\t"   /* Copy handler SS to {DS,ES}. */
  "  mov  %ax, %ds             \n\t"
  "  mov  %ax, %es             \n\t"
  "  cld                       \n\t"
  "  call r3h_branch_dynamic   \n\t"   /* Call C handler */
  "  jmp  __r3h_prime          \n\t"   /* Wrap up */
  );


asm (
  ".text                       \n\t"
  ".globl __r3h_branch_static  \n\t"
  "__r3h_branch_static:        \n\t"
  /* Save guest context modified by the handler */
  "  pushl %ds                 \n\t"   /* Save guest DS/ES since handler needs to use these */
  "  pushl %es                 \n\t"   /* and sets them to its own segment. */
  "  pushal                    \n\t"   /* Save guest general registers. */
  "  pushfl                    \n\t"   /* Save guest eflags */
  "  mov  %ss, %ax             \n\t"   /* Copy handler SS to {DS,ES}. */
  "  mov  %ax, %ds             \n\t"
  "  mov  %ax, %es             \n\t"
  "  cld                       \n\t"
  "  call r3h_branch           \n\t"   /* Call C handler */
  ".globl __r3h_prime          \n\t"
  "__r3h_prime:                \n\t"
  /* Restore guest context modified by the handler */
  "  popfl                     \n\t"   /* Restore guest eflags */
  "  popal                     \n\t"   /* Restore guest general registers */
  "  popl %es                  \n\t"   /* Restore guest DS/ES segments which were modified */
  "  popl %ds                  \n\t"   /* for use by handler. */
  "  cs; mov guest_SS,  %ss    \n\t"
  "  cs; mov guest_ESP, %esp   \n\t"
  "  cs; jmp *(r3h_target_EIP) \n\t");


asm (
  ".text            \n\t"
  ".globl __r3h_ret \n\t"
  "__r3h_ret:       \n\t"
  /* Save guest context modified by the handler */
  "  pushl %ds      \n\t"   /* Save guest DS/ES since handler needs to use these */
  "  pushl %es      \n\t"   /* and sets them to its own segment. */
  "  pushal         \n\t"   /* Save guest general registers. */
  "  pushfl         \n\t"   /* Save guest eflags */
  "  mov  %ss, %ax  \n\t"   /* Copy handler SS to {DS,ES}. */
  "  mov  %ax, %ds  \n\t"
  "  mov  %ax, %es  \n\t"
  "  cld            \n\t"
  "  call r3h_ret   \n\t"   /* Call C handler */
  /* Restore guest context modified by the handler */
  "  popfl          \n\t"   /* Restore guest eflags */
  "  popal          \n\t"   /* Restore guest general registers */
  "  popl %es       \n\t"   /* Restore guest DS/ES segments modified */
  "  popl %ds       \n\t"   /* for use by handler. */
  "  cs; mov guest_SS,  %ss    \n\t"
  "  cs; mov guest_ESP, %esp   \n\t"
  "  cs; jmp *(r3h_target_EIP) \n\t"
  );


  Bit32u
r3hToMonRequest(unsigned req, Bit32u data)
{
  switch (req) {
    case R3HToMonRequestG2T:
      r3h_request = req;
      r3h_data = data;
      __r3h2mon();
      return r3h_data;

    case R3HToMonRequestTerminate:
      r3h_request = R3HToMonRequestTerminate;
      r3h_data = 0;
      __r3h2mon();
      return 0;

    default:
      r3h_request = R3HToMonRequestPanic;
      r3h_data = data;
      __r3h2mon();
      return 0;
    }
}


/*
 *  This represents the C component of the dynamic branch convenience
 *  routine.  I just pretend the targets of the branches are already
 *  translated in the cache, and use a simple switch statement to
 *  get the translated code address.  Real table lookup logic would
 *  go here.  This should be coded extremely efficiently, as it will
 *  be executed a lot.
 *
 *  If we opened the code cache pages and related tables up for write
 *  permissions from ring3 (and thus the ring3 convenience handlers),
 *  conceivably we could code the translation here, kicking in upon
 *  cache misses.  Otherwise, we would defer to the monitor at ring0.
 */

  void
r3h_branch(gc_t gc, Bit32u tcodeRetOff, unsigned gOff)
{
  unsigned hashRow, i;

#if DT_UseBackpatch
  Bit32u *idPatchPtr, *jmpPatchPtr;
  Bit32u jmpDisplacement;
#endif

  /*
   *  Search the G2T table first, ideally the instruction will have
   *  been translated already, and the translation address is in
   *  there.
   */

/* +++ what about addr==0 */
  /* similar code as in dtTranslateG2T() */
  hashRow = DT_G2THashSelect(gOff);
  for (i = 0; i < DT_G2THashWidth; i++) {
    if (dtG2THash[hashRow][i].gOff == gOff) {
      r3h_target_EIP = dtG2THash[hashRow][i].tOff;
      InstrG2THit(i);
      goto done;
      }
    }

  InstrG2TMiss();

  /* Guest to Tcode address pair not in G2T hash table. */
  /* +++ Should look in L2M hash table first here */
  r3h_target_EIP = r3hToMonRequest(R3HToMonRequestG2T, gOff);

done:

#if DT_UseBackpatch
  /*
   *  We can now backpatch the current context switch ID and
   *  direct tcode branch into the branch instruction.  Using
   *  the code address saved from the CALL instruction, and
   *  some known displacements, we can easily calculate the
   *  addresses of instruction data to patch.
   */

  idPatchPtr = (Bit32u *) (tcodeRetOff - idPatchDispl);
  jmpPatchPtr = (Bit32u *) (tcodeRetOff - jmpPatchDispl);
  jmpDisplacement = r3h_target_EIP - (((Bit32u) jmpPatchPtr) + 4);
  *idPatchPtr = globalID;
  *jmpPatchPtr = jmpDisplacement;
#endif
}


  void
r3h_branch_dynamic(gc_t gc, Bit32u tcodeRetOff, unsigned gOff)
{
  unsigned hashRow, i;

  /*
   *  Search the G2T table first, ideally the instruction will have
   *  been translated already, and the translation address is in
   *  there.
   */

/* +++ what about addr==0 */
  /* similar code as in dtTranslateG2T() */
  hashRow = DT_G2THashSelect(gOff);
  for (i = 0; i < DT_G2THashWidth; i++) {
    if (dtG2THash[hashRow][i].gOff == gOff) {
      r3h_target_EIP = dtG2THash[hashRow][i].tOff;
      InstrG2THit(i);
      goto done;
      }
    }

  InstrG2TMiss();

  /* Guest to Tcode address pair not in G2T hash table. */
  /* +++ Should look in L2M hash table first here */
  r3h_target_EIP = r3hToMonRequest(R3HToMonRequestG2T, gOff);

done:
}


  void
r3h_ret(gc_t gc)
{
  r3hToMonRequest(R3HToMonRequestTerminate, 0);
}

