/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  system-nexus.c: The 'motherboard' logic which connects the entire
 *    PC system.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include "plex86.h"
#define IN_NEXUS_SPACE
#include "monitor.h"


static void monitorHeartbeat(vm_t *vm, unsigned i);

//#define USecToCycles(usec) (1 + ((usec)>>2))
#define USecToCycles(usec) ((usec)*400)




  void
initTimers(vm_t *vm)
{
  timerRegister_t timer;

  /* The vm->system is already memzero'd for us. */
  timer.thisPtr = 0;
  timer.callback = (void *) monitorHeartbeat;
  timer.useconds = 10 * 1000 * 1000; /* 10 seconds */
  timer.continuous = 1;
  timer.active = 1;
  registerTimer(vm, MonitorSpace, &timer);
}

  void
monitorHeartbeat(vm_t *vm, unsigned i)
{
  /* This is a placeholder timer callback, so that there is always one
   * active and continuous timer.  Could be used for some kind of
   * housekeeping.
   */
}

  int
registerTimer(vm_t *vm, unsigned space, timerRegister_t *timer)
{
  unsigned id;
  Bit64u cpuCycles;

  /* Sanity checks. */
  if (vm->system.num_timers >= MAX_TIMERS)
    return( -1 ); /* Too many timers. */

  if (timer->callback == (void *)0 )
    return( -1 ); /* Invalid callback address. */

  cpuCycles = USecToCycles(timer->useconds);
  id = vm->system.num_timers;
  vm->system.timer[id].period     = cpuCycles;
  vm->system.timer[id].remaining  = cpuCycles;
  vm->system.timer[id].inServiceElapsed  = vm->system.cyclesElapsed;
  vm->system.timer[id].active     = timer->active;
  vm->system.timer[id].continuous = timer->continuous;
  vm->system.timer[id].triggered  = 0;
  vm->system.timer[id].callback   = timer->callback;
  vm->system.timer[id].thisPtr    = timer->thisPtr;
  vm->system.timer[id].space      = space;

  if (timer->active) {
    if (vm->system.num_timers==0) {
      if (vm->system.cyclesElapsed != 0)
        return(-1); /* Should be no guest time elapsed yet. */
      /* This is the first timer registered. */
      vm->system.cyclesInPeriod  = cpuCycles;
      vm->system.cyclesRemaining = cpuCycles;
      }
    else {
      if ( (vm->system.cyclesElapsed + cpuCycles) <
           vm->system.cyclesRemaining ) {
        Bit64u newRemaining, shortenBy;
        /* If the cycles already elapsed plus the duration of this timer,
         * is less than the remaining cycles before the next expected
         * delivery of a time event, then adjust for a sooner event.
         */
        newRemaining = (vm->system.cyclesElapsed + cpuCycles);
        shortenBy    = (vm->system.cyclesRemaining - newRemaining);
        if (shortenBy >= vm->system.cyclesInPeriod)
          return( -1 ); /* Error. */
        /* Shorten this period by the difference of the old and new values. */
        vm->system.cyclesInPeriod  -= shortenBy;
        vm->system.cyclesRemaining  = newRemaining;
        }
      }
    }

  vm->system.num_timers++;

  /* Return timer id */
  return(id);
}

  int
activateTimer(vm_t *vm, unsigned space, timerRegister_t *timer)
{
  unsigned id;
  Bit64u cpuCycles;

  id = (unsigned) timer->callback;
  if (vm->system.num_timers==0)
    return( -1 ); /* Error, no timers defined. */
  if (id >= vm->system.num_timers)
    return( -1 ); /* Error, timer does not exist. */
  if (space != vm->system.timer[id].space)
    return( -1 ); /* Error, timer space does not match. */
  if (vm->system.timer[id].active)
    return( -1 ); /* Error, timer already active. */

  cpuCycles = USecToCycles(timer->useconds);
  vm->system.timer[id].active = 1;
  vm->system.timer[id].period     = cpuCycles;
  vm->system.timer[id].remaining  = cpuCycles;
  vm->system.timer[id].inServiceElapsed  = vm->system.cyclesElapsed;
  vm->system.timer[id].triggered = 0;

  if ( (vm->system.cyclesElapsed + cpuCycles) <
       vm->system.cyclesRemaining ) {
    Bit64u newRemaining, shortenBy;
    /* If the cycles already elapsed plus the duration of this timer,
     * is less than the remaining cycles before the next expected
     * delivery of a time event, then adjust for a sooner event.
     */
    newRemaining = (vm->system.cyclesElapsed + cpuCycles);
    shortenBy    = (vm->system.cyclesRemaining - newRemaining);
    if (shortenBy >= vm->system.cyclesInPeriod)
      return( -1 ); /* Error. */
    /* Shorten this period by the difference of the old and new values. */
    vm->system.cyclesInPeriod  -= shortenBy;
    vm->system.cyclesRemaining  = newRemaining;
    }
  return( 0 ); /* OK */
}

  int
deactivateTimer(vm_t *vm, unsigned space, int id)
{
  if (id >= vm->system.num_timers)
    return( 1 ); /* Error, timer does not exist. */
  if (space != vm->system.timer[id].space)
    return( 2 ); /* Error, timer space does not match. */
  if (vm->system.num_timers==0)
    return( 3 ); /* Error, no timers defined. */

  vm->system.timer[id].active = 0;
  vm->system.timer[id].triggered = 0;

  return( 0 ); /* OK */
}

  unsigned
ioInit(vm_t *vm)
{
  unsigned p;

  mon_memzero(&vm->ioHandlers, sizeof(vm->ioHandlers));
  for (p=0; p<64*1024+1; p++) {
    vm->ioHandlers.readHandlerID[p]  = NoIOHandlerID;
    vm->ioHandlers.writeHandlerID[p] = NoIOHandlerID;
    }
  initIODevices(vm);
  return(1); /* OK. */
}

  int
registerIORHandler(vm_t *vm, unsigned space, void *thisPtr, void *callback,
                   Bit32u base, unsigned len, char *name)
{
  unsigned h;

  /* Sanity checks. */
  if (base & 0xffff0000) {
    if (base ^ 0xffffffff) {
      return(-1); /* Error: port number too big. */
    }
    else {
      base = 64*1024; /* 0xFFFFFFFF = Default handler -> 64k index*/
    }
  }
  if (len > 4) {
    return(-2); /* Error: port size too big. */
    }
  if (!callback) {
    return(-3); /* Error: no callback defined. */
    }

  /* First, see if this IO handler is already registered for another
   * port and thus has a read handle ID.
   */
  for (h=0; h < vm->ioHandlers.readHandlerCount; h++) {
    if ( (vm->ioHandlers.read[h].space == space) &&
         (vm->ioHandlers.read[h].callback == callback) )
        break;
    }
  if (h >= vm->ioHandlers.readHandlerCount) {
    /* Handler not already registered, allocate a handle. */
    if (vm->ioHandlers.readHandlerCount >= MaxIOHandlers) {
      return(-4); /* Error: too many read handlers registered. */
      }
    vm->ioHandlers.readHandlerCount++;
    vm->ioHandlers.read[h].thisPtr  = thisPtr;
    vm->ioHandlers.read[h].callback = callback;
    vm->ioHandlers.read[h].space    = space;
    }
  if (vm->ioHandlers.readHandlerID[base] < MaxIOHandlers) {
    return(-5); /* Error: IO read handler already defined for this port. */
    }
  vm->ioHandlers.readHandlerID[base] = h;

  /* Return the read and write handler ids as a success code. */
  return( h );
}

  int
registerIOWHandler(vm_t *vm, unsigned space, void *thisPtr, void *callback,
                   Bit32u base, unsigned len, char *name)
{
  unsigned h;

  /* Sanity checks. */
  if (base & 0xffff0000) {
    if (base ^ 0xffffffff) {
      return(-1); /* Error: port number too big. */
    }
    else {
      base = 64*1024; /* 0xFFFFFFFF = Default handler -> 64k index*/
    }
  }
  if (len > 4) {
    return(-2); /* Error: port size too big. */
    }
  if (!callback) {
    return(-3); /* Error: no callback defined. */
    }

  /* First, see if this IO handler is already registered for another
   * port and thus has a write handle ID.
   */
  for (h=0; h < vm->ioHandlers.writeHandlerCount; h++) {
    if ( (vm->ioHandlers.write[h].space == space) &&
         (vm->ioHandlers.write[h].callback == callback) )
        break;
    }
  if (h >= vm->ioHandlers.writeHandlerCount) {
    /* Handler not already registered, allocate a handle. */
    if (vm->ioHandlers.writeHandlerCount >= MaxIOHandlers) {
      return(-4); /* Error: too many write handlers registered. */
      }
    vm->ioHandlers.writeHandlerCount++;
    vm->ioHandlers.write[h].thisPtr  = thisPtr;
    vm->ioHandlers.write[h].callback = callback;
    vm->ioHandlers.write[h].space    = space;
    }
  if (vm->ioHandlers.writeHandlerID[base] < MaxIOHandlers) {
    return(-5); /* Error: IO write handler already defined for this port. */
    }
  vm->ioHandlers.writeHandlerID[base] = h;


  /* Return the read and write handler ids as a success code. */
  return( h );
}

  int
registerIRQ(vm_t *vm, unsigned space, unsigned irq, const char *name)
{
  return(-1);
}

  int
unregisterIRQ(vm_t *vm, unsigned space, unsigned irq, const char *name)
{
  return(-1);
}

  void
setINTR(vm_t *vm, unsigned val)
{
  vm->system.intr = val;
  if (val)
    vm->guest_cpu.async_event = 1;
}
