/************************************************************************
 * $Id: bxemu.h,v 1.3 2002/08/12 18:43:18 DemonLord Exp $
 ************************************************************************
 *
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The plex86 developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifdef __cplusplus
extern "C" {
#endif

#ifndef _BXEMU_H
#define _BXEMU_H

#include <stdio.h>
#include <stddef.h>

#include <stdarg.h>
#ifndef PLEX86
#include "config.h"
#endif

#define BX_ENABLE_DEBUG 0
#define BX_ENABLE_INFO  1

/* === Mega hacks for bochs IO devices === */

typedef struct {
  void   (* set)(Bit32u);
  Bit32u (* get)(void);
  } Bit32uParam;

typedef struct {
  void   (* set)(Bit32s);
  Bit32s (* get)(void);
  } Bit32sParam;

typedef struct {
  void  (* setptr)(char *);
  char* (* getptr)(void);
  } StringParam;

typedef struct {
  void  (* setptr)(char *);
  char* (* getptr)(void);
  } pCharParam;

typedef struct {
  void    (* set)(Boolean);
  Boolean (* get)(void);
  } BooleanParam;


typedef struct {
  char path[512];
  unsigned type;
  unsigned initial_status;
  unsigned devtype;

  StringParam *Opath;
  Bit32uParam *Otype;
  Bit32uParam *Ostatus;
  Bit32uParam *Odevtype;
  } bx_floppy_options;

typedef struct {
  Boolean present;
  char path[512];
  unsigned int cylinders;
  unsigned int heads;
  unsigned int spt;

  BooleanParam *Opresent;
  StringParam  *Opath;
  Bit32uParam  *Ocylinders;
  Bit32uParam  *Oheads;
  Bit32uParam  *Ospt;
  } bx_disk_options;
 
typedef struct
{
  Boolean present;
  char dev[512];
  Boolean inserted;

  BooleanParam *Opresent;
  StringParam  *Opath;
  BooleanParam *Ostatus;
} bx_cdrom_options;
 
typedef struct {
  char *path;
  unsigned long address;

  pCharParam   *Opath;
  Bit32uParam  *Oaddress;
  } bx_rom_options;
 
typedef struct {
  char *path;

  pCharParam  *Opath;
  } bx_vgarom_options;
 
typedef struct {
  size_t megs;

  Bit32uParam *Oaddress;
  } bx_mem_options;
 
typedef struct {
  char      *path;
  Boolean   cmosImage;
  Bit32u    time0;

  pCharParam    *Opath;
  BooleanParam  *OcmosImage;
  Bit32uParam   *Otime0;
  } bx_cmos_options;
 
typedef struct {
  int       valid;
  unsigned  ioaddr;
  unsigned  irq;
  unsigned char macaddr[6];
  char      *ethmod;
  char      *ethdev;

  Bit32sParam   *Ovalid;
  Bit32uParam   *Oioaddr;
  Bit32uParam   *Oirq;
  StringParam   *Omacaddr;
  pCharParam    *Oethmod;
  pCharParam    *Oethdev;
  } bx_ne2k_options;

typedef struct {
  char *midifile, *wavefile, *logfile;
  unsigned int midimode, wavemode, loglevel;
  Bit32u dmatimer;

  pCharParam *Omidifile, *Owavefile, *Ologfile;
  Bit32uParam *Omidimode, *Owavemode, *Ologlevel;
  Bit32uParam *Odmatimer;
  } bx_sb16_options;

typedef struct {
  Boolean enable;
  char outfile[512];

  BooleanParam *Oenable;
  StringParam  *Ooutfile;
  } bx_parport_options;

typedef struct {
  Boolean present;
  char dev[512];

  BooleanParam *Opresent;
  StringParam  *Odev;
  } bx_serial_options;

typedef struct {
  Boolean useMapping;
  char    *keymap;

  BooleanParam *OuseMapping;
  pCharParam   *Okeymap;
  } bx_keyboard_options;

typedef struct {
  char    shortcut[40];
  } bx_user_shortcut_options;

typedef struct {
  bx_floppy_options floppya;
  bx_floppy_options floppyb;
  bx_disk_options   diskc;
  bx_disk_options   diskd;
  bx_cdrom_options  cdromd;
  bx_sb16_options   sb16;
  bx_cmos_options   cmos;
  bx_ne2k_options   ne2k;
  bx_parport_options par1;
  bx_serial_options com1;
  bx_keyboard_options keyboard;
  bx_user_shortcut_options usershortcut;

  char              bootdrive[6];
  unsigned long     vga_update_interval;
  unsigned long     keyboard_serial_delay;
  unsigned long     keyboard_paste_delay;
  unsigned long     keyboard_type;
  unsigned long     floppy_command_delay;
  unsigned long     ips;
  Boolean           mouse_enabled;
  Boolean           private_colormap;
  Boolean           i440FXSupport;
  Boolean           newHardDriveSupport;

  Bit32uParam       *Obootdrive;
  BooleanParam      *OfloppySigCheck;
  Bit32uParam       *Ovga_update_interval;
  Bit32uParam       *Okeyboard_serial_delay;
  Bit32uParam       *Okeyboard_paste_delay;
  Bit32uParam       *Okeyboard_type;
  Bit32uParam       *Ofloppy_command_delay;
  Bit32uParam       *Oips;
  BooleanParam      *Omouse_enabled;
  BooleanParam      *Oprivate_colormap;
  BooleanParam      *Oi440FXSupport;
  BooleanParam      *OnewHardDriveSupport;
  StringParam       *Ouser_shortcut;
  } bx_options_t;

extern bx_options_t bx_options;

// defines option constants
#define BX_BOOT_FLOPPYA 1
#define BX_BOOT_DISKC   2
#define BX_BOOT_CDROM   3

#define BX_KBD_XT_TYPE        0
#define BX_KBD_AT_TYPE        1
#define BX_KBD_MF_TYPE        2

// These has been changed to 1 and 0 because plex sets the option like that
#define BX_EJECTED   0
#define BX_INSERTED  1
 
void bxemu_init(void);

#ifdef __cplusplus
};
#endif

#ifdef __cplusplus

#define EMPTY do { } while(0)

#if BX_ENABLE_INFO==1
#define BX_INFO(x)  (LOG_THIS info) x
#else
#define BX_INFO(x)  EMPTY
#endif

#if BX_ENABLE_DEBUG==1
#define BX_DEBUG(x) (LOG_THIS ldebug) x
#else
#define BX_DEBUG(x)  EMPTY
#endif

#define BX_ERROR(x) (LOG_THIS error) x
#define BX_PANIC(x) (LOG_THIS panic) x
#define BX_ASSERT(x) do {if (!(x)) BX_PANIC(("failed assertion \"%s\" at %s:%d\n", #x, __FILE__, __LINE__));} while (0)

// Log Class defines
#define    IOLOG           0
#define    FDLOG           1
#define    GENLOG          2
#define    CMOSLOG         3
#define    CDLOG           4
#define    DMALOG          5
#define    ETHLOG          6
#define    G2HLOG          7
#define    HDLOG           8
#define    KBDLOG          9
#define    NE2KLOG         10
#define    PARLOG          11
#define    PCILOG          12
#define    PICLOG          13
#define    PITLOG          14
#define    SB16LOG         15
#define    SERLOG          16
#define    VGALOG          17
#define    STLOG           18   // state_file.cc
#define    DEVLOG          19
#define    MEMLOG          20
#define    DISLOG          21
#define    GUILOG          22
#define    IOAPICLOG       23
#define    APICLOG         24
#define    CPU0LOG         25
#define    CPU1LOG         26
#define    CPU2LOG         27
#define    CPU3LOG         28
#define    CPU4LOG         29
#define    CPU5LOG         30
#define    CPU6LOG         31
#define    CPU7LOG         32
#define    CPU8LOG         33
#define    CPU9LOG         34
#define    CPU10LOG         35
#define    CPU11LOG         36
#define    CPU12LOG         37
#define    CPU13LOG         38
#define    CPU14LOG         39
#define    CPU15LOG         40
#define    CTRLLOG         41
#define    UNMAPLOG        42
#define    SERRLOG         43
#define    BIOSLOG         42
#define    PIT81LOG        45
#define    PIT82LOG        46
#define    IODEBUGLOG      47

class logfunctions {
	char *prefix;
public:
	logfunctions(void);
	logfunctions(class iofunctions *);
	~logfunctions(void);

	void info(const char *fmt, ...);
	void error(const char *fmt, ...);
	void panic(const char *fmt, ...);
	void ldebug(const char *fmt, ...);
	void put(char *);
	void settype(int) { return; }
	char *getprefix () { return prefix; }
        };


class state_file {
  void init(void);
public:
  FILE *file;
  class logfunctions *log;

  FILE *get_handle();
  void write(Bit8u);
  void write(Bit16u);
  void write(Bit32u);
  void write(Bit64u);
  void write(const void*, size_t);
  void read(Bit8u &);
  void read(Bit16u &);
  void read(Bit32u &);
  void read(Bit64u &);
  void read(void *, size_t);
  void write_check(const char *);
  void read_check (const char *);

  state_file (const char *name, const char *options);
  state_file (FILE *f);
  ~state_file();
};

#endif /* cplusplus */

#endif /* __BXEMU_H */
