/************************************************************************
 * $Id: io_thread.c,v 1.2 2003/05/24 20:07:41 DemonLord Exp $
 ************************************************************************
 *
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2002  Kevin P. Lawton
 *
 *  io_thread.c: A framework for threaded IO events
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <stdlib.h>

#include "io_thread.h"

#define EVT_SYNC	0xFE
#define EVT_STOP	0xFF

typedef struct _io_event_t
{
  int	type;
  struct _io_event_t *next;

  void*	data;
  int	data_size;  
} io_event_t;

static eventAction	*io_handlers = NULL;

static pthread_mutex_t	evt_mutex, sync_mutex, queue_mutex;
static pthread_cond_t	evt_cond, sync_cond;
static pthread_t	dispatch_thread;

static io_event_t	*evt_head = NULL, *evt_tail = NULL;
static int		need_sync = 0;

static void* dispatcher(void* nil);

int thread_init (void)
{
  if (pthread_cond_init(&evt_cond, NULL)      || pthread_cond_init(&sync_cond, NULL) ||
      pthread_mutex_init(&evt_mutex, NULL)    || pthread_mutex_init(&sync_mutex, NULL) ||
      pthread_mutex_init(&queue_mutex, NULL))
  {
    fprintf(stderr, "Error creating condition events.\n");
    return -1;
  }

  if (pthread_create(&dispatch_thread, NULL, dispatcher, NULL))
  {
    fprintf(stderr, "Couldn't create thread, fail to single threaded mode.\n");
    dispatch_thread = -1;
    return -1;
  }

  return 0;
}

void thread_fini (void)
{
  queue_flush();
  post_evt(EVT_STOP, NULL, 0);

  if (dispatch_thread != -1)
    if (pthread_join(dispatch_thread, NULL))
      fprintf(stderr, "Error joining thread, might cause corruption");

  pthread_cond_destroy(&evt_cond);
  pthread_cond_destroy(&sync_cond);
  pthread_mutex_destroy(&evt_mutex);
  pthread_mutex_destroy(&sync_mutex);
  pthread_mutex_destroy(&queue_mutex);
}

void set_io_handlers (eventAction *handler_vector)
{
  io_handlers = handler_vector;
}

void post_evt (int type, void* data, int data_size)
{
  io_event_t *newevent;

  newevent = (io_event_t*) malloc(sizeof(io_event_t));
  newevent->type = type;
  newevent->next = NULL;
  newevent->data_size = data_size;
  newevent->data = malloc(data_size);
  memcpy(newevent->data, data, data_size);

  pthread_mutex_lock(&evt_mutex);
  pthread_mutex_lock(&queue_mutex);
  if (evt_head == NULL)
    evt_head = evt_tail = newevent;
  else
    evt_tail = evt_tail->next = newevent;
  need_sync = 1;
  pthread_mutex_unlock(&queue_mutex);

  pthread_cond_signal(&evt_cond);
  pthread_mutex_unlock(&evt_mutex);
}

void queue_flush (void)
{
  if (!need_sync)
    return;
  pthread_mutex_lock(&sync_mutex);
  post_evt(EVT_SYNC, NULL, 0);
  pthread_cond_wait(&sync_cond, &sync_mutex);
  pthread_mutex_unlock(&sync_mutex);
}

static void* dispatcher (void* nil)
{
  io_event_t *evt;
  int done = 0;

  while (!done)
  {
    pthread_mutex_lock(&evt_mutex);
    if (evt_head == NULL)
      pthread_cond_wait(&evt_cond, &evt_mutex);

    pthread_mutex_lock(&queue_mutex);
    evt = evt_head;
    evt_head = evt_head->next;
    pthread_mutex_unlock(&queue_mutex);
    pthread_mutex_unlock(&evt_mutex);
    switch (evt->type)
    {
    case EVT_SYNC:
      pthread_mutex_lock(&sync_mutex);
      need_sync = 0;
      pthread_cond_signal(&sync_cond);
      pthread_mutex_unlock(&sync_mutex);
      break;
    case EVT_STOP:
      done = 1;
      break;
    default:
      io_handlers[evt->type](evt->data);
      free(evt->data);
    }
    free(evt);
  }

  pthread_exit(NULL);
}
