/*
    pmacct (Promiscuous mode IP Accounting package)
    pmacct is Copyright (C) 2004 by Paolo Lucente
*/

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* includes */
#include "pmacct.h"
#include "plugin_hooks.h"
#include "imt_plugin.h"
#include "net_aggr.h"

/* global vars */
int got_hup_signal;

/* Functions */
void imt_plugin(int pipe_fd, struct configuration *cfgptr) 
{
  struct sockaddr cAddr;
  struct acc *acc_elem;
  struct pkt_data *data;
  struct networks_table nt;
  struct networks_table_entry *nte = NULL;
  unsigned char srvbuf[SRVBUFLEN];
  unsigned char *pipebuf;
  char path[] = "/tmp/collect.pipe";
  unsigned int insert_status;

  fd_set read_descs, bkp_read_descs; /* select() stuff */
  int select_fd;
  int cLen, num, sd, sd2;

  /* XXX: glue */
  memcpy(&config, cfgptr, sizeof(struct configuration));

  /* a bunch of default definitions and post-checks */
  pipebuf = (unsigned char *) malloc(config.buffer_size);

  setnonblocking(pipe_fd);
  memset(pipebuf, 0, config.buffer_size);
  memset(&empty_addr, 0, sizeof(struct pkt_primitives));
  no_more_space = FALSE;

  if (config.networks_file) load_networks(config.networks_file, &nt);

  if ((!config.num_memory_pools) && (!have_num_memory_pools))
    config.num_memory_pools = NUM_MEMORY_POOLS;
  
  if (!config.memory_pool_size) config.memory_pool_size = MEMORY_POOL_SIZE;  
  else {
    if (config.memory_pool_size < sizeof(struct acc)) {
      Log(LOG_ERR, "ERROR: minimum size for memory pools is: %d bytes\n", sizeof(struct acc));
      exit(1);
    }
  }

  if (!config.imt_plugin_path) config.imt_plugin_path = path; 
  if (!config.buckets) config.buckets = MAX_HOSTS;

  init_memory_pool_table(config);
  if (mpd == NULL) {
    Log(LOG_ERR, "ERROR: unable to allocate memory pools table\n");
    exit(1);
  }

  current_pool = request_memory_pool(config.buckets*sizeof(struct acc));
  if (current_pool == NULL) {
    Log(LOG_ERR, "ERROR: unable to allocate first memory pool, try with larger value.\n");
    exit(1);
  }
  a = current_pool->base_ptr;

  lru_elem_ptr = malloc(config.buckets*sizeof(struct acc *));
  if (lru_elem_ptr == NULL) {
    Log(LOG_ERR, "ERROR: unable to allocate LRU element pointers.\n");
    exit(1);
  }
  else memset(lru_elem_ptr, 0, config.buckets*sizeof(struct acc *));

  current_pool = request_memory_pool(config.memory_pool_size);
  if (current_pool == NULL) {
    Log(LOG_ERR, "ERROR: unable to allocate more memory pools, try with larger value.\n");
    exit(1);
  }

  signal(SIGHUP, reload); /* handles reopening of syslog channel */
  signal(SIGUSR1, my_sighup_handler); /* needed for clearing stats */
  signal(SIGINT, exit_now); /* exit lane */

  /* building a server for interrogations by clients */
  sd = build_query_server(config.imt_plugin_path);
  cLen = sizeof(cAddr);

  /* preparing for synchronous I/O multiplexing */
  select_fd = 0;
  FD_ZERO(&read_descs);
  FD_SET(sd, &read_descs);
  if (sd > select_fd) select_fd = sd;
  FD_SET(pipe_fd, &read_descs);
  if (pipe_fd > select_fd) select_fd = pipe_fd;
  select_fd++;
  memcpy(&bkp_read_descs, &read_descs, sizeof(read_descs));

  /* plugin main loop */
  while(1) {
    memcpy(&read_descs, &bkp_read_descs, sizeof(bkp_read_descs));
    select(select_fd, &read_descs, NULL, NULL, NULL);

    if (got_hup_signal) goto handle_hup_signal;

    /* doing server tasks */
    if (FD_ISSET(sd, &read_descs)) {
      sd2 = accept(sd, &cAddr, &cLen);
      switch (fork()) {
      case 0: /* Child */
        close(sd);
        num = recv(sd2, srvbuf, SRVBUFLEN, 0);
        if (num > 0) process_query_data(sd2, srvbuf, num);
        if (config.debug) Log(LOG_DEBUG, "Closing connection with client ...\n");
        close(sd2);
        exit(1);
      default: /* Parent */
        close(sd2);
      }
    }

    /* clearing stats if necessary */
    if (got_hup_signal) {
      handle_hup_signal:
      clear_memory_pool_table();
      current_pool = request_memory_pool(config.buckets*sizeof(struct acc));
      if (current_pool == NULL) {
        Log(LOG_ERR, "ERROR: Cannot allocate my first memory pool, try with larger value.\n");
        exit(1);
      }
      a = current_pool->base_ptr;

      current_pool = request_memory_pool(config.memory_pool_size);
      if (current_pool == NULL) {
        Log(LOG_ERR, "ERROR: Cannot allocate more memory pools, try with larger value.\n");
        exit(1);
      }
      got_hup_signal = FALSE;
      no_more_space = FALSE;
    }

    if (FD_ISSET(pipe_fd, &read_descs)) {
      if ((num = read(pipe_fd, pipebuf, config.buffer_size)) == 0)
        exit(0); /* we exit silently; something happened at the write end */

      if (num > 0) {
	data = (struct pkt_data *) (pipebuf+sizeof(struct ch_buf_hdr));
        insert_status = 0;
        acc_elem = NULL;

	while (((struct ch_buf_hdr *)pipebuf)->num) {
          if (config.what_to_count == COUNT_SUM_HOST) {
	    struct pkt_primitives addr;

	    memset(&addr, 0, sizeof(addr));
	    addr.src_ip.s_addr = data->primitives.src_ip.s_addr; 
            acc_elem = insert_accounting_structure(&addr, ntohs(data->pkt_len));
            addr.src_ip.s_addr = data->primitives.dst_ip.s_addr;
            acc_elem = insert_accounting_structure(&addr, ntohs(data->pkt_len));
            insert_status |= INSERT_ALREADY_DONE;
          }

	  if (config.what_to_count & COUNT_SRC_NET) 
	    binsearch(&nt, &data->primitives.src_ip);

	  if (config.what_to_count & COUNT_DST_NET) 
	    binsearch(&nt, &data->primitives.dst_ip);

          if (!insert_status)
            acc_elem = insert_accounting_structure(&data->primitives, ntohs(data->pkt_len));

	  ((struct ch_buf_hdr *)pipebuf)->num--;
	  if (((struct ch_buf_hdr *)pipebuf)->num) data++;
        }
      }
    } 
  }
}

void exit_now(int signum)
{
  exit(0);
}
