/*
    pmacct (Promiscuous mode IP Accounting package)
    pmacct is Copyright (C) 2004 by Paolo Lucente
*/

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if no, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#define __PLUGIN_HOOKS_C

/* includes */
#include "pmacct.h"
#include "plugin_hooks.h"
#include "pkt_handlers.h"

/* functions */

/* load_plugins() starts plugin processes; creates
   pipes and handles them inserting in channels_list
   structure */
void load_plugins(struct pcap_device *device)
{
  int pid, v, pipelen, index;
  struct plugins_list_entry *list = plugins_list;
  unsigned int l = sizeof(list->cfg.pipe_size);

  while (list) {
    if ((*list->type.func)) {
      /* creating communication channel */
      socketpair(AF_UNIX, SOCK_DGRAM, 0, list->pipe);
      if (list->cfg.pipe_size) {
	if (list->cfg.pipe_size < (sizeof(struct pkt_data)+sizeof(struct ch_buf_hdr)))
	  list->cfg.pipe_size = sizeof(struct pkt_data)+sizeof(struct ch_buf_hdr);
	setsockopt(list->pipe[0], SOL_SOCKET, SO_RCVBUF, &list->cfg.pipe_size, l);
	setsockopt(list->pipe[1], SOL_SOCKET, SO_SNDBUF, &list->cfg.pipe_size, l);
      }
      getsockopt(list->pipe[0], SOL_SOCKET, SO_RCVBUF, &v, &l);
      if (list->cfg.debug) Log(LOG_INFO, "INFO: %s-%s recv buffer: %d\n", list->name, list->type.string, v); 
      pipelen = v;
      getsockopt(list->pipe[1], SOL_SOCKET, SO_SNDBUF, &v, &l);
      if (list->cfg.debug) Log(LOG_INFO, "INFO: %s-%s send buffer: %d\n", list->name, list->type.string, v); 
      if (v < pipelen) pipelen = v;

      /* checking transfer buffer size between 'core' <-> 'plugins' */
      if (list->cfg.buffer_size < (sizeof(struct pkt_data)+sizeof(struct ch_buf_hdr)))
        list->cfg.buffer_size = sizeof(struct pkt_data)+sizeof(struct ch_buf_hdr);
      if (list->cfg.buffer_size > pipelen)
        list->cfg.buffer_size = pipelen; 

      /* compiling aggregation filter if needed */
      if (list->cfg.a_filter) {
	bpf_u_int32 localnet, netmask;  /* pcap library stuff */
	char errbuf[PCAP_ERRBUF_SIZE];

	pcap_lookupnet(config.dev, &localnet, &netmask, errbuf);
	if (pcap_compile(device->dev_desc, &list->cfg.bpfp_a_filter, list->cfg.a_filter, 0, netmask) < 0)
          Log(LOG_WARNING, "WARN: %s\nWARN: aggregation filter disabled.\n", pcap_geterr(device->dev_desc));
      }
      
      switch (pid = fork()) {  
      case 0: /* Child */
	close(list->pipe[1]);
	(*list->type.func)(list->pipe[0], &list->cfg);
	exit(0);
      default: /* Parent */
	list->pid = pid;
	close(list->pipe[0]);
	setnonblocking(list->pipe[1]);
	insert_pipe_channel(list->cfg.what_to_count, list->pipe[1], &list->cfg.bpfp_a_filter, list->cfg.buffer_size);
      }
    }
    list = list->next;
  }

  sort_pipe_channels();
}

void exec_plugins(struct packet_ptrs *pptrs) 
{
  int num, ret;
  struct pkt_data *pdata;
  struct plugins_list_entry *list = plugins_list;
  int index, pindex;

  index = 0; pindex = 0;
  while (channels_list[index].aggregation) {
    if (index > 0) pindex = index-1;
    if (channels_list[index].aggregation == -1); /* channel has been marked as deleted */
    else {
      if (bpf_filter(channels_list[index].filter->bf_insns,
	  pptrs->packet_ptr, pptrs->pkthdr->len, pptrs->pkthdr->caplen)) {
        /* constructing buffer: supported primitives + packet total length */
	/* XXX: to get optimized */
        num = 0;
        pdata = (struct pkt_data *) channels_list[index].bufptr;
        while (channels_list[index].phandler[num] != NULL) {
          (*channels_list[index].phandler[num])(pptrs, pdata);
          num++;
        }
        pdata->pkt_len = ((struct my_iphdr *) pptrs->iph_ptr)->ip_len;

        ((struct ch_buf_hdr *)channels_list[index].buf)->num++;
        channels_list[index].bufptr += sizeof(struct pkt_data);

        if (channels_list[index].bufptr >= channels_list[index].bufend) {
          if (write(channels_list[index].pipe, channels_list[index].buf, channels_list[index].bufsize) == -1) {
	    if (errno == EAGAIN) Log (LOG_ERR, "ERROR: pipe buffer full. Try with a larger value. Exiting ...\n");
	    list = search_plugin_by_pipe(channels_list[index].pipe);
	    if (list) {
	      Log(LOG_ERR, "ERROR: connection lost to '%s-%s'; closing connection.\n", list->name, list->type.string);
	      close(channels_list[index].pipe);
	      delete_pipe_channel(channels_list[index].pipe);
	      kill(list->pid, SIGINT); /* forces the plugin to take the exit lane if close() has been ignored */
              ret = delete_plugin_by_id(list->id);
	      if (!ret) {
	        Log(LOG_ERR, "ERROR: no more plugins active. Shutting down.\n");
	        exit(1);
	      } 
	    }
	    else Log(LOG_WARNING, "WARN: Hmmmm. Unable to find pipe %d.\n", channels_list[index].pipe); 
          }

          /* rewind pointer */
          channels_list[index].bufptr = channels_list[index].buf+sizeof(struct ch_buf_hdr);
          ((struct ch_buf_hdr *)channels_list[index].buf)->num = 0;
	}
      }
    }
    index++;
  }
}

int insert_pipe_channel(unsigned int aggregation, int pipe, struct bpf_program *bpfp, int bufsz)
{
  struct channels_list_entry *chptr; 
  int index = 0, done = FALSE;  

  while (index < MAX_N_PLUGINS) {
    chptr = &channels_list[index]; 
    if (chptr->aggregation == 0) { /* found room */
      chptr->aggregation = aggregation;
      chptr->pipe = pipe; 
      if (bpfp) chptr->filter = bpfp; 
      chptr->bufsize = bufsz;
      chptr->buf = malloc(bufsz);
      if (!chptr->buf) {
        Log(LOG_ERR, "ERROR: unable to allocate channel buffer. Exiting ...\n");
        exit(0);
      } 
      memset(chptr->buf, 0, bufsz);
      chptr->bufptr = chptr->buf+sizeof(struct ch_buf_hdr);
      chptr->bufend = chptr->buf+bufsz;
      
      done = TRUE;
      break;
    }
    index++;
  }

  return done;
}

void delete_pipe_channel(int pipe)
{
  struct channels_list_entry *chptr;
  int index = 0;

  while (index < MAX_N_PLUGINS) {
    chptr = &channels_list[index];

    if (chptr->pipe == pipe) {
      chptr->aggregation = -1;
      break;
    }

    index++;
  }
}

/* trivial sorting(tm) :-) */
void sort_pipe_channels()
{
  struct channels_list_entry ctmp;
  int x = 0, y = 0; 

  while (x < MAX_N_PLUGINS) {
    if (!channels_list[x].aggregation) break;
    y = x+1; 
    while (y < MAX_N_PLUGINS) {
      if (!channels_list[y].aggregation) break;
      if (channels_list[x].aggregation == channels_list[y].aggregation) {
	if (y == x+1) x++;
	else {
	  memcpy(&ctmp, &channels_list[x+1], sizeof(struct channels_list_entry));
	  memcpy(&channels_list[x+1], &channels_list[y], sizeof(struct channels_list_entry));
	  memcpy(&channels_list[y], &ctmp, sizeof(struct channels_list_entry));
	  x++;
	}
      }
      y++;
    }
    x++;
  }
}

