/* 
 * restorecon
 *
 * AUTHOR:  Dan Walsh <dwalsh@redhat.com>
 *
 * PURPOSE:
 * This program takes a list of files and sets their security context
 * to match the specification returned by matchpathcon.
 *
 * USAGE:
 * restorecon [-nv] pathname...
 * 
 * -n	Do not change any file labels.
 * -v	Show changes in file labels.  
 *
 * pathname...	The file(s) to label 
 *
 * EXAMPLE USAGE:
 * restorecon /dev/tty*
 *
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <selinux/selinux.h>
#include <getopt.h>

void usage(const char * const name)
{	
  fprintf(stderr,
	  "usage:  %s [-nv] pathname...\n",  name);
  exit(1);
}

int main(int argc, char **argv) {
  security_context_t scontext;
  security_context_t prev_context;
  int i=0;
  int retcontext=0;
  int retval=0;
  int errors=0;
  int change=1;
  int verbose=0;
  char opt;
  char *progname=argv[0];

  if (is_selinux_enabled() <= 0 )
    exit(0);

  while ((opt = getopt(argc, argv, "nv")) > 0) {
    switch (opt) {
    case 'n':
      change = 0;
      break;
    case 'v':
      verbose = 1;
      break;
    case '?':
      usage(argv[0]);
    }
  }
  for (i=optind; i< argc; i++) {
    char *filename=argv[i];
    int len=strlen(filename);
    /* 
       Eliminate trailing /
    */
    if (len > 0 && filename[len-1]=='/' && (strcmp(filename,"/") != 0)) {
      filename[len-1]=0;
    }
    retval = matchpathcon(filename, 0, &scontext);
    if (retval < 0) {
      fprintf(stderr,"matchpathcon(%s) failed\n", filename);
      errors++;
      continue;
    } 
    retcontext=lgetfilecon(filename,&prev_context);
    
    if (retcontext >= 0 || errno == ENODATA) {
      if (retcontext < 0 || strcmp(prev_context,scontext) != 0) {
	if (change) {
	  retval=lsetfilecon(filename,scontext);
	}
	if (retval<0) {
	  fprintf(stderr,"%s set context %s->%s failed:'%s'\n",
		  progname, filename, scontext, strerror(errno));
	  errors++;
	} else 	
	  if (verbose)
	    fprintf(stderr,"%s set context %s->%s\n",
		    progname, filename, scontext);
      } 
      if (retcontext >= 0)
	freecon(prev_context);
    } 
    else {
	errors++;
	fprintf(stderr,"%s get context on %s failed: '%s'\n",
		progname, filename, strerror(errno));
    }
    freecon(scontext);
  }
  return errors;
}
