/* 
 * restorecon
 *
 * AUTHOR:  Dan Walsh <dwalsh@redhat.com>
 *
 * PURPOSE:
 * This program takes a list of files and sets their security context
 * to match the specification returned by matchpathcon.
 *
 * USAGE:
 * restorecon [-Rnv] pathname...
 * 
 * -e   Specify directory to exclude
 * -n	Do not change any file labels.
 * -v	Show changes in file labels.  
 * -o filename save list of files with incorrect context
 * -F	Force reset of context to match file_context for customizable files
 *
 * pathname...	The file(s) to label 
 *
 * EXAMPLE USAGE:
 * restorecon /dev/tty*
 *
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <selinux/selinux.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#define __USE_XOPEN_EXTENDED 1	/* nftw */
#include <ftw.h>

static int change=1;
static int verbose=0;
static FILE *outfile=NULL;
static char *progname;
static int errors=0;
static int recurse=0;
static int force=0;

#define MAX_EXCLUDES 100
static int excludeCtr=0;
struct edir {
	char *directory;
        size_t size;
};
static struct edir excludeArray[MAX_EXCLUDES];
static int add_exclude(const char *directory) {
  struct stat sb;
  if(directory == NULL || directory[0] != '/') {
    fprintf(stderr, "Full path required for exclude: %s.\n", 
	    directory);
    return 1;
  }
  if(lstat(directory, &sb)) {
    fprintf(stderr, "Directory \"%s\" not found, ignoring.\n", directory);
    return 0;
  }
  if ((sb.st_mode & S_IFDIR) == 0 ) {
    fprintf(stderr, "\"%s\" is not a Directory: mode %o, ignoring\n", directory,sb.st_mode);
    return 0;
  }

  if (excludeCtr == MAX_EXCLUDES) {
    fprintf(stderr, "Maximum excludes %d exceeded.\n", MAX_EXCLUDES);
    return 1;
  }

   excludeArray[excludeCtr].directory = strdup(directory);
  if (!excludeArray[excludeCtr].directory) {
    fprintf(stderr, "Out of memory.\n");
    return 1;
  }
  excludeArray[excludeCtr++].size = strlen(directory);

  return 0;
}
static int exclude(const char *file) {
	int i=0;
	for(i=0; i < excludeCtr; i++) { 
		if (strncmp(file,excludeArray[i].directory,excludeArray[i].size)==0) {
			if (file[excludeArray[i].size]==0 || 
			    file[excludeArray[i].size]=='/') {
				return 1;
			}
		}
	}
	return 0;
}

/* Compare two contexts to see if their differences are "significant",
 * or whether the only difference is in the user. */
static int only_changed_user(const char *a, const char *b)
{
	char *rest_a, *rest_b; /* Rest of the context after the user */
	if (force) return 0;
	if (!a || !b) return 0;
	rest_a = strchr(a, ':');
	rest_b = strchr(b, ':');
	if (!rest_a || !rest_b) return 0;
	return  (strcmp(rest_a, rest_b) == 0);
}

void usage(const char * const name)
{	
  fprintf(stderr,
	  "usage:  %s [-Rnv] [-e excludedir ] [-o filename ] [-f filename | pathname... ]\n",  name);
  exit(1);
}
int restore(char *filename) {
  int retcontext=0;
  int retval=0;
  security_context_t scontext=NULL;
  security_context_t prev_context=NULL;
  int len=strlen(filename);
  struct stat st;
  char path[PATH_MAX+1];
  int user_only_changed=0;
  /* 
     Eliminate trailing /
  */
  if (len > 0 && filename[len-1]=='/' && (strcmp(filename,"/") != 0)) {
    filename[len-1]=0;
  }

  if (excludeCtr > 0 && exclude(filename)) {
      return 0;
  }

  if (lstat(filename, &st)!=0) {
    fprintf(stderr,"lstat(%s) failed: %s\n", filename,strerror(errno));
    return 1;
  }
  if (S_ISLNK(st.st_mode)) {
    if (verbose>1)
      fprintf(stderr,"Warning! %s refers to a symbolic link, not following last component.\n", filename);
    char *p = NULL, *file_sep;
    char *tmp_path = strdup(filename);
    if (!tmp_path) {
      fprintf(stderr,"strdup on %s failed:  %s\n", filename,strerror(errno));
      return 1;
    }
    file_sep = strrchr(tmp_path, '/');
    if(file_sep)
    {
      *file_sep = 0;
      file_sep++;
      p = realpath(tmp_path, path);
    }
    if (!p || strlen(path) + strlen(file_sep) + 1 > PATH_MAX) {
      fprintf(stderr,"realpath(%s) failed %s\n", filename, strerror(errno));
      free(tmp_path);
      return 1;
    }
    sprintf(p + strlen(p), "/%s", file_sep);
    filename = p;
    free(tmp_path);
  } else {
    char *p;
    p = realpath(filename, path);
    if (!p) {
      fprintf(stderr,"realpath(%s) failed %s\n", filename, strerror(errno));
      return 1;
    }
    filename = p;
  }
  if (excludeCtr > 0 && exclude(filename)) {
      return 0;
  }
  retval = matchpathcon(filename, st.st_mode, &scontext);
  if (retval < 0) {
    if (errno == ENOENT)
      return 0;
    fprintf(stderr,"matchpathcon(%s) failed %s\n", filename,strerror(errno));
    return 1;
  } 
  if (strcmp(scontext,"<<none>>")==0) {
    freecon(scontext);
    return 0;
  }
  retcontext=lgetfilecon(filename,&prev_context);
  
  if (retcontext >= 0 || errno == ENODATA) {
    int customizable=0;
    if (retcontext < 0) prev_context=NULL;
    if (retcontext < 0 || force || 
	(strcmp(prev_context,scontext) != 0 && 	
	 !(customizable=is_context_customizable(prev_context) > 0))) {
      if (outfile) {
	fprintf(outfile, "%s\n", filename);
      }
      user_only_changed = only_changed_user(scontext, prev_context);
      if (change && !user_only_changed) {
	retval=lsetfilecon(filename,scontext);
      }
      if (retval<0) {
	  fprintf(stderr,"%s set context %s->%s failed:'%s'\n",
		  progname, filename, scontext, strerror(errno));
	  if (retcontext >= 0)
	    freecon(prev_context);
	  freecon(scontext);
	  return 1;
      } else 	
	      if (verbose && 
		  (verbose > 1 || !user_only_changed))
		      fprintf(stderr,"%s reset %s context %s->%s\n",
			      progname, filename, (retcontext >= 0 ? prev_context : ""), scontext);
    }
    if (verbose > 1 && customizable>0) {
	    fprintf(stderr,"%s: %s not reset customized by admin to %s\n",
		      progname, filename, prev_context);
    }

    if (retcontext >= 0)
      freecon(prev_context);
  } 
  else {
    errors++;
    fprintf(stderr,"%s get context on %s failed: '%s'\n",
	    progname, filename, strerror(errno));
  }
  freecon(scontext);
  return errors;
}
static int apply_spec(const char *file,
		      const struct stat *sb_unused __attribute__((unused)),
		      int flag,
		      struct FTW *s_unused __attribute__((unused)))
{
	if (flag == FTW_DNR) {
		fprintf(stderr, "%s:  unable to read directory %s\n",
			progname, file);
		return 0;
	}
	errors=errors+restore((char *)file);
	return 0;
}
void process(char *buf) {
      if (recurse) {
	if (nftw
	    (buf, apply_spec, 1024, FTW_PHYS)) {
	  fprintf(stderr,
		  "%s:  error while labeling files under %s\n",
		  progname, buf);
	  errors++;
	}
      }
      else
	errors=errors+restore(buf);
}
int main(int argc, char **argv) {
  int i=0;
  char *file_name=NULL;
  int errors=0;
  int file=0;
  int opt;
  char buf[PATH_MAX];

  memset(excludeArray,0, sizeof(excludeArray));

  progname=argv[0];
  if (is_selinux_enabled() <= 0 )
    exit(0);

  memset(buf,0, sizeof(buf));

  while ((opt = getopt(argc, argv, "FRnvf:o:e:")) > 0) {
    switch (opt) {
    case 'n':
      change = 0;
      break;
    case 'R':
      recurse = 1;
      break;
    case 'F':
      force = 1;
      break;
    case 'e':
      if ( add_exclude(optarg) ) exit(1);
      break;
    case 'o':
      outfile = fopen(optarg,"w");
      if (!outfile) {
	 fprintf(stderr, "Error opening %s: %s\n", 
		 optarg, strerror(errno));
	 usage(argv[0]);
      }
      break;
    case 'v':
      verbose++;
      break;
    case 'f':
      file = 1;
      file_name=optarg;
      break;
    case '?':
      usage(argv[0]);
    }
  }
  if (file) {
    FILE *f=stdin;
    if (strcmp(file_name,"-")!=0) 
      f=fopen(file_name,"r");
	
    if (f==NULL) {
      fprintf(stderr,"Unable to open %s: %s\n", file_name, strerror(errno));
      usage(argv[0]);
    }
    while(fgets(buf,PATH_MAX,f)) {
      buf[strlen(buf)-1]=0;
      process(buf);
    }
    if (strcmp(file_name,"-")!=0) 
      fclose(f);
  } 
  else {
    for (i=optind; i< argc; i++) {
      process(argv[i]);
    }
  }
  if (outfile) 
    fclose(outfile);
  
  return errors;
}
