/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "polyxdef-main.h"
#include "polyxdef-defctxt.h"
#include "polyxdef-ui-polchemdef.h"
#include "polyxdef-ui-atomdef.h"


/****************************************************************************
		 This file is the entry point of the

			   polyxdef module

		   in the polyxmass software suite.
****************************************************************************/


PxmDefCtxt *
polyxdef_main_prepare_new_context (gchar *filename, gint def_type)
{
  /* The user has asked that a new definition context be launched. We
     just comply with the command. Depending on the 'def_type'
     parameter, we will launch a polymer chemistry definition context
     or an atom definition context.

     Note that 'filename' can be NULL, in which case the context will
     be launched without loading any data from disk, ie will be "naked".
  */
  PxmDefCtxt *defctxt = NULL;


  g_assert (def_type == ATOM_DEF || def_type == POLCHEM_DEF);
  

  /*
    First allocate the new defctxt instance: whatever the 'def_type'
    param, we'll need one !
  */
  defctxt = polyxdef_defctxt_new ();

  if (def_type == ATOM_DEF)
    {
      /* We are asked to launch a definition context that is for atom
	 definitions. Check if 'filename' is NULL or not.
      */
      if (filename != NULL)
	{
	  /* Test if the file actually exists...
	   */
	  if (FALSE == g_file_test (filename, G_FILE_TEST_EXISTS))
	    {
	      g_critical (
		     _("%s@%d: file asked for opening not found: '%s'\n"),
		     __FILE__, __LINE__, 
		     filename);
	      
	      return NULL;
	    }
	  
	  /* OK, at this point we know the file is there. Try to render it!
	   */
	  if (-1 == pxmchem_atom_render_xml_file (filename, 
						  defctxt->atomGPA))
	    {
	      g_critical (_("%s@%d: failed to parse the atom"
		       " definition file: '%s'.\n"),
		     __FILE__, __LINE__, filename);
	      
	      polyxdef_defctxt_free (defctxt);
      
	      return NULL;
	    }

	  /* Set that filename to the member of defctxt, so that we'll
	     be able to know what file the data were read from later.
	  */
	  defctxt->filename = g_strdup (filename);
	}
      /* End of 
	 if (filename != NULL)
      */
      else
	{
	  /* No filename is passed as parameter, which means that we
	     are starting work with an empty one. No problem, them
	     'filename' member of defctxt will remain NULL.
	  */
	}

      /* At this point we can set to the type member 'definition_type'
	 variable the value that tells we are dealing with an atom
	 definition.
      */
      defctxt->definition_type = ATOM_DEF;


      /* At this point, the data are in memory, all we have to do is
	 to construct the graphical interface stuff...
       */      
      
      /* The atom definition window, which is the main interface
	 window with the user.
      */
      if (NULL == 
	  polyxdef_ui_atomdef_wnd_setup_window (defctxt))
	{
	  g_critical (_("%s@%d: failed to set up a new atom "
		   "definition window.\n"),
		 __FILE__, __LINE__);
	  
	  polyxdef_defctxt_free (defctxt);
	  
	  return NULL;
	}
      
      /* At this point, we can add the new polyxdef context to the
	 array of such instances.
      */
      g_ptr_array_add (polyxdef_defctxtGPA, defctxt);
      
      return defctxt;
    }
  else if (def_type == POLCHEM_DEF)
    {
      /* We are asked to launch a definition context that is for atom
	 definitions. Check if 'filename' is NULL or not.
      */
      if (filename != NULL)
	{
	  /* Test if the file actually exists...
	   */
	  if (FALSE == g_file_test (filename, G_FILE_TEST_EXISTS))
	    {
	      g_critical (_("%s@%d: file asked for opening "
			    "not found: '%s'\n"),
		     __FILE__, __LINE__, 
		     filename);
	      
	      return NULL;
	    }
	  
	  /* OK, at this point we know the file is there. Try to render it!
	   */
	  defctxt->polchemdef = 
	    pxmchem_polchemdef_render_xml_file (filename);
	  
	  if (defctxt->polchemdef == NULL)
	    {
	      g_critical (_("%s@%d: failed to parse the polymer chemistry "
		       "definition file: '%s'.\n"),
		     __FILE__, __LINE__, filename);
	      
	      polyxdef_defctxt_free (defctxt);
	      
	      return FALSE;
	    }
 
	  /* We will very often need to make sure a given string, like
	     "Ala" for example, does actually correspond to a monomer
	     code. Instead of systematically iterating in the array of
	     monomers that is member of the polchemdef object, we can
	     ask if we find that code in a string (much quicker
	     stuff). For this we need to have a string of monomer
	     codes that is initialized using all the codes of all the
	     monomers in the polymer chemistry definition. This
	     delim_codes string is useful when parsing the cleavage
	     specifications for example.
	  */
	  pxmchem_polchemdef_update_delim_codes (defctxt->polchemdef,
						 libpolyxmass_globals_delim);
	  
	  
	  /* Set that filename to the member of defctxt, so that we'll
	     be able to know what file the data were read from later.
	  */
	  defctxt->filename = g_strdup (filename);
	}
      /* End of 
	 if (filename != NULL)
      */
      else
	{
	  /* No filename is passed as parameter, which means that we
	     are starting work with an empty one. No problem, them
	     'filename' member of defctxt will remain NULL.

	     But we still have to allocate a new polymer chemistry
	     definition, of course, so that when we set up the
	     definition window (defctxt->definition_wnd) we have
	     something to work on!
	  */

	  defctxt->polchemdef = pxmchem_polchemdef_new ();
	  
	  /* This string will contain no monomer code, but it won't be
	     NULL, so that when the user will start creating cleavage
	     specifications, the program will not crash.
	   */
	  pxmchem_polchemdef_update_delim_codes (defctxt->polchemdef,
						 libpolyxmass_globals_delim);
	}

      /* At this point we can set to the type member 'definition_type'
	 variable the value that tells we are dealing with a polymer
	 chemistry definition.
      */
      defctxt->definition_type = POLCHEM_DEF;

      /* At this point, the data are in memory, all we have to do is
	 to construct the graphical interface stuff...
      */      
      
      /* The polymer chemistry definition window, which is the main
	 interface window with the user.

	 Note that this is where the polchemdef->atomGPA is going to
	 be filled with PxmAtom instances corresponding to the atoms
	 defined in the atom definition file that is suited for the
	 polymer chemistry definition we are dealing with.
      */
      if (NULL == 
	  polyxdef_ui_polchemdef_wnd_setup_window (defctxt))
	{
	  g_critical (_("%s@%d: failed to set up a new polymer chemistry "
			"definition window.\n"),
		      __FILE__, __LINE__);
	  
	  polyxdef_defctxt_free (defctxt);
	  
	  return NULL;
	}

     
      /* At this point, we can add the new polyxdef context to the
	 array of such instances.
      */
      g_ptr_array_add (polyxdef_defctxtGPA, defctxt);
      
      return defctxt;
    }
  else
    g_assert_not_reached ();
  
  return NULL;
}


