/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-findmass-options.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxedit-ui-findmass-results.h"

    
GtkWidget *
polyxedit_findmass_opt_setup_wnd (GtkWidget *parent, GPtrArray *haystackGPA)
{
  PxmEditCtxt *editctxt;

  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkWidget *combo = NULL;
  GtkWidget *vbox = NULL;
  
  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;


  /* The array of oligomer arrays in which to find masses must not
     be NULL.
  */
  g_assert (haystackGPA != NULL);

  /* This function is called from windows where oligomer data are
     displayed. For example, this function is called from the window
     where oligomers obtained after cleavage of polymers or after
     fragmentation of oligomers.
     
     See polyxedit_cleave_results_wnd_find_button() to check that
     'parent' is actually the window where the oligomer results were
     displayed. We use that pointer to the parent window in order to
     set to it a pointer to *this window as a full datum. See the
     really_close function below. We want this options window to be
     automagically closed if the results window from which it was
     elicited is closed.

     Since that 'parent' window had a knowledge of what editctxt it
     was created in, we can get that context out of the 'parent'
     window ! 
  */
  g_assert (parent != NULL);

  
  gui_file = 
    g_strdup_printf ("%s/polyxedit-findmass-options.glade", 
		     userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "findmass_options_wnd", 
		       PACKAGE);

  g_free (gui_file);

  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "findmass_options_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the find mass window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }


  /* Immediately set a datum to the window, that contains the parent
     window, so that when the results of this mass find operation are
     displayed, the result window can display that value.

     The user will thus know from which cleave/fragment results window
     the findmass results come. Thus, this 'parent' window will turn
     out to be the grand_parent of the window in which this datum is
     effectively used, because the findmass-options window is the
     intermediate, in which this datum is not used.
  */
  g_object_set_data (G_OBJECT (window), "grand_parent", parent);

  editctxt = g_object_get_data (G_OBJECT (parent), "editctxt");
  g_assert (editctxt);
    g_object_set_data (G_OBJECT (window), "editctxt", editctxt);
  
  
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window), "messages_entry",
		     widget);

  /* Set the parent window pointer to the relative entry.
   */
  widget = glade_xml_get_widget (xml, "results_set_identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "results_set_identity_number_entry", 
		     widget);
  
  help = g_strdup_printf ("%p", parent);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  

  /* Also immediately set to the window a datum that points to the array
     of oligomer arrays so that we have it for later.
  */
  g_object_set_data (G_OBJECT (window), "haystackGPA", haystackGPA);
  
  /* Now deal systematically with the widgets that we will use later.
   */
  widget = glade_xml_get_widget (xml, 
				 "polseq_data_sel_only_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "polseq_data_sel_only_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "polseq_data_whole_seq_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "polseq_data_whole_seq_radiobutton",
		     widget);


  widget = glade_xml_get_widget (xml, 
				 "mono_unique_mass_entry");
  g_object_set_data (G_OBJECT (window), 
		     "mono_unique_mass_entry",
		     widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "activate",
     G_CALLBACK (polyxedit_findmass_opt_mass_entry_activate), 
     window);
  
  widget = glade_xml_get_widget (xml, 
				 "mono_masses_textview");
  g_object_set_data (G_OBJECT (window), 
		     "mono_masses_textview",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "avg_unique_mass_entry");
  g_object_set_data (G_OBJECT (window), 
		     "avg_unique_mass_entry",
		     widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "activate",
     G_CALLBACK (polyxedit_findmass_opt_mass_entry_activate), 
     window);
  

  widget = glade_xml_get_widget (xml, 
				 "avg_masses_textview");
  g_object_set_data (G_OBJECT (window), 
		     "avg_masses_textview",
		     widget);


  /*
    Initial version used when Glade-2 was not faulty.
    
    widget = glade_xml_get_widget (xml, 
    "mono_tolerance_combo");
    g_object_set_data (G_OBJECT (window), 
    "mono_tolerance_combo",
    widget);
  */
  /* Momentaneous manual creation of the GtkComboBox due to 
     a bug in Glade-2 that prevents it from saving a file that contains
     such a widget.
  */
  combo = gtk_combo_box_new_text ();
  g_assert (combo != NULL);
  
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_amu);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_pct);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_ppm);
  
  /* The "tolerance_units_amu" should be set active by default.
   */
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);

  g_object_set_data (G_OBJECT (window), 
		     "mono_tolerance_combo",
		     combo);
  /* Get the box where the combo must be packed.
   */
  vbox = glade_xml_get_widget (xml, 
			      "mono_tolerance_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "mono_tolerance_vbox",
		     vbox);
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);

  /* The combo has to be before the tolerance GtkEntry box.
   */
  gtk_box_reorder_child (GTK_BOX (vbox), combo, 0);

  widget = glade_xml_get_widget (xml, 
				 "mono_tolerance_entry");
  g_object_set_data (G_OBJECT (window), 
		     "mono_tolerance_entry",
		     widget);

  /*
    Initial version used when Glade-2 was not faulty.
    
    widget = glade_xml_get_widget (xml, 
    "avg_tolerance_combo");
    g_object_set_data (G_OBJECT (window), 
    "avg_tolerance_combo",
    widget);
  */
  /* Momentaneous manual creation of the GtkComboBox due to 
     a bug in Glade-2 that prevents it from saving a file that contains
     such a widget.
  */
  combo = gtk_combo_box_new_text ();
  g_assert (combo != NULL);
  
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_amu);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_pct);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
			     polyxedit_tolerance_units_ppm);

  /* The "tolerance_units_amu" should be set active by default.
   */
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);

  g_object_set_data (G_OBJECT (window), 
		     "avg_tolerance_combo",
		     combo);
  /* Get the box where the combo must be packed.
   */
  vbox = glade_xml_get_widget (xml, 
			      "avg_tolerance_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "avg_tolerance_vbox",
		     vbox);
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);

  /* The combo has to be before the tolerance GtkEntry box.
   */
  gtk_box_reorder_child (GTK_BOX (vbox), combo, 0);

  widget = glade_xml_get_widget (xml, 
				 "avg_tolerance_entry");
  g_object_set_data (G_OBJECT (window), 
		     "avg_tolerance_entry",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "unique_mass_find_mode_checkbutton");
  g_object_set_data (G_OBJECT (window), 
		     "unique_mass_find_mode_checkbutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "find_button");
  g_object_set_data (G_OBJECT (window), 
		     "find_button",
		     widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_findmass_opt_find_button), 
		    window);


  gtk_widget_show_all (GTK_WIDGET (window));
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));

  /* The signal of the window itself.
   */
  /* Signal / callback connections.
   */
  g_signal_connect 
    (G_OBJECT (window),
     "delete_event",
     G_CALLBACK (polyxedit_findmass_opt_wnd_delete_event), 
     NULL);

  g_signal_connect 
    (G_OBJECT (window),
     "destroy_event",
     G_CALLBACK (polyxedit_findmass_opt_wnd_destroy_event), 
     NULL);

  /* Set this window pointer as a full datum to the results window
     from which this window is created (typicall the results window is
     a window where oligomers obtained from a cleavage or a
     fragmentation are displayed. We want to make sure that, if the
     results window from which *this window is created is closed, then
     *this window is closed also.

     There might be more than one findmass options' window opened for
     a given results window, and we do not want that the second window
     destroys the datum name of the first window, so we create an
     uambiguous datum name each time.
  */
  help = g_strdup_printf ("findmass_options_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (parent),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_findmass_opt_wnd_really_close);
  
  g_free (help);

  return window;
}

void
polyxedit_findmass_opt_mass_entry_activate (GtkEntry *entry,
					      gpointer data)
{
  PxmEditCtxt *editctxt = NULL;

  GtkWidget *check = NULL;

  GtkWidget *window = data;
  GtkWidget *display_window = NULL;
  GtkWidget *parent_wnd = NULL;

  /* Textview stuff....
   */
  GtkTextBuffer *buffer = NULL;
  GtkTextView *textview = NULL;
  GtkTextIter start;
  GtkTextIter end;

  gchar *new_text = NULL;
  gchar *wnd_desc = NULL;

  PxmFindmassOpt *fmopt = NULL;

  PxmMassType mass_type = PXM_MASS_NONE;
  
  gchar *name;
  gchar *help;

  GPtrArray *alloligsGPA = NULL;
  GPtrArray *haystackGPA = NULL;
  GPtrArray *fmoptGPA = NULL;

  gint count = 0;
  
  gboolean unique_mode = FALSE;
  
  
  g_assert (window != NULL);
  g_assert (entry != NULL);
  
  /* Get the haystackGPA (array of oligomer arrays) pointer
     immediately. This is the array of oligomer arrays where the 
     oligomer masses are going to be found.
  */
  haystackGPA = g_object_get_data (G_OBJECT (window), "haystackGPA");
  g_assert (haystackGPA != NULL);

  /* Immediately establish if we are working on a mono or avg mass.
   */
  name = (gchar *) gtk_widget_get_name (GTK_WIDGET (entry));
  
  if (0 == strcmp ("mono_unique_mass_entry", name))
    mass_type = PXM_MASS_MONO;
  else
    mass_type = PXM_MASS_AVG;
  
  /* Get to know if we are in a multi- or unique-find mass mode.
   */
  check = g_object_get_data (G_OBJECT (window), 
			     "unique_mass_find_mode_checkbutton");
  g_assert (check != NULL);
  unique_mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));

  /* Depending on 'unique_mode' we'll have different actions to take.
   */
  if (TRUE == unique_mode)
    {
      /* First step is to allocate a PxmFindmassOpt that we'll fill
	 with proper data.
      */
      fmopt = pxmchem_findmassopt_new ();

      /* Immediately set the mass type that we already know.
       */
      fmopt->mass_type = mass_type;

      /* Now get to have a properly converted mass value from the 
	 corresponding entry:
      */
      if (FALSE == polyxedit_findmass_opt_get_mass_value (window,
							  mass_type,
							  fmopt))
	{
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("Failed to get a correct mass value"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_findmassopt_free (fmopt);
	  
	  return;
	}
	 
      /* Now get to have a properly converted tolerance value from the
	 corresponding entry:
      */
      if (FALSE == 
	  polyxedit_findmass_opt_get_tolerance_value (window,
							mass_type,
							fmopt))
	{
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("Failed to get a correct tolerance value"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_findmassopt_free (fmopt);
	  
	  return;
	}
      
      /* Now get the tolerance type from the proper GtkOptionMenu widget.
       */
      if (FALSE == 
	  polyxedit_findmass_opt_get_tolerance_type (window,
						       mass_type,
						       fmopt))
	{
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("Failed to get a correct tolerance type"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_findmassopt_free (fmopt);
	  
	  return;
	}

      /* Allocate the array that will hold the fmopt instance, so that
	 we can call a unified function which is expecting an array
	 of fmopt objects.
      */
      fmoptGPA = g_ptr_array_new ();
      g_ptr_array_add (fmoptGPA, fmopt);
      
      /* Allocate the array of arrays of oligomers that will hold all
	 the arrays of oligomers organized by found masses.
      */
      alloligsGPA = g_ptr_array_new ();

      /* Perform the find operation proper.
       */
      count = polyxedit_findmass_opt_find_masses (window,
						  haystackGPA,
						  alloligsGPA,
						  fmoptGPA);
      if (count == -1)
	{
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("Failed to find the mass in the oligomer array"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_findmassopt_GPA_free (fmoptGPA);
	  
	  return;
	}
      else if (count == 0)
	{
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("Mass not found in the polymer"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_findmassopt_GPA_free (fmoptGPA);
	  
	  return;
	}

      /* At this point, we do not need the array of fmopt instances,
	 because each oligomer in the arrays that sit in the
	 alloligsGPA has a copy of the fmopt object by which it was
	 found.
      */
      pxmchem_findmassopt_GPA_free (fmoptGPA);
	  
      /* At this point we just have to display the oligomers. We pass
	 to the found oligomers-displaying window the pointer to
	 _this_ window.
       */
      display_window = polyxedit_findmass_res_wnd_setup (window,
							 alloligsGPA);
      if (display_window == NULL)
	{
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("Failed to display the oligomers in a window"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  
	  return;
	}
      
      /* Register the results' window !
       */
      /* Get the window description string from the window where the
	 cleavage/fragmentation results were displayed. We will use
	 this string to make sure the user will know exactly from
	 which data the find mass results were extracted !
      */
      parent_wnd = g_object_get_data (G_OBJECT (window), "grand_parent");
      g_assert (parent_wnd != NULL);
      
      editctxt = g_object_get_data (G_OBJECT (parent_wnd), "editctxt");
      g_assert (editctxt != NULL);
      
      wnd_desc = g_object_get_data (G_OBJECT (parent_wnd), "wnd_desc");
      g_assert (wnd_desc != NULL);
      
      help = g_strdup_printf (_("Find Mass For Results in Window ID '%p'"
			      " ('%s')"), 
			      parent_wnd, wnd_desc);
      
      g_object_set_data_full (G_OBJECT (display_window),
			      "wnd_desc", help, 
			      (GDestroyNotify) free);

      if (NULL == polyxmass_winmngmt_register_window 
	  ("POLYXEDIT",
	   display_window,
	   window,
	   editctxt,
	   _("Results of Find Mass"),
	   help,
	   TRUE,
	   polyxedit_findmass_results_wnd_make_report))
	g_critical (_("%s@%d: failed to register the "
		      "find mass results' window\n"),
		    __FILE__, __LINE__);
      
      /* Do not free help, because we have set it as a full datum to
	 the window where the find mass results are to be displayed !
	 
	 g_free (help);
      */

      /* Finally we can return, our work here is finished.
       */
      return;
    }
  else
    {
      /* We are working in the multi find mass mode, which means
	 that we just validate the text that is in the entry and 
	 put it in the textview that corresponds to the entry from
	 which the "activate" signal was emitted.
      */
      if (0 == strcmp ("mono_unique_mass_entry", name))
	{
	  textview = g_object_get_data (G_OBJECT (window),
					"mono_masses_textview");
	}
      else if (0 == strcmp ("avg_unique_mass_entry", name))
	{
	  textview = g_object_get_data (G_OBJECT (window),
					"avg_masses_textview");
	}
      else
	g_error (_("%s@%d: failed to interpret the window layout\n"),
	       __FILE__, __LINE__);

      /* Get the buffer that holds the text presently (if any) in the
	 textview.
      */
      buffer = gtk_text_view_get_buffer (textview);

      /* We now know where the text to be read is and where it should
	 be put.
      */
      gtk_text_buffer_get_selection_bounds (buffer, &start, &end);
      
      /* At this point, the 'end' textiter should point to the insertion
	 point. This is where the contents of the entry that emitted 
	 the "activate" signal are to be put.
      */
      help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

      if (strlen (help) <= 0)
	{
	  g_free (help);
	  
	  return;
	}
      
      /* Append a space followed by a newline to the help text, so
	 that when it is inserted in the textview buffer the line feed
	 ensures that all values succcessively entered are displayed
	 in column. The space character ensures that the user may put
	 the cursor somewhere in the textview buffer and insert a new
	 value through the entry widget.
      */
      new_text = g_strconcat (help, " \n", NULL);
      g_free (help);

      gtk_text_buffer_insert (buffer, &end, new_text, -1);
      
      g_free (new_text);

      /* Now that the entry's contents have been copied to the textview
	 we can erase them from the entry.
      */
      gtk_entry_set_text (GTK_ENTRY (entry), "");
      
    }
  
  return;
}


gint
polyxedit_findmass_opt_find_masses (GtkWidget *window, 
				    GPtrArray *haystackGPA,
				    GPtrArray *alloligsGPA,
				    GPtrArray *fmoptGPA)
{
  gint count = 0;
  gint total = 0;
  gint iter = 0;
  
  PxmFindmassOpt *fmopt = NULL;

  GPtrArray *oligsGPA = NULL;
  

  g_assert (window != NULL);
  g_assert (haystackGPA != NULL);
  g_assert (alloligsGPA != NULL);
  g_assert (fmoptGPA != NULL);
 
  
  for (iter = 0; iter < fmoptGPA->len ; iter++)
    {
      fmopt = g_ptr_array_index (fmoptGPA, iter);
      g_assert (fmopt != NULL);

      /* Allocate an array for this specific mass find.
       */
      oligsGPA = g_ptr_array_new ();
      
      count = pxmchem_findmass_oligomer_GPA_of_GPA (haystackGPA,
						    oligsGPA,
						    fmopt);
      if (count == -1)
	{
	  polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	     _("Failed to find one mass in the oligomer array"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  return -1;
	}
      
      /* At this point we can add the new array to the array of arrays,
	 but only if it has data in it. Otherwise we free it immediately.
      */
      if (oligsGPA->len > 0)
	g_ptr_array_add (alloligsGPA, oligsGPA);
      else
	g_ptr_array_free (oligsGPA, TRUE);
      
      total += count;
    }
  
  return total;
}
void
polyxedit_findmass_opt_find_button (GtkWidget *button,
				    gpointer data)
{
  PxmEditCtxt *editctxt = NULL;

  GtkWidget *window = data;
  GtkWidget *display_window = NULL;
  GtkWidget *parent_wnd = NULL;

  GtkWidget *entry = NULL;
  GtkWidget *check = NULL;

  /* Textview stuff....
   */
  GtkTextBuffer *buffer = NULL;
  GtkTextView *textview = NULL;
  GtkTextIter start;
  GtkTextIter end;

  gchar *masses_text = NULL;
  gchar *wnd_desc = NULL;
  gchar *help = NULL;
    
  gdouble *mass = NULL;

  PxmMassType mass_types [] = 
    {
      PXM_MASS_MONO,
      PXM_MASS_AVG,
      PXM_MASS_NONE
    };
  
  
  PxmFindmassOpt *fmopt_template = NULL;
  PxmFindmassOpt *fmopt = NULL;

  GPtrArray *massesGPA = NULL;
  GPtrArray *fmoptGPA = NULL;

  GPtrArray *alloligsGPA = NULL;
  GPtrArray *haystackGPA = NULL;
  
  gint count = 0;
  gint iter = 0;
  
  gboolean mono_ok = FALSE;
  gboolean avg_ok = FALSE;
  gboolean unique_mode = FALSE;
  

  /* The find button was clicked, which means that we have to check
     a number of things:

     - If unique mass mode is activated, we need to know wether the
     entry widgets (mono and/or avg) do contain text. If so, only one
     must contain text. Otherwise we do not know which entry use !

     - If unique mass mode is not activated, we'll have to deal with
     both mono and avg textviews, and parse their content in sequence,
     so that we get all the masses in them. For each successfully
     parsed mass we do have to create a fmopt instance to add to the
     array of PxmFindmassOpt instances.

     - Finally, once all the relevant masses have been converted to 
     fmopt objects in the array of fmopts, then mass finding can
     go on through the call of the appropriate function.
  */

  g_assert (window != NULL);
  

  /* Get the haystackGPA (array of oligomer arrays) pointer
     immediately. This is the array of oligomer arrays where the 
     oligomer masses are going to be found.
  */
  haystackGPA = g_object_get_data (G_OBJECT (window), "haystackGPA");
  g_assert (haystackGPA != NULL);



  check = g_object_get_data (G_OBJECT (window), 
			     "unique_mass_find_mode_checkbutton");
  g_assert (check != NULL);
  unique_mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));

  if (TRUE == unique_mode)
    {
      /* There must one mass entry, and one only, that has 
	 a string in it. Get the pointer to this entry and call
	 the appropriate function.
      */
      mono_ok = polyxedit_findmass_opt_get_mass_value (window,
							 PXM_MASS_MONO,
							 NULL);

      avg_ok = polyxedit_findmass_opt_get_mass_value (window,
							PXM_MASS_AVG,
							NULL);

      /* If both entries did contain valid string, that is an error,
	 because we do not know which one to use for the unique mode
	 mass find.
      */
      if (mono_ok == TRUE && avg_ok == TRUE)
	{
	 polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	     _("Both MONO and AVG text entries have a mass value"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	}
      else if (mono_ok == TRUE)
	{
	  /* We have a valid string in the MONO mass entry, which means
	     that we simply get the pointer to the entry in question
	     and call the proper function.
	  */
	  entry = g_object_get_data (G_OBJECT (window), 
				     "mono_unique_mass_entry");
	  g_assert (entry != NULL);
	  
	  polyxedit_findmass_opt_mass_entry_activate (GTK_ENTRY (entry),
							window);

	  return;
	}
      else /* (avg_ok == TRUE) */
	{
	  /* We have a valid string in the AVG mass entry, which means
	     that we simply get the pointer to the entry in question
	     and call the proper function.
	  */
	  entry = g_object_get_data (G_OBJECT (window), 
				     "avg_unique_mass_entry");
	  g_assert (entry != NULL);
	  
	  polyxedit_findmass_opt_mass_entry_activate (GTK_ENTRY (entry),
							window);
	  return;
	}
    }
  else /* (FALSE == unique_mode) */
    {
      /* At this point we know that we have to deal with both MONO and
	 AVG textviews, get their contents, in sequence and parse
	 these contents into masses, which are in turn changed into
	 PxmFindmassOpt instances.
      */


      /* During the process of parsing the contents of the MONO/AVG
	 textview widgets, we'll allocate a number of PxmFindmassopt
	 objects (fmopt, for short). Inside one round of parsing (MONO
	 first and AVG next), all these objects will be identical
	 unless for the mass value.

	 So we allocate a template fmopt that we'll later use for the
	 duplications that we'll perform each time a valid mass was 
	 parsed. We'll thus only modify the mass in the newly allocated
	 fmopt.
      */
      fmopt_template = pxmchem_findmassopt_new ();

      /* For the parsing of the masses in the textview we'll use a 
	 GPtrArray of gdoubles, that we allocate right away.
      */
      massesGPA = g_ptr_array_new ();
      
      /* Now, for each mass in the array, we will need to prepare a
	 fmopt object that we'll add to the fmoptGPA that we can
	 allocate now.
      */
      fmoptGPA = g_ptr_array_new ();


      /* Sequentially study the contents of the MONO and AVG masses'
	 textviews:
       */
      while (mass_types [iter] != PXM_MASS_NONE)
	{
	  g_assert (mass_types [iter] == PXM_MASS_MONO 
		    || mass_types [iter] == PXM_MASS_AVG);
	  
	  fmopt_template->mass_type = mass_types [iter];


	  /* We start by checking if the textview has valid text in
	     it.  This is because if it has no text in it it is of no
	     interest to go further for the current mass type : just 
	     increment iter and go to the other mass type if there is 
	     one still available.
	  */
	  /* Now deal with the textview proper.
	   */
	  if (mass_types [iter] == PXM_MASS_MONO)
	    textview = g_object_get_data (G_OBJECT (window), 
					  "mono_masses_textview");
	  else
	    textview = g_object_get_data (G_OBJECT (window), 
					  "avg_masses_textview");
	  
	  g_assert (textview);
	  
	  buffer = gtk_text_view_get_buffer (textview);
	  
	  /* We now know where the text to be read is and where it should
	     be put.
	  */
	  gtk_text_buffer_get_start_iter (buffer, &start);
	  gtk_text_buffer_get_end_iter (buffer, &end);
	  
	  /* masses_text is going to hold an allocated UTF-8 string.
	   */
	  masses_text = gtk_text_buffer_get_text (buffer,
						  &start,
						  &end,
						  FALSE);
	  g_assert (masses_text != NULL);
	  
	  if (FALSE == g_utf8_validate (masses_text, -1, NULL))
	    {
	      polyxmass_timeoutmsg_message_set
		((GtkWindow *) window,
		 _("The data in the text view are not valid UTF-8"),
		 POLYXMASS_NORM_MSG_TIMEOUT);
	      
	      /* Here it makes sense to continue even if we do not have
		 masses for this round, only if in the previous round
		 (if this one is the second) there were masses in the
		 array of fmopts:
	      */
	      if (fmoptGPA->len > 0)
		break; /* exit the while loop */
	      else
		{
		  g_free (masses_text);
		  
		  pxmchem_findmassopt_free (fmopt_template);
		  g_ptr_array_free (massesGPA, TRUE);
		  g_ptr_array_free (fmoptGPA, TRUE);
		  
		  return;
		}
	    }
	  
	  if (g_utf8_strlen (masses_text, -1) <= 0)
	    {
	      /* Here it makes sense to continue even if we do not have
		 masses for this round, only if in the previous round
		 (if this one is the second) there were masses in the
		 array of fmopts:
	      */
	      if (fmoptGPA->len > 0)
		break; /* exit the while loop */
	      else
		{
		  g_free (masses_text);
		  
		  pxmchem_findmassopt_free (fmopt_template);
		  g_ptr_array_free (massesGPA, TRUE);
		  g_ptr_array_free (fmoptGPA, TRUE);
		  
		  return;
		}
	    }
	  
	  /* At this point we know that there are masses to be found
	     which means that it is useful to continue getting data 
	     from the other widgets.
	  */
	  
	  /* Now get to have a properly converted tolerance value from the
	     corresponding entry:
	  */
	  if (FALSE == 
	      polyxedit_findmass_opt_get_tolerance_value (window,
							  mass_types [iter],
							  fmopt_template))
	    {
	      polyxmass_timeoutmsg_message_set
		((GtkWindow *) window,
		 _("Failed to get a correct tolerance value"),
		 POLYXMASS_NORM_MSG_TIMEOUT);
	     
	      /* Only return, however, if there are no fmopts in the
		 fmoptGPA:
	      */
	      if (fmoptGPA->len > 0)
		break; /* exit the while loop */
	      else
		{
		  g_free (masses_text);
		  
		  pxmchem_findmassopt_free (fmopt_template);
		  g_ptr_array_free (massesGPA, TRUE);
		  g_ptr_array_free (fmoptGPA, TRUE);
		  
		  return;
		}
	    }

	  /* Now get the tolerance type from the proper GtkOptionMenu widget.
	   */
	  if (FALSE == 
	      polyxedit_findmass_opt_get_tolerance_type (window,
							 mass_types [iter],
							 fmopt_template))
	    {
	      polyxmass_timeoutmsg_message_set
		((GtkWindow *) window,
		 _("Failed to get a correct tolerance type"),
		 POLYXMASS_NORM_MSG_TIMEOUT);

	      /* Only return, however, if there are no fmopts in the
		 fmoptGPA:
	      */
	      if (fmoptGPA->len > 0)
		break; /* exit the while loop */
	      else
		{
		  g_free (masses_text);
		  
		  pxmchem_findmassopt_free (fmopt_template);
		  g_ptr_array_free (massesGPA, TRUE);
		  g_ptr_array_free (fmoptGPA, TRUE);
		  
		  return;
		}
	    }


	  /* We STILL have to parse the text in 'masses_text' and for each
	     successfully parsed mass we get a gdouble value that is
	     allocated and added to the 'massesGPA'.
	  */
	  libpolyxmass_globals_parse_masses_in_text (masses_text, massesGPA);
	  
	  if (massesGPA->len <= 0)
	    {
	      /* No masses in the array and since we allocated it here, 
		 and it was empty before the parsing, that means that no
		 mass could successfully be parsed in it.
	      */
	      /* Only return, however, if there are no fmopts in the
		 fmoptGPA:
	      */
	      if (fmoptGPA->len > 0)
		break; /* exit the while loop */
	      else
		{
		  g_free (masses_text);
		  
		  pxmchem_findmassopt_free (fmopt_template);
		  g_ptr_array_free (massesGPA, TRUE);
		  g_ptr_array_free (fmoptGPA, TRUE);
		  
		  return;
		}
	    }
	  
	  /* Now that we have our array of masses (gdouble ones), we 
	     can iterate in it and ensure that for each mass we create
	     a fmopt object that gets added to fmoptGPA. In the process
	     we free all the gdouble masses of the array, that in turn
	     will be freed...
	  */
	  while (massesGPA->len > 0)
	    {
	      mass = g_ptr_array_remove_index (massesGPA, 0);
	      g_assert (mass != NULL);
	      
	      fmopt = pxmchem_findmassopt_dup (fmopt_template,
					       PXM_FINDMASSOPT_DUP_DEEP_YES);
	      
	      fmopt->mass = *mass;
	      
	      g_ptr_array_add (fmoptGPA, fmopt);
	      
	      g_free (mass);
	    }
	  
	  /* We can free the text that was fetched from the textview.
	   */
	  g_free (masses_text);
	  
	  /* At this point, all the masses are freed and an according number
	     of fmopt objects have been allocated into fmoptGPA. We thus 
	     are now ready to either go proceed to the other mass type,
	     or to perform the mass find proper.
	  */
	  iter++;
	}
      /* end of
	 while (mass_types [iter] != PXM_MASS_NONE)
      */

      /* At this point, we have an array of fmopt objects that reflect
	 the masses that were listed in both the textview widgets:
	 MONO and AVG.
      */

      /* We do not need the template fmopt anymore.
       */
      pxmchem_findmassopt_free (fmopt_template);

      /* Since we had emptied the array of masses at each one of the
	 two above rounds, we can free it, because it is empty by now.
      */
      g_ptr_array_free (massesGPA, TRUE);


      /* Allocate the array of arrays of oligomers that will hold all
	 the arrays of oligomers organized by found masses.
      */
      alloligsGPA = g_ptr_array_new ();
      

      /* Perform the finding operation proper.
       */
      count = polyxedit_findmass_opt_find_masses (window,
						  haystackGPA,
						  alloligsGPA,
						  fmoptGPA);
      if (count == -1)
	{
	  polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	     _("Failed to find the masses in the polymer"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_findmassopt_GPA_free (fmoptGPA);
	  
	  return;
	}
      else if (count == 0)
	{
	  polyxmass_timeoutmsg_message_set
	((GtkWindow *) window,
	     _("No mass was found in the polymer"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  pxmchem_findmassopt_GPA_free (fmoptGPA);
	  
	  return;
	}

      /* At this point, we do not need the array of fmopt instances,
	 because each oligomer in the arrays that sit in the
	 alloligsGPA has a copy of the fmopt object by which it was
	 found.
      */
      pxmchem_findmassopt_GPA_free (fmoptGPA);
	  
      /* At this point we just have to display the found oligomers. We
	 pass to the found oligomers-displaying window the pointer to
	 _this_ window.
       */
      display_window = polyxedit_findmass_res_wnd_setup (window,
							 alloligsGPA);

      if (display_window == NULL)
	{
	  polyxmass_timeoutmsg_message_set
	    ((GtkWindow *) window,
	     _("Failed to display the oligomers in a window"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);
	  
	  return;
	}

      /* Register the results' window !
       */

      /* Get the window description string from the window where the
	 cleavage/fragmentation results were displayed. We will use
	 this string to make sure the user will know exactly from
	 which data the find mass results were extracted !
      */
      parent_wnd = g_object_get_data (G_OBJECT (window), "grand_parent");
      g_assert (parent_wnd != NULL);
      
      editctxt = g_object_get_data (G_OBJECT (parent_wnd), "editctxt");
      g_assert (editctxt != NULL);
      
      wnd_desc = g_object_get_data (G_OBJECT (parent_wnd), "wnd_desc");
      g_assert (wnd_desc != NULL);
      
      help = g_strdup_printf (_("Find Mass For Results in Window ID '%p'"
			      " ('%s')"), 
			      parent_wnd, wnd_desc);
      
      g_object_set_data_full (G_OBJECT (display_window),
			      "wnd_desc", help, 
			      (GDestroyNotify) free);

      if (NULL == polyxmass_winmngmt_register_window 
	  ("POLYXEDIT",
	   display_window,
	   window,
	   editctxt,
	   _("Results of Find Mass"),
	   help,
	   TRUE,
	   polyxedit_findmass_results_wnd_make_report))
	g_critical (_("%s@%d: failed to register the "
		      "find mass results' window\n"),
		    __FILE__, __LINE__);
      
      /* Do not free help, because we have set it as a full datum to
	 the window where the find mass results are to be displayed !
	 
	 g_free (help);
      */
      
      /* Finally we can return, our work here is finished.
       */
      return;
    }
  /* end of 
     else (FALSE == unique_mode)
  */

  return;
}
 

 
gboolean
polyxedit_findmass_opt_get_mass_value (GtkWidget *window, 
					 PxmMassType type,
					 PxmFindmassOpt *fmopt)
{
  gdouble mass;
  
  gboolean result = FALSE;

  gchar *help = NULL;

  GtkWidget *entry = NULL;
  
  
  /* Depending on the value of the 'type' parameter, we have to get
     the proper mass text value from the proper text entry. So,
     first get text, and next put the successfully converted value
     into the fmopt structure. Note that if fmopt is
     NULL, this function is called only to check that the entry
     contains text that can successfully be converted into gdouble.
  */

  g_assert (window != NULL);

  if (type == PXM_MASS_MONO)
    {
      entry = g_object_get_data (G_OBJECT (window),
				 "mono_unique_mass_entry");
      g_assert (entry != NULL);
    }
  else /* (type == PXM_MASS_AVG) */
    {
      entry = g_object_get_data (G_OBJECT (window),
				 "avg_unique_mass_entry");
      g_assert (entry != NULL);
    }
  
  /* Get the text value and next convert it to a gdouble value.
   */
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  
  result = libpolyxmass_globals_strtod (help, &mass);
  
  g_free (help);

  /* At this point we have successfully converted the text value  
     into a numerical value, so we can set this to the 
     fmopt param if non-NULL.
  */
  if (fmopt != NULL)
    fmopt->mass = mass;
  
  return result;
}


gboolean
polyxedit_findmass_opt_get_tolerance_value (GtkWidget *window, 
					      PxmMassType type,
					      PxmFindmassOpt *fmopt)
{
  gdouble tolerance;

  gboolean result = FALSE;
  
  gchar *help = NULL;

  GtkWidget *entry = NULL;
  
  
  /* Depending on the value of the 'type' parameter, we have to get
     the proper tolerance text value from the proper text entry. So,
     first get text, and next put the successfully converted value
     into the fmopt structure. Note that if fmopt is NULL, this
     function is called only to check that the entry contains text
     that can successfully be converted into gdouble.
  */

  g_assert (window != NULL);

  if (type == PXM_MASS_MONO)
    {
      entry = g_object_get_data (G_OBJECT (window),
				 "mono_tolerance_entry");
      g_assert (entry != NULL);
    }
  else /* (type == PXM_MASS_AVG) */
    {
      entry = g_object_get_data (G_OBJECT (window),
				 "avg_tolerance_entry");
      g_assert (entry != NULL);
    }
  
  /* Get the text value and next convert it to a gdouble value.
   */
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  
  /* Note that if help is empty, then we set it to 0 (no tolerance, which
     means that the mass find is EXACT.
  */
  if (strlen (help) <= 0)
    {
      tolerance = 0;
      result = TRUE;
    }
  else
    result = libpolyxmass_globals_strtod (help, &tolerance);
  
  g_free (help);
    
  /* At this point we have successfully converted the text value into
     a numerical value, so we can set this to the fmopt param if
     non-NULL.
  */
  if (fmopt != NULL)
    fmopt->tolerance = tolerance;
  
  return result;
}


gboolean
polyxedit_findmass_opt_get_tolerance_type (GtkWidget *window, 
					     PxmMassType type,
					     PxmFindmassOpt *fmopt)
{
  GtkWidget *combo = NULL;
  
  gint index = -1;
  
  /* Depending on the value of the 'type' parameter, we have to get
     the proper tolerance type from the proper optionmenu.
  */
  
  g_assert (window != NULL);

  if (type == PXM_MASS_MONO)
    {
      combo = g_object_get_data (G_OBJECT (window),
				 "mono_tolerance_combo");
      g_assert (combo != NULL);
    }
  else /* (type == PXM_MASS_AVG) */
    {
      combo = g_object_get_data (G_OBJECT (window),
				 "avg_tolerance_combo");
      g_assert (combo != NULL);
    }

  /* Get the index of the currently active item in the combo.  We will
     connect that index to the relative tolerance_unit_xxx that was
     appended to the GtkComboBox when it was setup (see above).
   */
  index = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  
  switch (index)
    {
    case 0:
      if (fmopt != NULL)
	fmopt->tolerance_type = PXM_MASS_TOLERANCE_AMU;
      return TRUE;
    case 1:
      if (fmopt != NULL)
	fmopt->tolerance_type = PXM_MASS_TOLERANCE_PCT;
      return TRUE;
    case 2:
      if (fmopt != NULL)
	fmopt->tolerance_type = PXM_MASS_TOLERANCE_PPM;
      return TRUE;
    }
  
  /* If we did not return TRUE by now, that means that the index was
     incorrect... something erroneous has happened.
  */
  return FALSE; 
}


/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_findmass_opt_wnd_really_close (GtkWidget *window)
{
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);


  gtk_widget_destroy (window);
}

  
gboolean
polyxedit_findmass_opt_wnd_delete_event (GtkWidget *window, 
					 GdkEventAny *event, 
					 gpointer data)
{
  gchar *help = NULL;

  GtkWidget *parent_wnd = NULL;
  
  g_assert (window != NULL);

  /* See top for the explanation of why we name that datum
     "grand_parent" and not "parent".
  */
  parent_wnd = g_object_get_data (G_OBJECT (window), "grand_parent");
  g_assert (parent_wnd != NULL);
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);

  /* This window pointer was set as a full datum to the parent window
     (a cleavage or fragmentatioon results window), which means that
     we have to remove that pointer, without triggering the callback
     function call.
  */
  help = g_strdup_printf ("findmass_options_wnd-%p", window);
  
  window = 
    g_object_steal_data (G_OBJECT (parent_wnd), help);
  
  g_free (help);
  
  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}

  
gboolean
polyxedit_findmass_opt_wnd_destroy_event (GtkWidget *window, 
					 GdkEventAny *event, 
					 gpointer data)
{
  return FALSE;
}
