/*
** pork_cstr.c - routines for dealing with strings of type chtype *.
** Copyright (C) 2002-2003 Ryan McCabe <ryan@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
*/

#include <config.h>

#include <unistd.h>
#include <ncurses.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>

#include <pork.h>
#include <pork_missing.h>
#include <pork_util.h>
#include <pork_color.h>
#include <pork_cstr.h>

/*
** I define what I call "cstrings" for
** use in this program.
**
** ncurses uses a data type called "chtype" to hold
** a character with attribute set (f.e. having color, being bold,
** underlined, etc).
**
** I define an array of these characters that's terminated with a chtype
** character having the value of zero as a "cstr"
*/

/*
** Return the length of the cstring pointed
** by "ch"
*/

inline size_t cstrlen(chtype *ch) {
	size_t i = 0;

	while (*ch++ != 0)
		++i;

	return (i);
}

/*
** Convert the cstring specified by "cstr"
** to its ASCII representation. This will result
** in the loss of all attributes of the characters
** that comprise it.
*/

char *cstr_to_plaintext(chtype *cstr, size_t len) {
	char *str = xmalloc(len + 1);
	size_t i;

	for (i = 0 ; i < len && cstr[i] != 0 ; i++)
		str[i] = chtype_get(cstr[i]);

	str[i] = '\0';
	return (str);
}

/*
** The reverse of above. The characters that comprise the cstring to
** be constructed will have their attributes set according to a format
** string starting with % and then containing color information in the
** form <foreground>[,<background>]. See the function color_parse_code() for
** more information.
*/

int plaintext_to_cstr(const char *str, chtype *ch, size_t len) {
	u_int32_t i;
	u_int32_t spos = 0;
	attr_t color_attr = 0;

	len--;

	for (i = 0 ; i < len && str[spos] != '\0' ; i++) {
		if (str[spos] == '%') {
			if (str[spos + 1] != '\0' && str[++spos] != '%') {
				int ret = color_parse_code(&str[spos], &color_attr);
				if (ret == -1)
					ch[i] = str[--spos];
				else {
					i--;
					spos += ret;
					continue;
				}
			} else
				ch[i] = str[spos];
		} else if (str[spos] == '\t') {
			size_t pad = PORK_TABSTOP - i % PORK_TABSTOP;
			size_t j;

			for (j = 0 ; j < pad && i < len ; j++)
				ch[i++] = ' ' | color_attr;
			i--;
		} else
			ch[i] = str[spos];

		ch[i] |= color_attr;

		spos++;
	}

	ch[i] = 0;
	return (i);
}

/*
** The same as above, only the string is translated literally; no color
** attributes are allowed.
*/

int plaintext_to_cstr_nocolor(const char *str, chtype *ch, size_t len) {
	u_int32_t i;

	len--;
	for (i = 0 ; i < len && str[i] != '\0' ; i++)
		ch[i] = str[i];

	ch[i] = 0;
	return (i);
}

/*
** Write the cstring pointed to by "ch"
** to the screen at the current cursor position.
*/

inline size_t wputstr(WINDOW *win, chtype *ch) {
	size_t i = 0;

	while (ch[i] != 0)
		waddch(win, ch[i++]);

	return (i);
}

/*
** Write the cstring pointed to by "ch"
** to the screen at the position (x, y).
*/

inline size_t mvwputstr(WINDOW *win, int y, int x, chtype *ch) {
	wmove(win, y, x);

	return (wputstr(win, ch));
}

/*
** Write the first n chtype chars of the cstring pointed to by "ch"
** to the screen at the current cursor position.
*/

inline size_t wputnstr(WINDOW *win, chtype *ch, size_t n) {
	size_t i;

	for (i = 0 ; i < n && ch[i] != 0 ; i++)
		waddch(win, ch[i]);

	return (i);
}

/*
** Write the first n chtype chars of the cstring pointed to by "ch"
** to the screen at position (x, y).
*/

inline size_t mvwputnstr(WINDOW *win, int y, int x, chtype *ch, size_t n) {
	wmove(win, y, x);

	return (wputnstr(win, ch, n));
}
