/*
 * Copyright 2013 Canonical Ltd.
 *
 * Authors:
 * Michael Frey: michael.frey@canonical.com
 * Matthew Fischer: matthew.fischer@canonical.com
 * Seth Forshee: seth.forshee@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>

#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <syslog.h>

#include <glib-object.h>
#include <gio/gio.h>
#include "powerd-internal.h"
#include "powerd-object.h"
#include "powerd-dbus.h"
#include "powerd-sensors.h"
#include "device-config.h"
#include "log.h"

#include <hybris/properties/properties.h>

#include "libsuspend.h"

#include <android/hardware/power.h>

static GThread *powerd_mainloop_thread;

namespace
{
static GMainLoop *main_loop = NULL;
static guint name_id;

static int g_exit_code = 0;

static int call_added = 0;

static uuid_t screen_cookie;

/* Used to track modem interfaces */
struct modem_data {
    char *obj_name;
    GDBusProxy *message_proxy;
    GDBusProxy *voice_proxy;
    GDBusProxy *ussd_proxy;
    GDBusProxy *radiosettings_proxy;
};

static void free_modem_data(void *data)
{
    struct modem_data *modem = (struct modem_data *) data;
    g_free(modem->obj_name);
    if (modem->message_proxy)
        g_object_unref(modem->message_proxy);
    if (modem->voice_proxy)
        g_object_unref(modem->voice_proxy);
    if (modem->ussd_proxy)
        g_object_unref(modem->ussd_proxy);
    if (modem->radiosettings_proxy)
        g_object_unref(modem->radiosettings_proxy);
}

static gint modem_data_cmp(gconstpointer a, gconstpointer b)
{
    const struct modem_data *modem = (const struct modem_data *) a;
    const char *modem_name = (const char *) b;

    return strcmp(modem->obj_name, modem_name);
}

/* List with detected modems */
static GSList *g_modems = NULL;

static GDBusProxy *g_ofono_proxy;
static GDBusProxy *g_unity_proxy;

static struct power_module* _power_module;

static void
sigterm_quit(int signal)
{
    powerd_warn("SIGTERM recieved, cleaning up");
    powerd_exit(0);
}


}   //namespace

void powerd_hal_signal_activity(void)
{
    powerd_warn("signalling activity via HAL");
    if (_power_module && _power_module->powerHint)
        _power_module->powerHint(_power_module, POWER_HINT_INTERACTION, NULL);
}

/*
 * Once dbus is ready, force the screen on to make the system state
 * match the powerd iternal state
 */
void powerd_dbus_init_complete(void)
{
}

static void watch_modem(struct modem_data *modem)
{
    /* for incoming SMS signals */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
        G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
        NULL,
        "org.ofono",
        modem->obj_name,
        "org.ofono.MessageManager",
        NULL,
        (GAsyncReadyCallback)ofono_proxy_connect_cb,
        &modem->message_proxy);

    /* for incoming calls Added/Removed signals */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
        G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
        NULL,
        "org.ofono",
        modem->obj_name,
        "org.ofono.VoiceCallManager",
        NULL,
        (GAsyncReadyCallback)ofono_proxy_connect_cb,
        &modem->voice_proxy);

    /* for USSD notifications/network requests */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
        G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
        NULL,
        "org.ofono",
        modem->obj_name,
        "org.ofono.SupplementaryServices",
        NULL,
        (GAsyncReadyCallback)ofono_proxy_connect_cb,
        &modem->ussd_proxy);

    /* To set/unset low power mode */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
        G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
        NULL,
        "org.ofono",
        modem->obj_name,
        "org.ofono.RadioSettings",
        NULL,
        (GAsyncReadyCallback)ofono_proxy_connect_cb,
        &modem->radiosettings_proxy);
}

void
ofono_get_modems_cb(GObject *source_object,
                    GAsyncResult *res,
                    gpointer user_data)
{
    GDBusProxy *client = G_DBUS_PROXY(source_object);
    GVariant *result;
    GVariant *item;
    GError *error = NULL;
    GVariantIter *iter;

    result = g_dbus_proxy_call_finish(client, res, &error);
    if (result == NULL) {
        powerd_warn("%s: call error", __func__);
        return;
    }

    g_variant_get(result, "(a(oa{sv}))", &iter);
    while ((item = g_variant_iter_next_value(iter))) {
        const char *obj_path;

        g_variant_get_child(item, 0, "&o", &obj_path);

        if (g_slist_find_custom(g_modems, obj_path, modem_data_cmp)
                == NULL) {
            struct modem_data *modem;

            powerd_debug("active ofono modem %s", obj_path);
            modem = (struct modem_data *) calloc(1, sizeof(*modem));
            modem->obj_name = g_strdup(obj_path);
            g_modems = g_slist_prepend(g_modems, modem);
            watch_modem(modem);
        }

        g_variant_unref(item);
    }

    g_variant_unref(result);
}

void on_ofono_manager_signal(GDBusProxy *proxy, gchar *sender_name,
    gchar *signal_name, GVariant *parameters, gpointer user_data)
{
    const gchar *object_path;
    GVariant *tmp;
    GSList *node;

    tmp = g_variant_get_child_value(parameters, 0);

    object_path = g_variant_get_string(tmp, NULL);
    node = g_slist_find_custom(g_modems, object_path, modem_data_cmp);

    if (strcmp("ModemAdded", signal_name) == 0) {
        /* Add if not already in list */
        if (node == NULL) {
            struct modem_data *modem;

            powerd_debug("watching ofono modem %s", object_path);
            modem = (struct modem_data *) calloc(1, sizeof(*modem));
            modem->obj_name = g_strdup(object_path);
            g_modems = g_slist_prepend(g_modems, modem);
            watch_modem(modem);
        }
    } else if(strcmp("ModemRemoved", signal_name) == 0) {
        /* Remove if found in list */
        if (node != NULL) {
            powerd_debug("stop watch on ofono modem %s", object_path);
            free_modem_data(g_modems->data);
            g_modems = g_slist_delete_link(g_modems, node);
        }
    }

    g_variant_unref(tmp);
}

void
on_ofono_voicecallmanager_signal (GDBusProxy *proxy,
           gchar      *sender_name,
           gchar      *signal_name,
           GVariant   *parameters,
           gpointer    user_data)
{
    if (!strcmp(signal_name, "CallAdded") && call_added++ <= 0) {
        turn_display_on(TRUE, UNITY_SCREEN_REASON_NORMAL);

        powerd_info("incoming call");

    } else if (!strcmp(signal_name, "CallRemoved")) {
        call_added--;

        //Turn the display back on when all calls are over
        if (call_added == 0)
            turn_display_on(TRUE, UNITY_SCREEN_REASON_NORMAL);

        powerd_info("call removed");        
    }
}

void
on_ofono_messagemanager_signal (GDBusProxy *proxy,
           gchar      *sender_name,
           gchar      *signal_name,
           GVariant   *parameters,
           gpointer    user_data)
{
    powerd_debug("we get signal from %s: %s", sender_name, signal_name);
    /* from org.ofono.MessageManager */
    if (((!strcmp(signal_name, "IncomingMessage")) ||
         (!strcmp(signal_name, "ImmediateMessage"))) &&
        (call_added == 0)) {
            powerd_debug("Waking up the device - Incoming SMS");
            turn_display_on(TRUE, UNITY_SCREEN_REASON_NOTIFICATION);
    }
}

void on_ofono_ussd_signal(GDBusProxy *proxy,
           gchar      *sender_name,
           gchar      *signal_name,
           GVariant   *parameters,
           gpointer    user_data)
{
    powerd_debug("we get signal from %s: %s", sender_name, signal_name);
    /* from org.ofono.MessageManager */
    if (((!strcmp(signal_name, "NotificationReceived")) ||
         (!strcmp(signal_name, "RequestReceived"))) && 
        (call_added == 0)) {
            powerd_debug("Waking up the device - Incoming USSD");
            turn_display_on(TRUE, UNITY_SCREEN_REASON_NORMAL);
    }
}

void on_unity_signal(GDBusProxy *proxy, gchar *sender_name,
    gchar *signal_name, GVariant *parameters, gpointer user_data)
{
    powerd_debug("we get signal from %s: %s", sender_name, signal_name);

    /* From com.canonical.Unity.Screen */
    if (strcmp(signal_name, "DisplayPowerStateChange") == 0) {
        int state, reason;
        GSList *it;

        g_variant_get(parameters, "(ii)", &state, &reason);
        powerd_debug("Received %s: state=%d flags=%d\n",
		     signal_name, state, reason);

        /* Set modem low power mode when reason is not proximity sensor */
        if (reason == UNITY_SCREEN_REASON_PROXIMITY)
            return;

        for (it = g_modems; it; it = it->next) {
            struct modem_data *modem = (struct modem_data *) it->data;
            GVariant *mode = g_variant_new_boolean(state ? FALSE : TRUE);

            g_dbus_proxy_call(modem->radiosettings_proxy, "SetProperty",
                              g_variant_new("(sv)", "FastDormancy", mode),
                              G_DBUS_CALL_FLAGS_NONE, -1, NULL, NULL, NULL);
        }
    }
}

void
unity_screen_name_appeared_cb(GDBusConnection *connection,
                              const gchar *name,
                              gpointer user_data)
{
    if (!uuid_is_null(screen_cookie)) {
        clear_sys_state_internal(screen_cookie);
        uuid_clear(screen_cookie);
    }
}

void
unity_screen_name_vanished_cb(GDBusConnection *connection,
                              const gchar *name,
                              gpointer user_data)
{
    if (uuid_is_null(screen_cookie)) {
        request_sys_state_internal("shutdown-request",
                                  POWERD_SYS_STATE_ACTIVE,
                                  &screen_cookie, NULL);
    }
}

/* Must be first to run on main loop */
static gboolean main_init(gpointer unused)
{
    powerd_mainloop_thread = g_thread_self();
    return FALSE;
}

/* Returns TRUE if current thread of execution is the default main loop */
int powerd_is_mainloop(void)
{
    return (g_thread_self() == powerd_mainloop_thread);
}

/* If running in emulator, make sure we keep the display on and
 * block suspend (can easily corrupt the data, not properly supported) */
static gboolean check_emulator(gpointer unused)
{
    char propbuf[PROP_VALUE_MAX];
    static uuid_t emulator_request_cookie;

    if (property_get("ro.kernel.qemu", propbuf, NULL) > 0) {
        powerd_info("Running in emulator, forcing display, blocking suspend.");
        keep_display_on();
        /* Block suspend */
        request_sys_state_internal("emulator-request",
                                  POWERD_SYS_STATE_ACTIVE,
                                  &emulator_request_cookie, NULL);
    }

    return FALSE;
}

void powerd_shutdown(void)
{
    static char poweroff_cmd[] = "poweroff";
    char *argv[] = {poweroff_cmd, NULL};
    g_spawn_async(NULL, argv, NULL, G_SPAWN_SEARCH_PATH, NULL, NULL, NULL, NULL);
}

void powerd_exit(int exit_code)
{
    g_exit_code = exit_code;
    g_main_loop_quit(main_loop);
}

int main(int argc, char** argv)
{
    powerd_log_init();

    powerd_info("Initializing powerd.");

    name_id = g_bus_own_name(G_BUS_TYPE_SYSTEM, "com.canonical.powerd",
        G_BUS_NAME_OWNER_FLAGS_REPLACE, powerd_bus_acquired_cb,
        powerd_name_acquired_cb, powerd_name_lost_cb, NULL, NULL);
    powerd_debug("owner id: %u", name_id);

    /* Listen to modem creation */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
                G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
                NULL,
                "org.ofono",
                "/",
                "org.ofono.Manager",
                NULL,
                (GAsyncReadyCallback) ofono_manager_proxy_connect_cb,
                &g_ofono_proxy);

    /* Listen to unity signals */
    g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
                G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
                NULL,
                "com.canonical.Unity.Screen",
                "/com/canonical/Unity/Screen",
                "com.canonical.Unity.Screen",
                NULL,
                (GAsyncReadyCallback) unity_proxy_connect_cb,
                &g_unity_proxy);

    /* Initialize screen cookie before setting up listener */
    uuid_clear(screen_cookie);

    /* Register for com.canonical.Unity.Screen name ownership changes */
    g_bus_watch_name(G_BUS_TYPE_SYSTEM, "com.canonical.Unity.Screen",
            G_BUS_NAME_WATCHER_FLAGS_NONE,
            (GBusNameAppearedCallback)unity_screen_name_appeared_cb,
            (GBusNameVanishedCallback)unity_screen_name_vanished_cb,
            NULL, NULL);

    main_loop = g_main_loop_new (NULL, FALSE);
    signal(SIGTERM, sigterm_quit);

    /* Init this first, data is used by other inits */
    device_config_init();

    libsuspend_init(0);
    powerd_info("libsuspend: detect module: %s.", libsuspend_getname());
    powerd_stats_init();
    powerd_client_init();
    power_request_init();
    dbus_name_watch_init();
    powerd_backlight_init();
    powerd_autobrightness_init();
    powerd_sensors_init(main_loop);
    powerd_ps_init();
    wakeup_init();

    int err = hw_get_module(POWER_HARDWARE_MODULE_ID,
        (hw_module_t const**)&_power_module);

    if (!err)
        _power_module->init(_power_module);

    /* Config use should be done during init, okay to free now */
    device_config_deinit();

    /* This needs to be the first thing to run on the main loop */
    g_idle_add_full(G_PRIORITY_HIGH, main_init, NULL, NULL);
    g_idle_add(check_emulator, NULL);

    g_main_loop_run(main_loop);
    g_bus_unown_name(name_id);
    g_main_loop_unref(main_loop);

    wakeup_deinit();
    powerd_ps_deinit();
    dbus_name_watch_deinit();
    powerd_autobrightness_deinit();
    powerd_backlight_deinit();
    power_request_deinit();
    powerd_client_deinit();
    powerd_stats_deinit();
    g_slist_free_full(g_modems, free_modem_data);
    if (g_ofono_proxy)
        g_object_unref(g_ofono_proxy);
    if (g_unity_proxy)
        g_object_unref(g_unity_proxy);

    return g_exit_code;
}
