/* ----------------------------------------------------------------------------
 * init.c
 * funtions that prepare program workflow and initialization
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <linux/input.h>
#include <gtk/gtk.h>

#ifdef HAVE_LANGINFO_CODESET
#  include <langinfo.h>
#endif

#include <pbb.h>

#include "init.h"
#include "interface.h"
#include "taghandler.h"
#include "callbacks.h"
#include "support.h"
#include "gettext_macros.h"

extern volatile sig_atomic_t prgexit;

int
init_powerprefs (struct basedata *base)
{
	int rc;

	base->kbds[0].fd = -1;
	base->flags.shift = base->flags.ctrl = base->flags.alt = 0;
	base->flags.learn_activated = 0;
	base->flags.learn_in_progress = 0;
	base->learnwindow = NULL;
	base->profile = &base->ac;
	base->ac.policy = 0;
	base->battery.policy = 0;

	/* setup IPC system for clients */
	if ((rc = ipc_init ("PowerPrefs", LIBMODE_CLIENT, CLIENT_REGISTER)) != 0) {
		if (rc == E_NOSERVER)
			print_msg (PBB_ERR, _("Server message port not found. Server isn't running.\n"));
		else if (rc == E_REGISTER)
			print_msg (PBB_ERR, _("Sending registration to server failed.\n"));
		else if (rc == E_MSGPORT)
			print_msg (PBB_ERR, _("Can't create message port for client.\n"));
		return rc;
	}

	if ((base->taglist = (struct tagitem*) malloc (TAGBUFFERLEN)) == NULL) {
		print_msg (PBB_ERR, _("Out of memory.\n"));
		return E_NOMEM;
	}
	taglist_init (base->taglist);
	base->bufferlen = TAGBUFFERLEN;
	base->strptr = (char*) base->taglist + base->bufferlen;

	base->mainwindow = create_mainwindow();  /* create & display GUI */
	gtk_widget_show (base->mainwindow);
	base->learnwindow = create_learnwindow();
	gtk_window_set_transient_for (GTK_WINDOW(base->learnwindow),
	                              GTK_WINDOW(base->mainwindow));

	if ((rc = init_guicontrol (base)))
		return rc;

	if ((collect_kbd_devices(base)))
		base->flags.learn_activated = 1; /* gui_finetuning does the work */

	return 0;
}

void
exit_powerprefs(struct basedata *base)
{
	int n;
	ipc_exit();   /* clean up IPC subsystem for clients */

	if (base->taglist)
		free (base->taglist);
	if (base->GUIControl)
		free (base->GUIControl);

	for (n=0; n < MAXINPUTS; n++)   /* close all additional input channels */
		if (base->kbds[n].fd >= 0) {
			gtk_input_remove(base->kbds[n].gdktag);
			close (base->kbds[n].fd);
		} else
			break;

	if (base->learnwindow)
		gtk_widget_destroy (base->learnwindow);
}

/* --- Signal Handler --- */

void
signalhandler (int signum)
{
	prgexit = 1;            /* cleanup and exit the program */
}

int
install_sighandler ()
{
	struct sigaction sa = { {signalhandler}, {{0}}, SA_RESTART, 0 };

	/* install the signal handlers */
	if (!sigaction (SIGINT, &sa, NULL))
		if (!sigaction (SIGTERM, &sa, NULL))
			return 0;
	
	print_msg (PBB_ERR, _("Can't install signal handler\n"));
	return -1;
}

int
evaluate_args(int argc, char *argv[])
{
	struct option const long_options[] = {
		  {"help", no_argument, 0, ARG_HELP},
		  {"version", no_argument, 0, ARG_VERSION},
		  {NULL, 0, NULL, 0}
	};
	char *prgname;
	int c, err = 0;

#if defined(ENABLE_NLS) && defined(HAVE_LANGINFO_CODESET)
	bind_textdomain_codeset (GETTEXT_PACKAGE, nl_langinfo(CODESET));
#endif
	if((prgname = strrchr(argv[0],'/')) == NULL)
		prgname = argv[0];
	else prgname++;		/* ignore first slash*/
	argv[0] = prgname;

	while ((c = getopt_long (argc, argv, ARG_ALL, long_options, (int *) 0)) != EOF) {
		switch (c) {
			case ARG_VERSION:
				printf(_("%s, version %s"), PACKAGE, VERSION);
				printf(", (c) 2002-2005 Matthias Grimm\n");
				err = E_INFO;
				break;
			case ARG_HELP:
			default:
				printf(_("%s - Preferences client for pbbuttonsd, make some options easy accessable.\n"), prgname);
				printf(_("Usage: %s [OPTION]... \n"), prgname);
				printf (_("Options:\n"
					"   -%c, --help           display this help and exit\n"
					"   -%c, --version        display version information and exit\n"),
					ARG_HELP, ARG_VERSION);
				err = E_INFO;
				break;
		}
		if (err != 0)
			break; /* while */
	}
#ifdef ENABLE_NLS
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
	return err;
}

int
init_guicontrol (struct basedata *base)
{
	int count, n;

	struct GTItem GT[] = {
	  /* machine / general */
	  { TAG_IDENTITY,           "ms_itext_machine",       NULL, th_showtext,   GCF_RONLY},
	  { TAG_PMUDEVICE,          "ms_entry_pmudev",        NULL, th_textentry,  0},
	  { TAG_ADBDEVICE,          "ms_entry_adbdev",        NULL, th_textentry,  0},
	  { TAG_SYSINFO,            NULL,                     NULL, th_system,     GCF_RONLY},
	  /* machine / inputs */
	  { TAG_TPMODE,             "ms_radio_trackpad",      NULL, th_radiobutton,0},
	  { TAG_KBDMODE,            "ms_radio_keyboard",      NULL, th_radiobutton,0},
	  { TAG_TPMODEUPKEY,        "ms_entry_tpmodeupkey",   NULL, th_keyentry,   0},
	  { TAG_TPMODEUPMOD,        "ms_entry_tpmodeupkey",   NULL, th_modentry,   0},
	  { TAG_TPMODEDOWNKEY,      "ms_entry_tpmodednkey",   NULL, th_keyentry,   0},
	  { TAG_TPMODEDOWNMOD,      "ms_entry_tpmodednkey",   NULL, th_modentry,   0},
	  { TAG_NOTAPTYPING,        "ms_check_notaptyping",   NULL, th_checkbox,   0},
	  /* cdrom / general */
	  { TAG_CDROMDEVICE,        "dr_entry_cdromdev",      NULL, th_textentry,  0},
	  { TAG_EJECTCDKEYDELAY,    "dr_check_ejectcddelay",  NULL, th_checkbox,   0},
	  { TAG_EJECTCDKEY,         "dr_entry_ejectcdkey",    NULL, th_keyentry,   0},
	  { TAG_EJECTCDMOD,         "dr_entry_ejectcdkey",    NULL, th_modentry,   0},
	  /* audio / general */
	  { TAG_SOUNDSYSTEMCFG,     "au_radio_sound",         NULL, th_radiobutton,0},
	  { TAG_MIXERDEVICE,        "au_entry_ossdevice",     NULL, th_textentry,  0},
	  { TAG_MIXERCARD,          "au_entry_alsacard",      NULL, th_textentry,  0},
	  { TAG_MIXERCHANNELS,      "au_entry_osschannels",   NULL, th_textentry,  0},
	  { TAG_MIXERELEMENTS,      "au_entry_alsaelements",  NULL, th_textentry,  0},
	  { TAG_MIXERINITDELAY,     "au_check_delaymixer",    NULL, th_checkbox,   0},
	  { TAG_VOLUMEUPKEY,        "au_entry_volupkey",      NULL, th_keyentry,   0},
	  { TAG_VOLUMEUPMOD,        "au_entry_volupkey",      NULL, th_modentry,   0},
	  { TAG_VOLUMEDOWNKEY,      "au_entry_voldnkey",      NULL, th_keyentry,   0},
	  { TAG_VOLUMEDOWNMOD,      "au_entry_voldnkey",      NULL, th_modentry,   0},
	  { TAG_MUTEKEY,            "au_entry_mutekey",       NULL, th_keyentry,   0},
	  { TAG_MUTEMOD,            "au_entry_mutekey",       NULL, th_modentry,   0},
	  { TAG_SOUNDSYSTEM,        "au_label_soundsystem",   NULL, th_soundsystem,0},
	  /* video / LCD */
	  { TAG_LCDBRIGHTNESS,      "vi_itext_lcdbrightness", NULL, th_showvalue,  GCF_RONLY},
	  { TAG_LCDILLUMUPKEY,      "vi_entry_lcdbrightupkey",NULL, th_keyentry,   0},
	  { TAG_LCDILLUMUPMOD,      "vi_entry_lcdbrightupkey",NULL, th_modentry,   0},
	  { TAG_LCDILLUMDOWNKEY,    "vi_entry_lcdbrightdnkey",NULL, th_keyentry,   0},
	  { TAG_LCDILLUMDOWNMOD,    "vi_entry_lcdbrightdnkey",NULL, th_modentry,   0},
	  { TAG_LCDFADINGSPEED,     "vi_spin_lcdfadespeed",   NULL, th_spinbutton, 0},
	  { TAG_DIMFULLYDARK,       "vi_check_dimdark",       NULL, th_checkbox,   0},
	  /* video / keyboard */
	  { TAG_KBDBRIGHTNESS,      "vi_itext_kbdbrightness", NULL, th_showvalue,  GCF_RONLY},
	  { TAG_KBDILLUMUPKEY,      "vi_entry_kbdbrightupkey",NULL, th_keyentry,   0},
	  { TAG_KBDILLUMUPMOD,      "vi_entry_kbdbrightupkey",NULL, th_modentry,   0},
	  { TAG_KBDILLUMDOWNKEY,    "vi_entry_kbdbrightdnkey",NULL, th_keyentry,   0},
	  { TAG_KBDILLUMDOWNMOD,    "vi_entry_kbdbrightdnkey",NULL, th_modentry,   0},
	  { TAG_KBDILLUMONKEY,      "vi_entry_kbdbrightonkey",NULL, th_keyentry,   0},
	  { TAG_KBDILLUMONMOD,      "vi_entry_kbdbrightonkey",NULL, th_modentry,   0},
	  { TAG_KBDFADINGSPEED,     "vi_spin_kbdfadespeed",   NULL, th_spinbutton, 0},
	  { TAG_KBDONBRIGHTNESS,    "vi_spin_kbdstart",       NULL, th_spinbutton, 0},
	  /* video / autoadjustment */
	  { TAG_LCDAUTOADJUST,      "vi_check_lcdautoadjust", NULL, th_checkbox,   0},
	  { TAG_LCDTHRESHOLD,       "vi_spin_lcdthreshold",   NULL, th_spinbutton, 0}, 
	  { TAG_LCDAUTOADJMINBAT,   "vi_spin_aa_bat_low",     NULL, th_spinbutton, 0},
	  { TAG_LCDAUTOADJMAXBAT,   "vi_spin_aa_bat_high",    NULL, th_spinbutton, 0},
	  { TAG_LCDAUTOADJMINAC,    "vi_spin_aa_ac_low",      NULL, th_spinbutton, 0},
	  { TAG_LCDAUTOADJMAXAC,    "vi_spin_aa_ac_high",     NULL, th_spinbutton, 0},
	  { TAG_KBDAUTOADJUST,      "vi_check_kbdautoadjust", NULL, th_checkbox,   0},
	  { TAG_KBDTHRESHOLD,       "vi_spin_kbdthreshold",   NULL, th_spinbutton, 0}, 
	  /* video / options */
	  { TAG_FRAMEBUFFERDEVICE,  "vi_entry_fbdev",         NULL, th_textentry,  0},
	  { TAG_BLANKFRAMEBUFFER,   "vi_check_blankfb",       NULL, th_checkbox,   0},
	  { TAG_CRTMIRRORKEY,       "vi_entry_extvideokey",   NULL, th_keyentry,   0},
	  { TAG_CRTMIRRORMOD,       "vi_entry_extvideokey",   NULL, th_modentry,   0},
	  /* power / commands */
	  { TAG_SCRIPTPMCS,         "pv_entry_cmdchprofile",  NULL, th_textentry,  GCF_ROOT},
	  { TAG_TIMEFORCMD,         "pv_spin_cmdtimeout",     NULL, th_spinbutton, GCF_ROOT},
	  /* power / sleeplocks */
	  { TAG_CPULOADSLEEPLOCK,   "pv_check_cpulock",       NULL, th_checkbox,   0},
	  { TAG_CPULOADMIN,         "pv_spin_cpulockload",    NULL, th_spinbutton, 0},
	  { TAG_CPULOADPERIOD,      "pv_spin_cpulocksec",     NULL, th_spinbutton, 0},
	  { TAG_NETLOADSLEEPLOCK,   "pv_check_netlock",       NULL, th_checkbox,   0},
	  { TAG_NETLOADMIN,         "pv_spin_netlockload",    NULL, th_spinbutton, 0},
	  { TAG_NETLOADPERIOD,      "pv_spin_netlocksec",     NULL, th_spinbutton, 0},
	  { TAG_NETLOADDEV,         "pv_entry_netlockdev",    NULL, th_textentry,  0},
	  /* power / emergency action */
	  { TAG_EMERGENCYACTION,    "pv_radio_ema",           NULL, th_radiobutton,0},
	  { TAG_SLEEPKEYDELAY,      "pv_check_sleepdelay",    NULL, th_checkbox,   0},
	  { TAG_SLEEPKEY,           "pv_entry_sleepkey",      NULL, th_keyentry,   0},
	  { TAG_SLEEPMOD,           "pv_entry_sleepkey",      NULL, th_modentry,   0},
	  /* power / battery */
	  { TAG_CURRENTBWL,         "pv_itext_warnlevel",     NULL, th_showvalue,  GCF_RONLY},
	  { TAG_BWLFIRST,           "pv_spin_firstbwl",       NULL, th_spinbutton, 0},
	  { TAG_BWLSECOND,          "pv_spin_secondbwl",      NULL, th_spinbutton, 0},
	  { TAG_BWLLAST,            "pv_spin_lastbwl",        NULL, th_spinbutton, 0},
	  /* power / profiles */
	  { TAG_ONAC_POLICY,        "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONAC_TIMERACTION,   "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONAC_COVERACTION,   "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONAC_KEYACTION,     "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONAC_TIMESUSPEND,   "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONAC_TIMEDIM,       "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONBATT_POLICY,      "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONBATT_TIMERACTION, "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONBATT_COVERACTION, "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONBATT_KEYACTION,   "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONBATT_TIMESUSPEND, "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_ONBATT_TIMEDIM,     "pv_combo_profile",       NULL, th_profile,    0},
	  { TAG_HEARTBEATBEEP,      "pv_check_heartbeat",     NULL, th_checkbox,   0},
	  /* about */
	  { TAG_VERSION,            "ab_itext_sversion",      NULL, th_showtext,   GCF_RONLY},
	  /* misc operations through flags*/
	  { 0,                      "pv_func_cmdchprofile",   NULL, th_dummy,      GCF_ROOT},
	  { 0,                      "pv_func_cmdtimeout",     NULL, th_dummy,      GCF_ROOT},
	  /* learn buttons */
	  { 0,  "pv_learn_sleepkey",      "pv_entry_sleepkey",      th_dummy, GCF_LEARN},
	  { 0,  "vi_learn_lcdbrightupkey","vi_entry_lcdbrightupkey",th_dummy, GCF_LEARN},
	  { 0,  "vi_learn_lcdbrightdnkey","vi_entry_lcdbrightdnkey",th_dummy, GCF_LEARN},
	  { 0,  "vi_learn_kbdbrightupkey","vi_entry_kbdbrightupkey",th_dummy, GCF_LEARN},
	  { 0,  "vi_learn_kbdbrightdnkey","vi_entry_kbdbrightdnkey",th_dummy, GCF_LEARN},
	  { 0,  "vi_learn_kbdbrightonkey","vi_entry_kbdbrightonkey",th_dummy, GCF_LEARN},
	  { 0,  "vi_learn_extvideokey",   "vi_entry_extvideokey",   th_dummy, GCF_LEARN},
	  { 0,  "au_learn_volupkey",      "au_entry_volupkey",      th_dummy, GCF_LEARN},
	  { 0,  "au_learn_voldnkey",      "au_entry_voldnkey",      th_dummy, GCF_LEARN},
	  { 0,  "au_learn_mutekey",       "au_entry_mutekey",       th_dummy, GCF_LEARN},
	  { 0,  "dr_learn_ejectcdkey",    "dr_entry_ejectcdkey",    th_dummy, GCF_LEARN},
	  { 0,  "ms_learn_tpmodeupkey",   "ms_entry_tpmodeupkey",   th_dummy, GCF_LEARN},
	  { 0,  "ms_learn_tpmodednkey",   "ms_entry_tpmodednkey",   th_dummy, GCF_LEARN}
	};

	count = sizeof(GT) / sizeof(struct GTItem);
	if ((base->GUIControl = (struct GUIControlItem*) malloc (count*sizeof(struct GUIControlItem))) == NULL) {
		print_msg (PBB_ERR, _("Out of memory.\n"));
		return E_NOMEM;
	}

	base->GUIControlCount = count;
	for (n=0; n < count; n++) {
		base->GUIControl[n].tag     = GT[n].tag;

		if (GT[n].widgetname != NULL)
			base->GUIControl[n].widget  = lookup_widget(base->mainwindow, GT[n].widgetname);
		else
			base->GUIControl[n].widget = NULL;
		if (GT[n].linkedname != NULL)
			base->GUIControl[n].linked  = lookup_widget(base->mainwindow, GT[n].linkedname);
		else
			base->GUIControl[n].linked = NULL;

		base->GUIControl[n].handler = GT[n].handler;
		base->GUIControl[n].flags   = GT[n].flags;
	}
	return 0;
}

void
gui_finetuning (struct basedata *base)
{
	GtkWidget *widget;
	GSList *radiolist;
	gchar *text;
	int n;

	/* adjust GUI if user is authorized */
	if (geteuid() != 0) {
		for (n=0; n < base->GUIControlCount; n++)
			if (base->GUIControl[n].flags & GCF_ROOT)
				gtk_widget_set_sensitive (base->GUIControl[n].widget, FALSE);

		if ((widget = lookup_widget(base->mainwindow, "main_btn_save")))
			gtk_widget_set_sensitive (widget, FALSE);
			
		if ((widget = lookup_widget(base->mainwindow, "au_radio_sound"))) {
			radiolist = gtk_radio_button_get_group (GTK_RADIO_BUTTON(widget));
			while (radiolist) {
				gtk_widget_set_sensitive (GTK_WIDGET (radiolist->data), FALSE);
				radiolist = g_slist_next(radiolist);
			}
		}
	}

	/* make learn buttons sensitive if learn mode is activated */
	if (base->flags.learn_activated)
		for (n=0; n < base->GUIControlCount; n++)
			if (base->GUIControl[n].flags & GCF_LEARN) {
				gtk_widget_set_sensitive (base->GUIControl[n].widget, TRUE);
				g_signal_connect (base->GUIControl[n].widget, "toggled",
					G_CALLBACK (cb_start_learn), (gpointer) base);
			}

	/* update version label */
	if ((widget = lookup_widget(base->mainwindow, "ab_itext_version"))) {
		text = g_strdup_printf("PowerPrefs v%s", VERSION);
		gtk_label_set_text(GTK_LABEL(widget), text);
		g_free(text);
	}

	/* set initial power profile */
	if ((widget = lookup_widget(base->mainwindow, "pv_combo_profile")))
		gtk_combo_box_set_active (GTK_COMBO_BOX(widget), 0);

	/* install signal handler which glade-2 is not able to */
	if ((widget = lookup_widget(base->mainwindow, "pv_combo_policy")))
		g_signal_connect (widget, "changed", G_CALLBACK (cb_update_profile), (gpointer) 0);
	if ((widget = lookup_widget(base->mainwindow, "pv_combo_timeraction")))
		g_signal_connect (widget, "changed", G_CALLBACK (cb_update_profile), (gpointer) 1);
	if ((widget = lookup_widget(base->mainwindow, "pv_combo_coveraction")))
		g_signal_connect (widget, "changed", G_CALLBACK (cb_update_profile), (gpointer) 2);
	if ((widget = lookup_widget(base->mainwindow, "pv_combo_keyaction")))
		g_signal_connect (widget, "changed", G_CALLBACK (cb_update_profile), (gpointer) 3);
	if ((widget = lookup_widget(base->mainwindow, "pv_spin_sleeptime")))
		g_signal_connect (widget, "value_changed", G_CALLBACK (cb_update_profile), (gpointer) 4);
	if ((widget = lookup_widget(base->mainwindow, "pv_spin_dimtime")))
		g_signal_connect (widget, "value_changed", G_CALLBACK (cb_update_profile), (gpointer) 5);

	if ((widget = lookup_widget(base->mainwindow, "main_btn_apply")))
		g_signal_connect (widget, "clicked", G_CALLBACK (cb_apply_clicked), (gpointer) base);
	if ((widget = lookup_widget(base->mainwindow, "main_btn_save")))
		g_signal_connect (widget, "clicked", G_CALLBACK (cb_save_clicked), (gpointer) base);
	if ((widget = lookup_widget(base->mainwindow, "main_btn_reset")))
		g_signal_connect (widget, "clicked", G_CALLBACK (cb_reset_clicked), (gpointer) base);

	g_signal_connect (base->mainwindow, "configure-event", G_CALLBACK (cb_mainwindow_moved), (gpointer) base);
	g_signal_connect_after(base->learnwindow, "show", G_CALLBACK(cb_learnwindow_show), base);
  }

int
collect_kbd_devices(struct basedata *base)
{
	int i, fd, n = 0;
	char filename[20];
	unsigned long bit[NBITS(EV_MAX)];

	base->kbds[n].fd = -1;
	for (i = 0; i < 32; i++) {
		sprintf(filename, "/dev/input/event%d", i);
		if ((fd = open(filename, O_RDONLY)) >= 0) {
			ioctl(fd, EVIOCGBIT(0, EV_MAX), bit);
		    if (test_bit(EV_KEY, bit) && test_bit(EV_REP, bit)) {
				if (n < MAXINPUTS) {
					base->kbds[n].fd = fd;
					base->kbds[n++].gdktag = gtk_input_add_full (fd,
						GDK_INPUT_READ, (GdkInputFunction) cb_keyboard_handler,
						NULL, (gpointer) base, NULL);
					base->kbds[n].fd = -1;
				} else
					close(fd);
			} else
				close(fd);
/*		} else if (errno != ENOENT) {
			print_msg (PBB_WARN, _("Problems accessing event device 'event%d': %s\n"), i, strerror(errno));
*/
		}
	}
	return n;
}

GtkWidget *
create_learnwindow()
{
	GtkWidget *window;
	GtkWidget *frame;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *pix1, *pix2;
	GdkColor bgcolor = {0, 0xffff, 0xbbbb, 0x3333};

	if ((window = gtk_window_new (GTK_WINDOW_POPUP))) {
		gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER_ON_PARENT);
		gtk_window_set_decorated (GTK_WINDOW (window), FALSE);
		gtk_window_set_modal (GTK_WINDOW (window), TRUE);
		gtk_window_set_policy (GTK_WINDOW (window), FALSE, FALSE, FALSE);
		gtk_widget_modify_bg (window, GTK_STATE_NORMAL, &bgcolor);
		frame = gtk_frame_new (NULL);
		gtk_container_add (GTK_CONTAINER (window), frame);
		gtk_widget_show (frame);
		hbox = gtk_hbox_new (FALSE, 0);
		gtk_container_add (GTK_CONTAINER (frame), hbox);
		gtk_widget_show (hbox);

		pix1 = gtk_image_new_from_stock ("gtk-go-down", GTK_ICON_SIZE_BUTTON);
		gtk_box_pack_start (GTK_BOX (hbox), pix1, FALSE, FALSE, 0);
		gtk_widget_show (pix1);

		label = gtk_label_new (_("Please enter new key for this function."));
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
		gtk_misc_set_padding (GTK_MISC (label), 4, 4);
		gtk_widget_show (label);

		pix2 = gtk_image_new_from_stock ("gtk-go-down", GTK_ICON_SIZE_BUTTON);
		gtk_box_pack_start (GTK_BOX (hbox), pix2, FALSE, FALSE, 0);
		gtk_widget_show (pix2);
	}
	return window;
}

