/* ----------------------------------------------------------------------------
 * msghandler.c
 * code to receive messages from the server.
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <gtk/gtk.h>

#include <pbb.h>

#include "init.h"
#include "msghandler.h"
#include "taghandler.h"
#include "gettext_macros.h"

extern volatile int prgexit;

int
set_server_config (struct basedata *base)
{
	int n, d;
	long nval, oval;
	struct tagitem *taglist;
	struct GUIControlItem *GC;

	if ((taglist = malloc ((base->GUIControlCount + 1) * sizeof(struct tagitem))) == 0) {
		print_msg (PBB_ERR, _("Out of memory.\n"));
		return E_NOMEM;
	}

	taglist_init(taglist);
	for (n=0; n < base->GUIControlCount; n++) {
		GC = &base->GUIControl[n];
		if ((GC->widget) && !(GC->flags & GCF_RONLY)
			 && GTK_WIDGET_SENSITIVE(GC->widget) && GC->tag != 0) {
			oval = tagfind(base->taglist, GC->tag, -1);
			nval = GC->handler (THMODE_GET, GC->tag, GC->widget, 0);
			d = (GC->tag & FLG_STRING) ? strcmp((char*) nval, (char*) oval) : (nval - oval);
			if (d)
				taglist_add (taglist, GC->tag, (long) nval);
		}
	}
	ipc_send (0, CHANGEVALUE, taglist); /* setting the options */
	free(taglist);
	
	get_server_config(base);  /* check if they are really set */
	return 0;
}

int
get_server_config (struct basedata *base)
{
	int n;
	struct tagitem *taglist;

	if ((taglist = malloc ((base->GUIControlCount + 1) * sizeof(struct tagitem))) == 0) {
		print_msg (PBB_ERR, _("Out of memory.\n"));
		return E_NOMEM;
	}

	taglist_init(taglist);
	for (n=0; n < base->GUIControlCount; n++)
		if (base->GUIControl[n].tag != 0) {
			if (base->GUIControl[n].tag & FLG_STRING)
				taglist_add (taglist, base->GUIControl[n].tag, 0);
			else
				taglist_add (taglist, base->GUIControl[n].tag, -1);
		}
	ipc_send (0, READVALUE, taglist);
	free(taglist);
	return 0;
}

gint
timeout_callback (gpointer data)
{
	struct basedata *base = (struct basedata*) data;
	char *msgbuffer[MSGMAX];
	struct tagitem *taglist;
	int n;

	struct pbbmessage *msg = (struct pbbmessage *) msgbuffer;

	if (prgexit == 1) {
		gtk_main_quit();
		return FALSE;		/* fini */
	}

	if ((ipc_receive (msgbuffer, sizeof(msgbuffer))) == 0) {
		switch (msg->action) {
		case REGFAILED:
			print_msg (PBB_ERR, _("Client registration denied.\n"));
			gtk_main_quit();
			break;
		case CLIENTEXIT:
			gtk_main_quit();
			break;
		case WARNING:
		case CHANGEVALUE:
			taglist = msg->taglist;
			while (taglist->tag != TAG_END) {
				if ((store_config_tag (base, taglist->tag, taglist->data)) == -1) {
					if ((garbage_collection (base)) == -1) {
						print_msg (PBB_ERR, _("Out of memory.\n"));
						gtk_main_quit();
					}
					store_config_tag (base, taglist->tag, taglist->data);
				}
				for (n=0; n < base->GUIControlCount; n++)
					if (base->GUIControl[n].tag == taglist->tag) {
						base->GUIControl[n].handler (THMODE_SET, taglist->tag,
						       base->GUIControl[n].widget, taglist->data);
						break;
					}
				taglist++;
			}
		}
	}
	return TRUE;    /* keep timeout always running */
}

int
store_config_tag (struct basedata *base, long tag, long data)
{
	struct tagitem *titem;
	long taglistend;
	int rc = 0, memleft;

	taglistend = (long) base->taglist
	             + (taglist_count(base->taglist) + 1) * sizeof(struct tagitem);
	memleft = (int) ((long) base->strptr - taglistend);

	if ((titem = find_tagitem (base->taglist, tag)) == NULL) {
		if (tag & FLG_STRING) {
			if (memleft > sizeof(struct tagitem) + strlen((char*)data)) {
				base->strptr -= (strlen((char*)data) + 1);
				strcpy (base->strptr, (char*) data);
				taglist_add (base->taglist, tag, (long) base->strptr);
			} else
				rc = -1; /* garbage collection necessary */
		} else
			if (memleft >= sizeof(struct tagitem))
				taglist_add (base->taglist, tag, data);
			else
				rc = -1; /* garbage collection necessary */
	} else {
		if (tag & FLG_STRING) {
			if (strlen((char*)titem->data) >= strlen((char*) data))
				strcpy ((char*) titem->data, (char*) data); /* old strlen is sufficient */
			else
				if (memleft > strlen((char*)data)) {
					base->strptr -= (strlen((char*) data) + 1);
					strcpy (base->strptr, (char*) data); /* it is still mem free in stringbuffer */
					titem->data = (long) base->strptr;
				} else
					rc = -1; /* garbage collection necessary */
		} else
			titem->data = data;  /* new content for existing tag */
	}
	return rc;
}

int
garbage_collection (struct basedata *base)
{
	struct tagitem *newtaglist;
	struct tagitem *oldtaglist;
	char *newstrptr;
	int	newbufferlen;

	newbufferlen = base->bufferlen + TAGBUFFERINCR;
	if (newbufferlen > TAGBUFFERMAX)
		newbufferlen = TAGBUFFERMAX;

	if ((newtaglist = (struct tagitem*) malloc (newbufferlen)) != NULL) {
		taglist_init (newtaglist);
		newstrptr = (char*) newtaglist + newbufferlen;

		oldtaglist = base->taglist;
		while (oldtaglist->tag != TAG_END) {
			if (oldtaglist->tag & FLG_STRING) {
				newstrptr -= (strlen((char*) oldtaglist->data) + 1);
				strcpy (newstrptr, (char*) oldtaglist->data);
				taglist_add (newtaglist, oldtaglist->tag, (long) newstrptr);
			} else
				taglist_add (newtaglist, oldtaglist->tag, oldtaglist->data);
			oldtaglist++;
		}
		oldtaglist = base->taglist;
		base->taglist = newtaglist;
		base->bufferlen = newbufferlen;
		base->strptr = newstrptr;
		free (oldtaglist);
		return 0;
	}
	return -1;
}

#ifdef HAVE_OLDLIBPBB     /* support for older libpbb.a */
struct tagitem*
find_tagitem (struct tagitem *taglist, long tag)
{
	while (taglist->tag != TAG_END && taglist->tag != tag)
		taglist++;
	if (taglist->tag == TAG_END)
		return NULL;
	else
		return taglist;
}

int
taglist_count (struct tagitem *taglist)
{
	int n = 0;
	while (taglist->tag != TAG_END) {
		n++; taglist++;
	}
	return n;
}
#endif
