/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <sstream>
#include <cpufreq.h>

#include "cpufreq_interface.h"
#include "globals.h"

CPUFreq_Userspace::CPUFreq_Userspace(list< int > cpu_cores)
	: CPUFreq_Interface(cpu_cores)
{
	USERSPACE_STRING = "userspace";

	_current_speed = 0;
	_last_cpu_load = 50;
	_new_speed = 0;
	_last_step = 0;

	ostringstream strstr;

	strstr << SYSFS_FILES << "cpu" << _cpu_base << "/cpufreq/scaling_setspeed";
	CURRENT_SPEED_FILE = strstr.str();
	strstr.str("");

#ifdef CPUFREQ_MEASURE
	for (int x = 0; x <= MAX_SPEEDS; x++)
		time_spent[x] = 0;
	t_count = 0;
	cpu_load_sum = 0;
	start_time = new(time_t);
#endif
}

CPUFreq_Userspace::~CPUFreq_Userspace()
{
	// uncomment to set to maximum speed on exit
//	setSpeed(_current_speed, 0);        
}

bool CPUFreq_Userspace::init()
{
	if (setGovernor(USERSPACE_STRING) < 0) {
		/* no userspace governor set */
		pDebug(DBG_INFO, "Could not set userspace governor.");
		return false;
	}

	return true;
}

bool CPUFreq_Userspace::readFrequencies()
{
	if (!online())
		return false;

	if (initFreqsViaFile() < 0)
		if (initFreqsViaTesting() < 0)
			return false;
	createHysteresisArray();
	pDebug(DBG_DIAG, "Available speeds:");
	for (int x = 0; _speeds_kHz[x]; x++) {
		pDebug(DBG_DIAG, " %2u: %9uKHz", x, _speeds_kHz[x]);
	}
	return true;
}

unsigned CPUFreq_Userspace::getSpeed()
{
	return cpufreq_get_freq_kernel(_cpu_base);
}

bool CPUFreq_Userspace::setSpeed(unsigned kHz)
{
	// userspace already checked in setSpeed(unsigned, unsigned)
	if (cpufreq_set_frequency(_cpu_base, kHz) < 0) {
		pDebug(DBG_ERR, "Could not set speed to %u kHz: %s", kHz, strerror(errno));
		return false;
	} else {
		pDebug(DBG_INFO, "wrote %u to %s", kHz, CURRENT_SPEED_FILE.c_str());
		return true;
	}
}

void CPUFreq_Userspace::setSpeed(unsigned current, unsigned target)
{

	if (current == target)
		return;
	int delta = (current > target) ? -1 : 1;
	do {
		current += delta;
		setSpeed(_speeds_kHz[current]);
	}
	while (current != target);
	_current_speed = target;
}

int CPUFreq_Userspace::increaseSpeed()
{
	if (_mode != _DYNAMIC)
		return -1;

	if (_new_speed != 0) {
		_new_speed--;
	} else
		return 0;
	if (_current_speed != _new_speed) {
		pDebug(DBG_INFO, "current: %u new: %u",
		       _speeds_kHz[_current_speed], _speeds_kHz[_new_speed]);
		setSpeed(_current_speed, _new_speed);
	}
	return 1;
}

int CPUFreq_Userspace::decreaseSpeed()
{
	if (_mode != _DYNAMIC)
		return -1;

	if (_speeds_kHz[_new_speed + 1] != 0) {
		_new_speed++;
	} else
		return 0;
	if (_current_speed != _new_speed) {
		pDebug(DBG_INFO, "current: %u new: %u",
		       _speeds_kHz[_current_speed], _speeds_kHz[_new_speed]);
		setSpeed(_current_speed, _new_speed);
	}
	return 1;
}

int CPUFreq_Userspace::adjustSpeed()
{
	int ret = calcCPULoad(Powersave::Globals::config_obj->current_scheme->CONSIDER_NICE);

	if (ret < 0) {
		pDebug(DBG_WARN, "calcCPULoad failed. Cannot adjust speeds: %d", ret);
		return false;
	}

	int CPULoad = 0;
	for (list< int >::iterator it = _cpu_cores.begin();
	     it != _cpu_cores.end(); ++it) {
		
		pDebug(DBG_DIAG, "checking cpu %d: cpu_core: %d\n", _cpu_base, *it);
		CPULoad = max(getCPULoad(*it), CPULoad);
	}

	ret = 0;

	/* DEBUG */
	switch (_mode) {
	case _DYNAMIC:
		pDebug(DBG_DEBUG, "Dynamic mode");
		break;
	case _PERFORMANCE:
		pDebug(DBG_DEBUG, "Performance mode");
		break;
	case _POWERSAVE:
		pDebug(DBG_DEBUG, "Powersave mode");
		break;
	}

	/* Debug */
	pDebug(DBG_DEBUG, "cpu_max: %d high_cpu_limit: %d consider_nice: %d hysteresis: %d",
	       _cpu_max, _high_cpu_limit, _consider_nice, _cpu_hysteresis);
	pDebug(DBG_DEBUG, "Current: %u; current speed: %u MHz, New: %u, new_speed: %u MHz",
	       _current_speed, _speeds_kHz[_current_speed], _new_speed, _speeds_kHz[_new_speed]);
	pDebug(DBG_DEBUG,
	       "CPU load: %d, Last CPU load %d, CPULoad diff: %d, last_step: %d, demotion: %u", CPULoad,
	       _last_cpu_load, CPULoad - _last_cpu_load, _last_step, _demotion[_current_speed]);

	/* Debug */
#ifdef CPUFREQ_MEASURE
    time_spent[_current_speed] += polling_interval;
    count++;
    cpu_load_sum += CPULoad;
#endif
	if (_mode == _DYNAMIC) {
		/* directly increase speed to maximum if cpu load jumped */
		if (_high_cpu_limit && (CPULoad - _last_cpu_load) > _high_cpu_limit) {
			if (_current_speed != 0) {
				_new_speed = 0;
				setSpeed(_current_speed, _new_speed);
				pDebug(DBG_INFO, "jumped to max (%d kHz)", _speeds_kHz[_current_speed]);
				ret = 1;
			}
		} else if (CPULoad > _cpu_max && _current_speed > 0) {
			increaseSpeed();
			pDebug(DBG_INFO, "increased to %d kHz", _speeds_kHz[_new_speed]);
			ret = 1;
		} else if (CPULoad < (int)_demotion[_current_speed] && _current_speed < _last_step) {
			decreaseSpeed();
			pDebug(DBG_INFO, "decreased to %d kHz", _speeds_kHz[_new_speed]);
			ret = -1;
		} else {
			ret = 0;
			pDebug(DBG_DEBUG, "Speed not changed");
		}
	} else if (_mode == _POWERSAVE && _current_speed != _last_step) {
		_new_speed = _last_step;
		if (_current_speed != _new_speed) {
			pDebug(DBG_INFO, "current speed: %uKHz new speed: %uKHz",
			       _speeds_kHz[_current_speed], _speeds_kHz[_new_speed]);
			setSpeed(_current_speed, _new_speed);
		}
		ret = -1;
	} else if (_mode == _PERFORMANCE && _current_speed != 0) {
		_new_speed = 0;
		if (_current_speed != _new_speed) {
			pDebug(DBG_INFO, "current speed: %uKHz new speed: %uKHz",
			       _speeds_kHz[_current_speed], _speeds_kHz[_new_speed]);
			setSpeed(_current_speed, _new_speed);
		}
		ret = 1;
	} else
		ret = 0;
	_last_cpu_load = CPULoad;
	return ret;
}

void CPUFreq_Userspace::createHysteresisArray()
{
	if (_last_step > 0) {
		for (unsigned i = 0; i < _last_step; i++) {
			_demotion[i] = (_cpu_max - _cpu_hysteresis) * _speeds_kHz[i + 1] / _speeds_kHz[i];
			pDebug(DBG_DIAG, "Speed: %2u, kHz: %9u, demotion: %3u %%",
			       i, _speeds_kHz[i], _demotion[i]);
		}
	}
	_demotion[_last_step] = 0;
}

void CPUFreq_Userspace::reinitSpeed()
{
	if (!online())
		return;

	setSpeed(_speeds_kHz[_current_speed]);
	pDebug(DBG_DIAG, "forced speed to %d kHz", _speeds_kHz[_current_speed]);
	_new_speed = _current_speed;
}

void CPUFreq_Userspace::setConfig()
{
	createHysteresisArray();
}

#ifdef CPUFREQ_MEASURE

void CPUFreq_Userspace::startMeasures()
{
    for (int x = 0; x < cpus; x++) {
        for (int y = 0; y <= MAX_SPEEDS; y++)
            cpu_objects[x]->time_spent[y] = 0;
        cpu_objects[x]->count = 0;
        cpu_objects[x]->cpu_load_sum = 0;
    }
    time(start_time);
    struct tm *tm = new(struct tm);
    tm = localtime(start_time);

    FILE *logFile = fopen(MEASURE_LOG_FILE, "a+");
    if (!logFile) {
        pDebug(DBG_ERR, "Could not open log file %s", MEASURE_LOG_FILE);
        exit(1);
    }
    fprintf(logFile, "\nStart cpufreq measures at: %d.%d.%d at %d:%d:%d\n",
        tm->tm_mday, tm->tm_mon, tm->tm_year, tm->tm_hour, tm->tm_min, tm->tm_sec);
    fclose(logFile);
    free(tm);
}

void CPUFreq_Userspace::stopMeasures()
{
    FILE *logFile = fopen(MEASURE_LOG_FILE, "a+");
    if (!logFile) {
        pDebug(DBG_ERR, "Could not open log file %s", MEASURE_LOG_FILE);
        exit(1);
    }
    time_t *stop_time = new(time_t);
    tm *tm = new(struct tm);
    time(stop_time);
    tm = localtime(stop_time);
    unsigned long measure_time = *stop_time - *start_time;
    for (int x = 0; x < cpus; x++) {
        for (int k = 0; cpu_objects[x]->_speeds_kHz[k] != 0; k++) {
            fprintf(logFile, "%d\t: %d minutes %d seconds\n",
                cpu_objects[x]->_speeds_kHz[k],
                (cpu_objects[x]->time_spent[k] / 60000), (cpu_objects[x]->time_spent[k] / 1000) % 60);
        }
        if (cpu_objects[x]->count > 0)
            fprintf(logFile, "Average CPU usage: %##f - Count: %d\n",
                (double)cpu_objects[x]->cpu_load_sum / cpu_objects[x]->count, cpu_objects[x]->count);
    }
    fprintf(logFile, "\nStop cpufreq measures at: %d.%d.%d at %d:%d:%d\n",
        tm->tm_mday, tm->tm_mon, tm->tm_year, tm->tm_hour, tm->tm_min, tm->tm_sec);

    fprintf(logFile, "Total measure time: %d minutes %d seconds - Check: %d minutes %d seconds\n",
        measure_time / 60, (measure_time) % 60,
        (cpu_objects[0]->count * polling_interval / 1000) / 60,
        (cpu_objects[0]->count * polling_interval / 1000) % 60);
    fclose(logFile);
    free(tm);
    free(stop_time);
}
#endif


int CPUFreq_Userspace::initFreqsViaFile()
{
	int speed_no = 1;

	cpufreq_available_frequencies *freqs;

	freqs = cpufreq_get_available_frequencies(_cpu_base);
	if (freqs == NULL) {
		_speeds_kHz[0] = 0;
		_last_step = 0;
		return -1;
	}

	for (speed_no = 0; freqs != NULL; speed_no++, freqs = freqs->next) {
		_speeds_kHz[speed_no] = freqs->frequency;

		if (freqs->next == NULL) {
			cpufreq_put_available_frequencies(freqs->first);
			break;
		}
	}

	_speeds_kHz[speed_no + 1] = 0;
	_last_step = speed_no;

	_current_speed = 0;
	reinitSpeed(); // sets the frequency and _new_speed=_current_speed

	int i = 0;
	while (_speeds_kHz[i] != 0) {
		i++;
	}

	return speed_no;
}

int CPUFreq_Userspace::initFreqsViaTesting()
{
	unsigned long min, max;

	if (!getMinMaxSpeeds(&min, &max)) {
		/* Place this somewhere else where we could get out */
		pDebug(DBG_WARN, "Could not read min and max speed");
		return -1;
	}
	// initialise speed steps that can be set
	unsigned step;
	step = (max - min) / (MAX_SPEEDS - 1);
	if (step < MIN_SPEED_STEP)
		step = MIN_SPEED_STEP;
	pDebug(DBG_DIAG, "Speed step: %u kHz", step);
	/* Go to max speed if we are not already there  */
	for (unsigned current = getSpeed(); current < max; current += step)
		setSpeed(current);
	/* loop CPU from the maximum speed to the minimum speed and try
	   to set every possible speed divisible by step! */
	_speeds_kHz[0] = max;
	unsigned current_init_speed = 0;
	for (unsigned current = max - step; current > min - step; current -= step) {
		setSpeed(current);
		unsigned real = getSpeed();
		if (real != _speeds_kHz[current_init_speed]) {
			_speeds_kHz[++current_init_speed] = real;
			if (current_init_speed + 1 == MAX_SPEEDS)
				break;
		}
	}
	_speeds_kHz[current_init_speed + 1] = 0;
	// go back to maximum speed because program expects to start there 
	_new_speed = 0;
	setSpeed(current_init_speed, _new_speed);
	// the last step is the lowest found speed 
	_last_step = current_init_speed;

	return 1;
}

bool CPUFreq_Userspace::getMinMaxSpeeds(unsigned long *min, unsigned long *max)
{
	
	if (cpufreq_get_hardware_limits(_cpu_base, min, max) < 0) {
		return false;
	}

	if (*min > 0 && *max > 0) {
		pDebug(DBG_DIAG, "Speed min: %lu kHz  - Speed max: %lu kHz", *min, *max);
		return true;
	} else {
		pDebug(DBG_WARN, "Speedstepping not supported");
		return false;
	}
}
