/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "powersave_hal.h"
#include "device.h"
#include "pm_interface.h"

char *DPM_CLASS_NAMES[DPM_CLASS_MAX] = {
	"usb",
	"lan",
	"wlan",
	"sound"
};

using namespace std;

/*** class Device ***/

Device::Device(const string &udi)
{
	_udi = udi;

	char *device_path = ps_hal_get_property_string((char*)_udi.c_str(),
						       "linux.sysfs_path");
	if (device_path == NULL || strlen(device_path) == 0) {
		pDebug(DBG_WARN, "Could not get linux.sysfs_path of device '%s'",
		       _udi.c_str());
		return;
	}

	_device_path.append(device_path);
	libhal_free_string(device_path);
}

Device::~Device()
{
}

bool Device::suspend(DPM_POWER_STATE state, bool force)
{
	if (updateState() == state) {
		pDebug(DBG_INFO, "Power state %d already set in '%s'",
		       state, _state_file.c_str());
		return false;
	}

	if ( !force && isActive() ) {
		pDebug(DBG_INFO, "Tried to suspend device '%s', but it is in use",
		       _udi.c_str());
		return false;
	}
	
	errno = 0;
	FILE *file = fopen(_state_file.c_str(), "w");

	if (file == NULL) {
		pDebug(DBG_DIAG, "Could not open power state file '%s': %s",
		       _state_file.c_str(), strerror(errno));
		return false;
	}

	errno = 0;
	if (fprintf(file, "%d", state) < 0) {
		pDebug(DBG_ERR, "Could not write to state file '%s'",
		       _state_file.c_str());
		fclose(file);

		return false;
	}

	pDebug(DBG_INFO, "wrote %d to power state file '%s'\n",
	       state, _state_file.c_str());

	// set current state the device is in
	_current_state = state;

	errno = 0;
	if (fclose(file) != 0) {
		pDebug(DBG_WARN, "Unable to close power state file '%s': %s",
		       _state_file.c_str(), strerror(errno));
		return false;
	}
	
	return true;
}

bool Device::resume()
{
	return suspend(DPM_STATE_D0, true);
}

DPM_POWER_STATE Device::updateState()
{
	int state = -1;

	errno = 0;
	FILE *file = fopen(_state_file.c_str(), "r");

	if (file == NULL) {
		pDebug(DBG_DIAG, "Could not open power state file '%s': %s",
		       _state_file.c_str(), strerror(errno));
		return DPM_STATE_UNKNOWN;
	}

	errno = 0;
	if (fscanf(file, "%d", &state) < 0) {
		pDebug(DBG_ERR, "Could not read from state file '%s'",
		       _state_file.c_str());
		fclose(file);

		return DPM_STATE_UNKNOWN;
	}

	errno = 0;
	if (fclose(file) != 0) {
		pDebug(DBG_WARN, "Unable to close power state file '%s': %s",
		       _state_file.c_str(), strerror(errno));
		return DPM_STATE_D0;
	}

	if (state < 0) {
		pDebug(DBG_WARN, "Read value from state file '%s' is no power state",
		       _state_file.c_str());
		return DPM_STATE_UNKNOWN;
	}

	// update current state
	_current_state = (DPM_POWER_STATE)state;
	
	return _current_state;
}

DPM_DEVICE_CLASS Device::deviceClass()
{
	return _device_class;
}

string Device::udi()
{
	return _udi;
}

/*** class UsbDevice ***/

UsbDevice::UsbDevice(const string &udi) : Device(udi)
{
	_device_class = DPM_CLASS_USB;
	_state_file = _device_path + "/power/state";	
	pDebug(DBG_INFO, "Set power state file of usb devices to %s",
	       _state_file.c_str());
	updateState();
}

UsbDevice::~UsbDevice()
{
}

bool UsbDevice::isActive()
{
	return false;
}

void UsbDevice::getDevices(std::list<std::string> &device_udis)
{
	if (!ps_hal_init()) {
		return;
	}

	char **usb_devs;
	int num_devs;
	usb_devs = libhal_manager_find_device_string_match(ps_hal_context(),
							   "info.bus",
							   "usb",
							   &num_devs,
							   ps_hal_dbus_error());

	pDebug(DBG_INFO, "Found %d usb devices", num_devs);

	if (num_devs == 0) {
		libhal_free_string_array(usb_devs);
		return;
	}

	for (int i = 0; i < num_devs; i++) {
		device_udis.push_back(usb_devs[i]);
		pDebug(DBG_INFO, "Added usb device '%s'", usb_devs[i]);
	}

	libhal_free_string_array(usb_devs);
}

bool UsbDevice::isDevice(const char *udi) {
	bool ret = true;
	char *bus = ps_hal_get_property_string((char*)udi, "info.bus");

	if ( bus == NULL || strcmp(bus, DPM_CLASS_NAMES[DPM_CLASS_USB]) != 0) {
		ret = false;
	}

	libhal_free_string(bus);
	return ret;
}

/*** class Wlan Device ***/

WlanDevice::WlanDevice(const string &udi) : Device(udi)
{
	_device_class = DPM_CLASS_WLAN;
	_state_file = _device_path + "/device/power/state";
	pDebug(DBG_INFO, "Set power state file of wlan devices to %s",
	       _state_file.c_str());
	updateState();
}

WlanDevice::~WlanDevice()
{
}

bool WlanDevice::isActive()
{
	// this does not work yet, because hal does not update the
	// interface status. We will ask the Networkmanager for that in
	// the future
	int active = ps_hal_get_property_bool((char*)_udi.c_str(), "net.interface_up");
	pDebug(DBG_DIAG, "Check if wlan device '%s' is active: %s",
	       _udi.c_str(), active ? "yes" : "no");

	return false;
}

void WlanDevice::getDevices(std::list<std::string> &device_udis)
{
	if (!ps_hal_init()) {
		return;
	}

	char **wlan_devs;
	int num_devs;
	wlan_devs = libhal_find_device_by_capability(ps_hal_context(),
						     "net.80211",
						     &num_devs,
						     ps_hal_dbus_error());

	pDebug(DBG_INFO, "Found %d wlan devices", num_devs);

	if (num_devs == 0) {
		libhal_free_string_array(wlan_devs);
		return;
	}

	for (int i = 0; i < num_devs; i++) {
		device_udis.push_back(wlan_devs[i]);
		pDebug(DBG_INFO, "Added wlan device '%s'", wlan_devs[i]);
	}

	libhal_free_string_array(wlan_devs);
}

bool WlanDevice::isDevice(char *udi) {
	if (!ps_hal_query_capability(udi, "net.80211")) {
		return false;
	}
	else {
		return true;
	}
}

/*** class SoundDevice ***/

SoundDevice::SoundDevice(const string &udi) : Device(udi)
{
	_device_class = DPM_CLASS_SOUND;
	_state_file = _device_path + "/power/state";
	pDebug(DBG_INFO, "Set power state file of wlan devices to %s",
	       _state_file.c_str());
	updateState();
}

SoundDevice::~SoundDevice()
{
}

bool SoundDevice::isActive()
{
	return false;
}

void SoundDevice::getDevices(std::list<std::string> &device_udis)
{
	if (!ps_hal_init()) {
		return;
	}

	char **sound_devs;
	int num_devs;
	sound_devs = libhal_manager_find_device_string_match(ps_hal_context(),
							     "info.category",
							     "alsa",
							     &num_devs,
							     ps_hal_dbus_error());

	pDebug(DBG_INFO, "Found %d sound devices", num_devs);

	if (num_devs == 0) {
		libhal_free_string_array(sound_devs);
		return;
	}

	/* there are multiple sound devices, but we only want the parent */
	for (int i = 0; i < num_devs; i++) {
		char *physical_device = ps_hal_get_property_string(sound_devs[i],
								   "alsa.physical_device");

		if (physical_device != NULL && strlen(physical_device) > 0) {
			device_udis.push_back(physical_device);
			pDebug(DBG_INFO, "Added sound device '%s'", sound_devs[i]);
			libhal_free_string(physical_device);
			libhal_free_string_array(sound_devs);
			break;
		}
	}
}

bool SoundDevice::isDevice(const char *udi) {
	bool ret = true;
	char *category = ps_hal_get_property_string((char*)udi, "info.category");

	if (category != "alsa") {
		ret = false;
	}

	libhal_free_string(category);
	return ret;
}

/*** class LanDevice ***/

LanDevice::LanDevice(const string &udi) : Device(udi)
{
	_device_class = DPM_CLASS_LAN;
	_state_file = _device_path + "/device/power/state";
	pDebug(DBG_INFO, "Set power state file of wlan devices to %s",
	       _state_file.c_str());
	updateState();
}

LanDevice::~LanDevice()
{
}

bool LanDevice::isActive()
{
	// we will ask the Networkmanager for that in the future
	return false;
}

void LanDevice::getDevices(std::list<std::string> &device_udis)
{
	if (!ps_hal_init()) {
		return;
	}

	char **lan_devs;
	int num_devs;

	lan_devs = libhal_find_device_by_capability(ps_hal_context(),
						    "net.80203",
						    &num_devs,
						    ps_hal_dbus_error());

	pDebug(DBG_INFO, "Found %d sound devices", num_devs);

	if (num_devs == 0) {
		libhal_free_string_array(lan_devs);
		return;
	}

	for (int i = 0; i < num_devs; i++) {
		device_udis.push_back(lan_devs[i]);
		pDebug(DBG_INFO, "Added lan device '%s'", lan_devs[i]);
	}

	libhal_free_string_array(lan_devs);
}

bool LanDevice::isDevice(char *udi)
{
	if (!ps_hal_query_capability(udi, "net.80203")) {
		return false;
	}
	else {
		return true;
	}
}
