/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_POWERLIB_LOCAL_H
#define POWERSAVE_POWERLIB_LOCAL_H

/** @defgroup libpower_local local powerlib implementations
 * @ingroup libpower
 * @{
 */

#include <string.h>
#include <time.h>

#include "powerlib.h"

/** @brief apm related header holding apm information */
typedef struct apm_info {
	char driver_version[10];
	int apm_version_major;
	int apm_version_minor;
	int apm_flags;
	int ac_line_status;
	int battery_status;
	int battery_flags;
	int battery_percentage;
	int battery_time;
	int using_minutes;
} apm_info;

/** @brief An array of this one is used to estimate the remaining 
    battery capacity in minutes. */
typedef struct capacity_snapshot {
	time_t measure_point;
	int remaining_capacity;
} capacity_snapshot;

/** @brief get amount of entries in a directory
 *
 * use this to determine available devices
 *
 * @param dir_name directory
 *
 * @return number of devices, < 0 on error
 *
 */
int getDevicesNum(const char *dir_name);

/** @brief get next value which is seperated by a colon
 *
 * reads a line and parsed it according to:
 *
 * "def[spaces]*:[spaces]*value[spaces]*\n
 *
 * e.g.: var:  56 mWh  returns 56 mWh in value and var in def
 *
 * @param file the filename
 * @param value value to store the left side of the colon
 * @param value_size the size of the value
 * @param def value to store the right side of the colon
 * @param def_size size of def
 *
 * @retval -1 if no colon ':' if found
 */
int getColonValue(FILE * file, char *value, const size_t value_size, char *def, const size_t def_size);

/** @brief get a directory entry
 *
 * file: directory entry searched, e.g.: BAT0
 *
 * give directory and position of the searched file
 *
 * position is determined by just looping over the directory
 *
 * directory . and .. are not counted.
 *
 * @param num 0-(number of directories - 1)
 * @param file the filename to search for
 * @param size_file the size of the file
 * @param dir_in the directory the file is in
 *
 * @retval 1 if num directory entry(e.g. BAT0) has beend found
 * @retval -1 if the num directory entry does not exist
 */
int getDirEntry(const int num, char *file, const size_t size_file, const char *dir_in);

/** @brief write a line to a file
 *
 * (e.g.: write "3\n" to /acpi/sleep)
 *
 * @param filename the filename to write to
 * @param fmt format rule
 * @param ... lines to write
 *
 * @return < 0 on error
 */
int _write_line(const char *filename, const char *fmt, ...);

/** @brief reads out information from /proc/apm
 *
 * @param i @ref apm_info struct which gets filled with information
 * 
 * @return < 0 on error
 * @retval 1 on success,
 * @retval -1 if APM notinstalled
 * @retval -2 if APM installed, but old version. 
 */
int read_apm_info(apm_info * i);

/** @brief check whether module has been loaded and device is available
 *
 * @param dir acpi directory
 *
 * @retval 1 on success
 * @retval NO_MODULE_ERROR
 * @retval NO_DEVICE_ERROR*/
int check_ACPI_dir(const char *dir);

#endif /* POWERSAVE_POWERLIB_LOCAL_H */

/** @} */// end of libpower_local group
