/* Test Box::refine_with_constraints(const Constraint_System&) with
   instances that may require a watchdog timer.
   Copyright (C) 2001-2008 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02111-1307, USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#include "ppl_test.hh"
#include "pwl.hh"
#include <sstream>

namespace {

class Timeout : virtual public std::exception,
                public Parma_Polyhedra_Library::Throwable {
public:
  const char* what() const throw() {
    return "Timeout in refine1.cc";
  }

  void throw_me() const {
    throw *this;
  }

  int priority() const {
    return 0;
  }

  Timeout() {
  }

  ~Timeout() throw() {
  }
};

Timeout t;

bool
test01() {
  Variable A(0);
  Variable B(1);

  Constraint_System cs;
  cs.insert(A >= -5);
  cs.insert(A <= 5);
  cs.insert(A == B);
  cs.insert(A == 2*B);
  print_constraints(cs, "*** cs ***");

  Rational_Box known_result(2);
  known_result.add_constraint(A == 0);
  known_result.add_constraint(B == 0);
  print_constraints(known_result, "*** known_result ***");

  TBox box(2);

  bool ok = false;

  typedef TBox::interval_type::boundary_type boundary_type;
  if (std::numeric_limits<boundary_type>::is_exact
      && !std::numeric_limits<boundary_type>::is_integer) {
    // With interval boundaries made of rational numbers, this
    // refinement instance does not terminate: we use a watchdog timer.
    try {
      // Set a 0.1 seconds timeout.
      Parma_Watchdog_Library::Watchdog
        w(10, abandon_expensive_computations, t);

      box.refine_with_constraints(cs);

      // We should never get here.
      abandon_expensive_computations = 0;
      nout << "unexpected termination" << endl;
      ok = false;
    }
    catch (const Timeout&) {
      abandon_expensive_computations = 0;
      nout << "timeout, as expected" << endl;

      // The box will have been shrunk, nonetheless.
      ok = check_result(box, known_result, "1.0e-6", "1.0e-6", "1.0e-6");
    }
    catch (...) {
      abandon_expensive_computations = 0;
      nout << "unexpected exception" << endl;
      ok = false;
    }
  }
  else {
    // With interval boundaries other than rational numbers, this instance
    // of refinement terminates rather quickly: no timer is necessary.
    box.refine_with_constraints(cs);

    ok = check_result(box, known_result, "5.61e-45", "2.81e-45", "1.41e-45");
  }

  print_constraints(box, "*** box.refine_with_constraints(cs) ***");

  return ok;
}

bool
test02() {
  std::stringstream ss;
  ss.str("topology NECESSARILY_CLOSED\n"
         "175 x 80 (not_sorted)\n"
         "index_first_pending 175\n"
         "0 -292733975779082240 -24769797950537728 -3602879701896397 -20266198323167232 -6755399441055744 -27021597764222976 -14861878770322636 -26120877838748876 0 49539595901075456 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 -18014398509481984 -12249790986447750 0 -4863887597560136 -2161727821137838 -3422735716801577 -1261007895663739 -9007199254740992 0 9007199254740992 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 18014398509481984 17293822569102704 0 17473966554197524 6485183463413514 6305039478318694 5224175567749775 11168927075878830 0 -9007199254740992 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 9007199254740992 9007199254740992 7025615418697974 9007199254740992 8556839292003942 8016407336719483 8736983277098762 8827055269646172 0 -8106479329266893 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 219325301852943168 189151184349560832 197933203622933312 220451201759785792 136459068709326032 213470622337361504 220451201759785792 217298682020626432 4728779608739021 -225179981368524800 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 449909602774312576 396767127171340672 373348409109014144 451711042625260736 343624651568368832 435047724003989888 451711042625260736 441803123445045632 3152519739159347 -450359962737049600 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 94237822202727632 89396452603304352 83992133050459744 106960491150049280 77236733609404000 94350412193411888 106960491150049280 90972712472884016 2589569785738035 -101330991615836160 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 5981343255101440 5854679515581645 5671720780719718 6896136929411072 5404319552844595 6108006994621235 6896136929411072 5735052650479616 70368744177664 -6333186975989760 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 -1 3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 1 1 1 1 1 1 1 1 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 -292733975779082240 -24769797950537728 -3602879701896397 -6755399441055744 -12159718993900340 -26120877838748876 0 49539595901075456 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 -18014398509481984 -10268207150404730 0 -2161727821137838 -2341871806232658 -8286623314361713 0 9007199254740992 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 576460752303423488 576460752303423488 5764607523034235 207525870829232448 161409010644958592 386228704043293760 0 -288230376151711744 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 9007199254740992 8827055269646172 8556839292003942 7115687411245384 9007199254740992 0 -8106479329266893 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 13707831365808948 11821949021847552 12370825226433332 8528691794332877 10963450342880052 13581167626289152 492581209243648 -12525636463624192 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 224954801387156288 198383563585670336 186674204554507072 171812325784184416 183296504833979200 220901561722522816 4278419646001971 -200410183417987072 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 188475644405455264 178792905206608704 167984266100919488 154473467218808000 158977066846178496 181945424945768032 7656119366529843 -184647584722190336 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 382805968326492160 374699488997225280 362990129966061952 345876451382054080 333266372425416704 367043369630695424 8106479329266893 -369295169444380672 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -78812993478983680 -10696049115004928 -3039929748475085 -12159718993900340 -1688849860263936 -6980579422424269 -6755399441055744 -5066549580791808 -6192449487634432 -7318349394477056 0 12384898975268864 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -18014398509481984 -12610078956637388 -2341871806232658 -17473966554197524 -2161727821137838 -3422735716801577 -3422735716801577 -4863887597560136 -12249790986447750 -8646911284551352 0 9007199254740992 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 7475975381435023 2522015791327478 9007199254740992 3242591731706757 3152519739159347 3152519739159347 8736983277098762 8646911284551352 5044031582654956 0 -4503599627370496 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 9007199254740992 7115687411245384 9007199254740992 8556839292003942 8016407336719483 8016407336719483 9007199254740992 9007199254740992 8736983277098762 0 -8106479329266893 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 220226021778417248 200635363399355584 175415205486080832 190502264237771968 136459068709326032 215947602132415296 213470622337361504 220451201759785792 189151184349560832 217298682020626432 4278419646001971 -225179981368524800 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 921436483760003456 828662331436171264 733186019335916800 807945773150267008 687249303136737664 879102647262720768 870095448007979776 903422085250521472 793534254342681344 874599047635350272 8106479329266893 -900719925474099200 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 213470622337361504 174289305579238208 158977066846178496 188250464424086720 154473467218808000 192303704088720192 188700824386823776 213920982300098560 178792905206608704 185097944684927392 5404319552844595 -202661983231672320 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 898918485623150976 721476660304753408 666532744850833408 805243613373844736 691752902764108160 790832094566259072 781824895311518080 882705526964617216 749398977994450560 750299697919924608 8106479329266893 -810647932926689280 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -78812993478983680 -10696049115004928 -3039929748475085 -12159718993900340 -6980579422424269 -7318349394477056 -1688849860263936 -6192449487634432 0 12384898975268864 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -36028797018963968 -24499581972895500 -3242591731706757 -33506781227636492 -5404319552844595 -16212958658533786 -4323455642275676 -24499581972895500 0 18014398509481984 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 18014398509481984 16212958658533786 6665327448508334 18014398509481984 8106479329266893 11349071060973650 6485183463413514 17293822569102704 0 -9007199254740992 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 9007199254740992 8286623314361713 9007199254740992 8827055269646172 9007199254740992 8556839292003942 9007199254740992 0 -8106479329266893 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 220226021778417248 200635363399355584 175415205486080832 190502264237771968 215947602132415296 217298682020626432 136459068709326032 189151184349560832 8782019273372467 -200410183417987072 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 230359120940000864 207165582859042816 183296504833979200 201986443287566752 219775661815680192 218649761908837568 171812325784184416 198383563585670336 3152519739159347 -200410183417987072 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 6670956948042547 5446540799351194 4968033338943078 5882827013252710 6009490752772506 5784310771403981 4827295850587750 5587278287706522 246290604621824 -5770237022568448 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 449459242811575488 360738330152376704 333266372425416704 402621806686922368 395416047283129536 375149848959962304 345876451382054080 374699488997225280 5854679515581645 -369295169444380672 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -252201579132747776 -8106479329266893 -6305039478318694 -47738156050127256 -11258999068426240 -51791395714760704 -6755399441055744 -18915118434956084 -19365478397693132 0 45035996273704960 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -72057594037927936 0 0 -48999163945791000 0 -55484347409204512 -8646911284551352 -5764607523034235 -5044031582654956 0 36028797018963968 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 18014398509481984 0 0 15672526703249326 0 16753390613818246 6485183463413514 5404319552844595 4863887597560136 0 -9007199254740992 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 9007199254740992 4863887597560136 9007199254740992 5854679515581645 9007199254740992 8556839292003942 8196551321814303 8106479329266893 0 -8106479329266893 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 223828901480313664 197933203622933312 194105143939668384 224954801387156288 201761263306198208 179018085187977216 136459068709326032 224054081461682176 228332501107684160 3602879701896397 -227431781182210048 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 463870761619161088 412529725867137408 405323966463344640 452161402587997824 412980085829874496 383256328289229184 343624651568368832 449909602774312576 457115362178105344 3602879701896397 -454863562364420096 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 6804657561980109 6199486362052198 6192449487634432 5685794529555251 5580241413288755 5643573283048653 4827295850587750 6164301989963366 6262818231812096 140737488355328 -6403555720167424 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 455764282289894208 414331165718085632 411178645978926272 367944089556169536 369745529407117696 388210287879336768 345876451382054080 400820366835974144 406224686388818752 3602879701896397 -409827566090715136 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -126100789566373888 -23869078025063628 -5629499534213120 -8106479329266893 -25895697857380352 -4503599627370496 -32876277279804620 -13735978863480012 -9457559217478042 0 24769797950537728 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -144115188075855872 -56204923349583792 0 -38911100780481088 -105204087295374784 0 -144115188075855872 -47558012065032440 -5764607523034235 0 72057594037927936 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 9007199254740992 0 9007199254740992 8827055269646172 3422735716801577 9007199254740992 5854679515581645 3242591731706757 0 -4503599627370496 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 9007199254740992 7836263351624663 9007199254740992 9007199254740992 9007199254740992 9007199254740992 9007199254740992 8827055269646172 0 -8106479329266893 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 110000420898524368 110563370851945680 98966601811466656 88720912659198768 88045372715093200 75322703767771552 84104723041144016 79601123413773520 110113010889208624 3715469692580659 -100205091708993536 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 226531061256735936 220676381741154304 201085723362092640 186223844591770016 186223844591770016 160778506697126720 179693625132082784 170911605858710336 219550481834311680 3602879701896397 -200410183417987072 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 106397541196627968 88495732677830240 86806882817566304 84555083003881056 87820192733724672 76110833702561376 87032062798934832 78362633516246624 96151852044360096 4728779608739021 -92323792361095168 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 443604563295993856 355784370562269184 357585810413217408 362539770003324928 376050568885436416 332365652499942592 373798769071751168 339121051940998336 388660647842073792 7656119366529843 -369295169444380672 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 3 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "7 0 -1 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "7 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "31525197391593472 0 0 0 0 -4503599627370496 -4953959590107546 -5404319552844595 0 0 0 0 0 0 0 0 -4503599627370496 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "21 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "31525197391593472 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -4503599627370496 -5404319552844595 -4953959590107546 0 0 0 0 0 0 0 0 0 0 -4953959590107546 0 -4503599627370496 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "10 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 >=\n"
         "10 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 >=\n"
         "17 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 0 >=\n"
         "58546795155816448 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -4503599627370496 -4953959590107546 -5404319552844595 0 0 0 0 0 0 0 0 0 0 -4953959590107546 0 0 0 0 0 0 0 0 0 0 0 -4953959590107546 -5404319552844595 0 0 >=\n"
         "3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 0 0 >=\n"
         "3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 0 0 0 0 0 0 0 -2 0 0 0 0 0 >=\n"
         "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 >=\n"
         "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "-15 0 0 0 0 0 0 0 0 0 1 -1 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "-15 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "-20 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "-20 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "-15 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 -1 -1 0 0 0 0 0 0 0 0 0 0 0 =\n"
         "1 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -3 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -3 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -126100789566373888 -23869078025063628 -5629499534213120 -8106479329266893 -25895697857380352 -4503599627370496 -32876277279804620 -9457559217478042 -9682739198846566 0 22517998136852480 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -144115188075855872 -56204923349583792 0 -38911100780481088 -105204087295374784 0 -144115188075855872 -5764607523034235 -10088063165309912 0 72057594037927936 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 9007199254740992 0 9007199254740992 8827055269646172 3422735716801577 9007199254740992 3242591731706757 2431943798780068 0 -4503599627370496 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 9007199254740992 9007199254740992 7836263351624663 9007199254740992 9007199254740992 9007199254740992 9007199254740992 8827055269646172 8106479329266893 0 -8106479329266893 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 110000420898524368 110563370851945680 98966601811466656 88720912659198768 88045372715093200 75322703767771552 84104723041144016 110113010889208624 114166250553842080 3715469692580659 -101330991615836160 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 226531061256735936 220676381741154304 201085723362092640 186223844591770016 186223844591770016 160778506697126720 179693625132082784 219550481834311680 228557681089052672 3602879701896397 -202661983231672320 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 106397541196627968 88495732677830240 86806882817566304 84555083003881056 87820192733724672 76110833702561376 87032062798934832 96151852044360096 100205091708993536 4728779608739021 -93449692267937792 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 443604563295993856 355784370562269184 357585810413217408 362539770003324928 376050568885436416 332365652499942592 373798769071751168 388660647842073792 406224686388818752 7656119366529843 -373798769071751168 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 -1 3 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 0 -1 =\n"
         "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 >=\n"
         "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 >=\n");
  Constraint_System cs;
  cs.ascii_load(ss);
  print_constraints(cs, "*** cs ***");

  Rational_Box known_result(cs.space_dimension());
  ss.str("+EUP -EM -UN space_dim 79\n"
         "[0, 10314880348817334512601175099915806371166908797205087714999656461/4472327120746785297653956552193407232008520545204266972149396275]\n"
         "[0, 13845204640026631030889305159390169337029989311711657705741484032/5852044125547558797457743823140996705204670473363875812631956685]\n"
         "[0, 6922602320013315515444652579695084668514994655855828852870742016/4472327120746785297653956552193407232008520545204266972149396275]\n"
         "[0, 1]\n"
         "[0, 183286621582592395603923918336706397030221465067768516480006942467094251874476342276273313773966732258762031104/201947514318058825266405552296969018502505571136310660027103675348978638253530036293093560899136162654074928785]\n"
         "[0, 748667973775216965289800480279583789489744816759688735086770433008145914407394803991979991720232554703997362949/288082953845409310261586095056492490791362249192985502452443896193756261634064606718440121711088142441174012160]\n"
         "[2686500548319859668267236271679836106854450377612806461753177278269968849965950567943501971086918159087023381675088683299453183/778446170153986271646293722222758610239588615644139185506718044413398598788584196557239652607927138993387106707838558560137600, 31525197391593472/5404319552844595]\n"
         "[808260464002815979451978388304660281069312131948424687841199686824624428575770834454261530624/151009237862627616327241484028267709860068350280703545777427164042459931581501950526059886939, 73859033888876134/5404319552844595]\n"
         "[226618556208645903758643992043214159409275947657289604887186627910330909797925643152677900801504922848811608768/10961195840757945196265067281093921849892738542919421265335678633895419160365563327020735054453549917239687515, 79624855970192365478639383910721447943227661124538323699857998144/1170408825109511759491548764628199341040934094672775162526391337]\n"
         "[15, 26541618656730788492879794636907149314409220374846107899952666048/1170408825109511759491548764628199341040934094672775162526391337]\n"
         "[0, 8985486280088112100506563167484159198795208954754480462056795993/1170408825109511759491548764628199341040934094672775162526391337]\n"
         "[0, 748667973775216965289800480279583789489744816759688735086770433008145914407394803991979991720232554703997362949/139580617693730330144431108207543093864914258318465321284436138492151206127797310314742739840929485352303263010]\n"
         "[0, 303804758953955866721752289291478575769374180673152882292480754701942640399851473710715528744543627330227738410675228287069127359/114636083423731409790347440941157613447471749368242157860749224890585062371407435860707842957895607705984937192636439527142281664]\n"
         "[0, 7]\n"
         "[0, 7]\n"
         "[0, 748667973775216965289800480279583789489744816759688735086770433008145914407394803991979991720232554703997362949/386268417790710553084111681668112997768232491914126126127983236005267549264797679992985537041321412971263426560]\n"
         "[0, 3]\n"
         "[0, 177573125946037698558604385378801413571089307288393133079217002458991879409204238029/17892588555239238988473154459237463544171166888738351339728888719297576255502553600]\n"
         "[25839794736957542736653447787390098039029695261748271831543709793310100059819748166779436608293786761483582251489687769847889922549/3692318603621377070171731880846638790832352647482537903463700684578188218693428418242417440634684128553797348741645618454197176800, 60]\n"
         "[15, 20]\n"
         "[0, 72409676610081820024337606323284033316057667404425726484298112/61528739464365799250640879039394801804157129845741815314149415]\n"
         "[0, 215364736487606238026344927308510412614242449964/134794645266934963747177709084220527189140255405]\n"
         "[0, 2061072662362948098854568832072986384021826066051896235110957056/3670355764874540374735204645644094629677039803631891367408505105]\n"
         "[0, 515268165590737024713642208018246596005456516512974058777739264/430701176250560594754486153275763612629099908920192707199045905]\n"
         "[0, 3661200520289306046447863764244677014442121649388/8769571208077881580806264580025357648475702951979]\n"
         "[0, 31525197391593472/5404319552844595]\n"
         "[0, 15762598695796736/2476979795053773]\n"
         "[0, 1]\n"
         "[0, 915300130072326511611965941061169253610530412347/914192110113563926269710132379456852481344786795]\n"
         "[100569907736012592200176731907593393653603286897200373375948598819289159112952295415766216387512943/146346253870074105858602763694412510197753270126660267092506527819743231998370774150586226021951840, 1423138307257267058304665020078992042801252847756788883057586059/184586218393097397751922637118184405412471389537225445942448245]\n"
         "[14869667571313552605650378924578127204258391821680034719538033247871471932211925243936500890280188256699229259926555073134812416/718752927277209687656244167506910570008434195163317640707227921525181116141905353375202615336471652237552905414680120645944371, 21857749969521291174186682493347326296602048361498167335484176256/430701176250560594754486153275763612629099908920192707199045905]\n"
         "[8570646096554564291866668301938830916818802465337146975839074062/1170408825109511759491548764628199341040934094672775162526391337, 21857749969521291174186682493347326296602048361498167335484176256/1292103528751681784263458459827290837887299726760578121597137715]\n"
         "[0, 8797647701304867238000079012369780913937948508972739599385139620161632864004094691936722362939144743295396038778/1399148734839437673237156659726880257722168016620759104314518136891683218621465660068206424960522725506715274445]\n"
         "[0, 2669624945007775364069874412248427045229215715349940210565159012362111738459851082209412940870318845779551526695718690690481550044170778794259142059204115616411333581967780864779/1708360668329758714025236540492916320511806002258624502201417631387139555375059087959692708852788041301919137087636450154229857124447755653498823376904941199764632073673925197824]\n"
         "[0, 3]\n"
         "[0, 3]\n"
         "[0, 3]\n"
         "[0, 4398823850652433619000039506184890456968974254486369799692569810080816432002047345968361181469572371647698019389/1157782275275578837083828258845188344809959076054449796563020342561686251725343246520483048742598889585601640032]\n"
         "[0, 177573125946037698558604385378801413571089307288393133079217002458991879409204238029/17892588555239238988473154459237463544171166888738351339728888719297576255502553600]\n"
         "[0, 4398823850652433619000039506184890456968974254486369799692569810080816432002047345968361181469572371647698019389/1052529341159617039636766879026295001378528414597855069798212223052291123121300381836309227297285312255918866432]\n"
         "[0, 7]\n"
         "[0, 1498223798318303919831024844733424211709929066732763569681630325246353050135890887826963426113547608433634486015043774792176591439896471154476282777/26847311172107165218056405640147044592053233844602064424295328321862465989246561483111745735713688610433228398430982602027530225437614337760952320]\n"
         "[0, 499407932772767973277008281577808070569976355577587856560543441748784350045296962608987808704515869477878162005014591597392197146632157051492094259/26847311172107165218056405640147044592053233844602064424295328321862465989246561483111745735713688610433228398430982602027530225437614337760952320]\n"
         "[0, 5]\n"
         "[1088990638376279656349232564077841878585326883393145232156543106725860769701036461875975422248087277562164759965636299994676381419779389192434465452555574437/2258962538374806529068089961097446198274435865824587927247946112063701506797953147532001062632455227539779726419333176316008587778726957833593452101170626560, 7771524006488802993376382138936142531043834044589401854804790150420533543300059/4977137688523269900666388869765991278013084676989364557386155543450133944795744]\n"
         "[0, 3/2]\n"
         "[0, 3/2]\n"
         "[1344731246397095180937221057427674208422663814958361282355618977488441268849128875802333363211497752676459590600393921949694634262748267846174416688274480521783/987516079534567299332969262619453896185534874711494006494411351584617022135972707236848778500132112834346494590435128956512560266700006817592445466535345334400, 2926486520230784671697951406927216601955418993295/524827521561835640889301183905776373669342216192]\n"
         "[0, 8098698173131643610422973994024390152024259063050436511914034781041627844575232/7928431521824465019198574989109348959534008394399277106209182654398428320014455]\n"
         "[255625743144916938376557465993431512631549899985942527668219134605026453252022742017194652158692271084961998287730522703741817071443625417096646865856016016637/1468623525604320066434070727098777983107130613489350358641431650251104389265591309698118507847541408759534492581891411442809473326529888133873151920707605626880, 7170392166438080056989470254311656820039851912472813793909407744/2245971024645116171876361734571755974833113225752842920420057075]\n"
         "[0, 226537011835850170920922349483199724532147106658753468864728245623542037610496/295286773686929687042241100901562780978972101417629942418684891714427504870407]\n"
         "[0, 29273397577908224/2476979795053773]\n"
         "[0, 58546795155816448/5404319552844595]\n"
         "[4886121758410660121399898393468498287761348147836794297860689782559815498292668383511688347221116029983439921364194244223876464951616007309127944670355058347958525880280511104/161787872311589039754537174148841335581508847196926049073414475652896143384975252871620319949599560382345933056706166004377459069700243126385807323566214512647413320655096595, 60]\n"
         "[19185326995483886226743054182167824240505411823442442486905223117672031508425218509427406136271309766838909450122/1399148734839437673237156659726880257722168016620759104314518136891683218621465660068206424960522725506715274445, 20]\n"
         "[0, 538537423212680385284373989330528551512140313847671962569697057921440092965209437981361342634620164343644389826716557499329872961929701779022177574622920411231181548734688467653205269453086868420694127457409/216325670355300629343398793111970587004244916598977668851214602484751463124754806790692265920288849668696149072929310323627620010574599881816934452674152407689569704749716632512928181545672403224920682659840]\n"
         "[0, 8530429548948577812392471568766864753432684932156578782588494538357728952510598196566154421789823375667005355303957367615430968404251800329750037977078972822217/987516079534567299332969262619453896185534874711494006494411351584617022135972707236848778500132112834346494590435128956512560266700006817592445466535345334400]\n"
         "[0, 10]\n"
         "[0, 3/2]\n"
         "[0, 12227674224491803626313043714346181343779060314673535520783949892529360855505503390416812664545409703371081188658346474560138706204060423720825144460158631811843/1468623525604320066434070727098777983107130613489350358641431650251104389265591309698118507847541408759534492581891411442809473326529888133873151920707605626880]\n"
         "[0, 3/2]\n"
         "[0, 1]\n"
         "[0, 1]\n"
         "[0, 5015518816216880836227562971795126288850074754377521725169510371984550419411570290926991422560225824756813804684469870576416363836800284814252084824351761391267887183715721855/939707401646611517828337186389124783204238566165786389129683484730581046324608742585765514020780418411169290173914166419503216856559045828015278273579257936090154907695639808]\n"
         "[0, 60]\n"
         "[103329071798316531709026909095463703501013053247478398826357779304145021771658399939262244128683394924564952448387225814185493208137888063349687824244782667478861/79269283303952994524846315029296066792799612173049976075893590468760030484709218718674284127122889338165671718089949371224177700480858989945121469063628471664640, 20]\n"
         "[0, 5]\n"
         "[0, 20]\n"
         "[0, +inf)\n"
         "[0, 8530429548948577812392471568766864753432684932156578782588494538357728952510598196566154421789823375667005355303957367615430968404251800329750037977078972822217/987516079534567299332969262619453896185534874711494006494411351584617022135972707236848778500132112834346494590435128956512560266700006817592445466535345334400]\n"
         "[0, 10]\n"
         "[0, 3/2]\n"
         "[0, 12227674224491803626313043714346181343779060314673535520783949892529360855505503390416812664545409703371081188658346474560138706204060423720825144460158631811843/1468623525604320066434070727098777983107130613489350358641431650251104389265591309698118507847541408759534492581891411442809473326529888133873151920707605626880]\n"
         "[0, 3/2]\n"
         "[0, 1]\n"
         "[0, 5015518816216880836227562971795126288850074754377521725169510371984550419411570290926991422560225824756813804684469870576416363836800284814252084824351761391267887183715721855/939707401646611517828337186389124783204238566165786389129683484730581046324608742585765514020780418411169290173914166419503216856559045828015278273579257936090154907695639808]\n"
         "[0, 1003103763243376167245512594359025257770014950875504345033902074396910083882314058185398284512045164951362760936893974115283272767360056962850416964870352278253577436743144371/205027069450169761565873382712224453082020642139182002012346602023286123956967691684038960374515291662700700381411197137724083103996816374687476082968091232453611203577584512]\n"
         "[0, +inf)\n"
         "[0, +inf)\n");
  known_result.ascii_load(ss);
  print_constraints(known_result, "*** known_result ***");

  TBox box(cs.space_dimension());

  bool ok = false;

  typedef TBox::interval_type::boundary_type boundary_type;
  if (std::numeric_limits<boundary_type>::is_exact
      && !std::numeric_limits<boundary_type>::is_integer) {
    // With interval boundaries made of rational numbers, this
    // refinement instance either does not terminate or terminates
    // very slowly: we use a watchdog timer.
    try {
      // Set a 0.5 seconds timeout.
      Parma_Watchdog_Library::Watchdog
        w(50, abandon_expensive_computations, t);

      box.refine_with_constraints(cs);

      // We should never get here.
      abandon_expensive_computations = 0;
      nout << "unexpected termination" << endl;
      ok = false;
    }
    catch (const Timeout&) {
      abandon_expensive_computations = 0;
      nout << "timeout, as expected" << endl;

      // The box will have been shrunk, but no bounds on the distances
      // can be set: we check containment though.
      ok = check_result(box, known_result, "+inf", "+inf", "+inf");
    }
    catch (...) {
      abandon_expensive_computations = 0;
      nout << "unexpected exception" << endl;
      ok = false;
    }
  }
  else {
    // With interval boundaries other than rational numbers, this instance
    // of refinement terminates: no timer is necessary.
    box.refine_with_constraints(cs);

    ok = check_result(box, known_result, "624", "158", "121");
  }

  print_constraints(box, "*** box.refine_with_constraints(cs) ***");

  return ok;
}

} // namespace

BEGIN_MAIN
// Temporarily set to always fail: this is meant to be a test for
// constraint propagation, rather than syntactical refinement.
//  DO_TEST(test01);
  DO_TEST_F(test01);
  // Temporarily always failing.
  // (Wait for ascii_dump/load to stabilize before correct/restore test.)
  // DO_TEST_F32(test02);
  DO_TEST_F(test02);
END_MAIN
