/*
 * Copyright © 2022 Soren Stoutner <soren@stoutner.com>.
 *
 * This file is part of Privacy Browser PC <https://www.stoutner.com/privacy-browser-pc>.
 *
 * Privacy Browser PC is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Privacy Browser PC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Privacy Browser PC.  If not, see <http://www.gnu.org/licenses/>.
 */

// Application headers.
#include "DomainsDatabase.h"
#include "helpers/UserAgentHelper.h"

// Define the private static schema constants.
const int DomainsDatabase::SCHEMA_VERSION = 5;

// Define the public static database constants.
const QString DomainsDatabase::CONNECTION_NAME = "domains_database";
const QString DomainsDatabase::DOMAINS_TABLE = "domains";

// Define the public static database field names.
const QString DomainsDatabase::_ID = "_id";
const QString DomainsDatabase::DOMAIN_NAME = "domain_name";
const QString DomainsDatabase::JAVASCRIPT = "javascript";
const QString DomainsDatabase::LOCAL_STORAGE = "local_storage";
const QString DomainsDatabase::DOM_STORAGE = "dom_storage";
const QString DomainsDatabase::USER_AGENT = "user_agent";
const QString DomainsDatabase::ZOOM_FACTOR = "zoom_factor";
const QString DomainsDatabase::CUSTOM_ZOOM_FACTOR = "custom_zoom_factor";

// Construct the class.
DomainsDatabase::DomainsDatabase() {}

void DomainsDatabase::addDatabase()
{
    // Add the domain settings database.
    QSqlDatabase domainsDatabase = QSqlDatabase::addDatabase(QStringLiteral("QSQLITE"), CONNECTION_NAME);

    // Set the database name.
    domainsDatabase.setDatabaseName(QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/domains.db");

    // Open the database.
    if (domainsDatabase.open())  // Opening the database succeeded.
    {
        // Check to see if the domains table already exists.
        if (domainsDatabase.tables().contains(DOMAINS_TABLE))  // The domains table already exists.
        {
            // Query the database schema version.
            QSqlQuery schemaVersionQuery = domainsDatabase.exec(QStringLiteral("PRAGMA user_version"));

            // Move to the first record.
            schemaVersionQuery.first();

            // Get the current schema version.
            int currentSchemaVersion = schemaVersionQuery.value(0).toInt();

            // Check to see if the schema has been updated.
            if (currentSchemaVersion < SCHEMA_VERSION)
            {
                // Run the schema update code.
                switch (currentSchemaVersion)
                {
                    // Upgrade from schema version 0 to schema version 1.
                    case 0:
                    {
                        // Add the JavaScript column.
                        domainsDatabase.exec("ALTER TABLE " + DOMAINS_TABLE + " ADD COLUMN " + JAVASCRIPT + " INTEGER DEFAULT 0");

                        // Fallthrough to the next case.
                        [[fallthrough]];
                    }

                    // Upgrade from schema version 1 to schema version 2.
                    case 1:
                    {
                        // Add the User Agent column.
                        domainsDatabase.exec("ALTER TABLE " + DOMAINS_TABLE + " ADD COLUMN " + USER_AGENT + " TEXT DEFAULT '" + UserAgentHelper::SYSTEM_DEFAULT_DATABASE + "'");

                        // Fallthrough to the next case.
                        [[fallthrough]];
                    }

                    // Upgrade from schema version 2 to schema version 3.
                    case 2:
                    {
                        // Add the Zoom Factor columns.
                        domainsDatabase.exec("ALTER TABLE " + DOMAINS_TABLE + " ADD COLUMN " + ZOOM_FACTOR + " INTEGER DEFAULT 0");
                        domainsDatabase.exec("ALTER TABLE " + DOMAINS_TABLE + " ADD COLUMN " + CUSTOM_ZOOM_FACTOR + " REAL DEFAULT 1.0");

                        // Fallthrough to the next case.
                        [[fallthrough]];
                    }

                    // Upgrade from schema version 3 to schema version 4.
                    case 3:
                    {
                        // Add the DOM Storage column.
                        domainsDatabase.exec("ALTER TABLE " + DOMAINS_TABLE + " ADD COLUMN " + DOM_STORAGE + " INTEGER DEFAULT 0");

                        // Fallthrough to the next case.
                        [[fallthrough]];
                    }

                    // Upgrade from schema version 4 to schema version 5.
                    case 4:
                    {
                        // Add the Local Storage column.
                        domainsDatabase.exec("ALTER TABLE " + DOMAINS_TABLE + " ADD COLUMN " + LOCAL_STORAGE + " INTEGER DEFAULT 0");

                        // Fallthrough to the next case.
                        // [[fallthrough]];
                    }
                }

                // Update the schema version.
                domainsDatabase.exec("PRAGMA user_version = " + QString::number(SCHEMA_VERSION));
            }
        }
        else  // The domains table does not exist.
        {
            // Instantiate a create table query.
            QSqlQuery createTableQuery(domainsDatabase);

            // Prepare the create table query.
            createTableQuery.prepare("CREATE TABLE " + DOMAINS_TABLE + "(" +
                _ID + " INTEGER PRIMARY KEY, " +
                DOMAIN_NAME + " TEXT, " +
                JAVASCRIPT + " INTEGER DEFAULT 0, " +
                LOCAL_STORAGE + " INTEGER DEFAULT 0, " +
                DOM_STORAGE + " INTEGER DEFAULT 0, " +
                USER_AGENT + " TEXT DEFAULT '" + UserAgentHelper::SYSTEM_DEFAULT_DATABASE + "', " +
                ZOOM_FACTOR + " INTEGER DEFAULT 0, " +
                CUSTOM_ZOOM_FACTOR + " REAL DEFAULT 1.0)"
            );

            // Execute the query.
            if (!createTableQuery.exec())
            {
                // Log any errors.
                qDebug().noquote().nospace() << "Error creating table:  " << domainsDatabase.lastError();
            }

            // Set the schema version.
            domainsDatabase.exec("PRAGMA user_version = " + QString::number(SCHEMA_VERSION));
        }
    }
    else  // Opening the database failed.
    {
        // Write the last database error message to the debug output.
        qDebug().noquote().nospace() << "Error opening database:  " << domainsDatabase.lastError();
    }
};

QSqlQuery DomainsDatabase::getDomainQuery(const QString &hostname)
{
    // Get a handle for the domains database.
    QSqlDatabase domainsDatabase = QSqlDatabase::database(CONNECTION_NAME);

    // Instantiate the all domain names query.
    QSqlQuery allDomainNamesQuery(domainsDatabase);

    // Set the query to be forward only (increases performance while iterating over the query).
    allDomainNamesQuery.setForwardOnly(true);

    // Prepare the query.
    allDomainNamesQuery.prepare("SELECT " + _ID + "," + DOMAIN_NAME + " FROM " + DOMAINS_TABLE);

    // Execute the query.
    allDomainNamesQuery.exec();

    // Create a domains settings map.
    QMap<QString, int> domainSettingsMap;

    // Populate the domain settings map.
    while (allDomainNamesQuery.next())
    {
        // Add the domain name and database ID to the map.
        domainSettingsMap.insert(allDomainNamesQuery.record().field(DOMAIN_NAME).value().toString(), allDomainNamesQuery.record().field(_ID).value().toInt());
    }

    // Initialize the database ID tracker.
    int databaseId = -1;

    // Get the database ID if the hostname is found in the domain settings set.
    if (domainSettingsMap.contains(hostname))
    {
        databaseId = domainSettingsMap.value(hostname);
    }

    // Create a subdomain string.
    QString subdomain = hostname;

    // Check all the subdomains of the hostname.
    while ((databaseId == -1) && subdomain.contains("."))  // Stop checking when a match is found or there are no more `.` in the hostname.
    {
        // Check to see if the domain settings map contains the subdomain with a `*.` prepended.
        if (domainSettingsMap.contains("*." + subdomain))
        {
            // Get the database ID.
            databaseId = domainSettingsMap.value("*." + subdomain);
        }

        // Strip out the first subdomain.
        subdomain = subdomain.section('.', 1);
    }

    // Instantiate the domain lookup query.
    QSqlQuery domainLookupQuery(domainsDatabase);

    // Prepare the domain lookup query.
    domainLookupQuery.prepare("SELECT * FROM " + DOMAINS_TABLE + " WHERE " + _ID + " = " + QString::number(databaseId));

    // Execute the query.
    domainLookupQuery.exec();

    // Move to the first entry.
    domainLookupQuery.first();

    // Return the query.
    return domainLookupQuery;
}
