/*
    Debugger.m

    Implementation of the Debugger class for the ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "Debugger.h"

#import <AppKit/NSTextField.h>
#import <AppKit/NSFont.h>
#import <AppKit/NSColor.h>

#import <Foundation/NSString.h>
#import <Foundation/NSBundle.h>
#import <Foundation/NSNotification.h>

#import "DebuggerProtocol.h"
#import "DebuggerLoader.h"
#import "ProjectDocument.h"

@implementation Debugger

- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  TEST_RELEASE(debuggerType);
  TEST_RELEASE(dbg);
  TEST_RELEASE(view);

  [super dealloc];
}

- initWithDocument: (ProjectDocument *) aDocument
      debuggerType: (NSString *) aDebuggerType
{
  if ([self initWithDocument: aDocument])
    {
      ASSIGN(dbg, [[DebuggerLoader shared]
        debuggerForTypeID: aDebuggerType project: document]);
      if (dbg != nil)
        {
          ASSIGN(view, [dbg view]);

          [[NSNotificationCenter defaultCenter]
            addObserver: self
               selector: @selector(projectModuleChanged:)
                   name: CurrentProjectModuleDidChangeNotification
                 object: nil];
        }
      else
        {
          NSTextField * tf;

          NSLog(_(@"Warning: debugger of type \"%@\" not supported. "
            @"Debugging not possible for project %@."), aDebuggerType,
            [document projectName]);

          tf = [[NSTextField alloc]
            initWithFrame: NSMakeRect(0, 0, 100, 20)];
          [tf setEditable: NO];
          [tf setSelectable: NO];
          [tf setDrawsBackground: NO];
          [tf setBordered: NO];
          [tf setBezeled: NO];
          [tf setAlignment: NSCenterTextAlignment];
          [tf setTextColor: [NSColor darkGrayColor]];
          [tf setFont: [NSFont userFontOfSize: 18]];
          [tf setStringValue: _(@"Debugger Not Available")];

          view = tf;
        }

      return self;
    }
  else
    {
      return nil;
    }
}

- (NSString *) debuggerType
{
  return debuggerType;
}

- (NSView *) view
{
  return view;
}

- (void) projectModuleChanged: (NSNotification *) notif
{
  if ([[notif userInfo] objectForKey: @"Module"] == self)
    {
      [dbg activate: self];
    }
  else
    {
      [dbg deactivate: self];
    }
}

@end
