/*
    FileManager.m

    Implementation of the FileManager class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "FileManager.h"

#import <Foundation/NSString.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSBundle.h>
#import <Foundation/NSFileManager.h>
#import <Foundation/NSNotification.h>
#import <Foundation/NSUserDefaults.h>

#import <AppKit/NSBrowser.h>
#import <AppKit/NSBrowserCell.h>
#import <AppKit/NSMatrix.h>
#import <AppKit/NSPanel.h>
#import <AppKit/NSWorkspace.h>
#import <AppKit/NSImage.h>
#import <AppKit/NSDragging.h>
#import <AppKit/NSPasteboard.h>
#import <AppKit/NSTextField.h>

#import "ProjectDocument.h"
#import "ProjectType.h"
#import "ProjectImageView.h"

NSString * const ProjectFilesPboardType = @"ProjectFilesPboardType";

static inline void
SetTextFieldEditable(NSTextField * tf, BOOL flag, id target)
{
  if ([tf isEditable] != flag)
    {
      [tf setEditable: flag];
      [tf setSelectable: flag];
      [tf setDrawsBackground: flag];
      if (flag)
        [tf setTarget: target];
      else
        [tf setTarget: nil];
    }
}

static NSString *
MakeSizeStringFromValue(unsigned long long size)
{
  if (size < 1024)
    {
      return [NSString stringWithFormat: @"%i bytes", size];
    }
  else if (size < 1024 * 1024)
    {
      return [NSString stringWithFormat: @"%.2f kB", (double) size / 1024];
    }
  else if (size < 1024 * 1024 * 1024)
    {
      return [NSString stringWithFormat: @"%.2f MB", (double) size /
      (1024 * 1024)];
    }
  else
    {
      return [NSString stringWithFormat: @"%.3f GB",
        (double) size / (1024 * 1024 * 1024)];
    }
}

@interface FileManager (Private)

- (unsigned long long) measureSizeOfCategory: (NSString *) category;

- (unsigned long long) measureSizeOfFile: (NSString *) filename
                              inCategory: (NSString *) category;

@end

@implementation FileManager (Private)

- (unsigned long long) measureSizeOfCategory: (NSString *) category
{
  unsigned long long size = 0;
  NSEnumerator * e;
  NSString * entry;

  e = [[document filesInCategory: category] objectEnumerator];
  while ((entry = [e nextObject]) != nil)
    {
      size += [self measureSizeOfFile: entry inCategory: category];
    }

  e = [[document subcategoriesInCategory: category] objectEnumerator];
  while ((entry = [e nextObject]) != nil)
    {
      size += [self measureSizeOfCategory: [category
        stringByAppendingPathComponent: entry]];
    }

  return size;
}

- (unsigned long long) measureSizeOfFile: (NSString *) filename
                              inCategory: (NSString *) category
{
  NSString * path = [[document projectType]
    pathToFile: filename inCategory: category];
  NSFileManager * fm = [NSFileManager defaultManager];
  NSDictionary * fattrs = [fm fileAttributesAtPath: path traverseLink: NO];
  unsigned long long size = [fattrs fileSize];

  // if it's a directory count the underlying files as well
  if ([[fattrs fileType] isEqualToString: NSFileTypeDirectory])
    {
      NSDirectoryEnumerator * de = [fm enumeratorAtPath: path];

      while ([de nextObject] != nil)
        {
          size += [[de fileAttributes] fileSize];
        }
    }

  return size;
}

@end

@implementation FileManager

- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  [super dealloc];
}

- initWithDocument: (ProjectDocument *) doc
{
  if ([super initWithDocument: doc])
    {
      [[NSNotificationCenter defaultCenter]
        addObserver: self
           selector: @selector(filesChanged:)
               name: ProjectFilesDidChangeNotification
             object: document];

      return self;
    }
  else
    {
      return nil;
    }
}

- (void) awakeFromNib
{
  [super awakeFromNib];

  [browser setDoubleAction: @selector(openFile:)];
  [browser setMaxVisibleColumns: 4];
  [self selectFile: browser];
}

- (void) openFile: (id)sender
{
  if ([[browser selectedCell] isLeaf] == YES)
    {
      NSString * path = [browser path];
      NSString * filename = [path lastPathComponent];

      if ([document openFile: filename
                  inCategory: [path stringByDeletingLastPathComponent]] == NO)
        {
          NSRunAlertPanel(_(@"Unable to open file"),
            _(@"Cannot open file %@."),
            nil, nil, nil, filename);
        }
    }
}

- (void)      browser: (NSBrowser *) sender
  createRowsForColumn: (int) column
             inMatrix: (NSMatrix *) matrix
{
  NSString * path = [browser path];
  NSArray * categories, * files;
  NSEnumerator * e;
  NSString * name;
  NSFont * boldFont = [NSFont boldSystemFontOfSize: 0];

  categories = [[document subcategoriesInCategory: path]
    sortedArrayUsingSelector: @selector (caseInsensitiveCompare:)];
  files = [[document filesInCategory: path]
    sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];

  e = [categories objectEnumerator];
  while ((name = [e nextObject]) != nil)
    {
      NSBrowserCell * cell;

      [matrix addRow];
      cell = [matrix cellAtRow: [matrix numberOfRows] - 1 column: 0];
      [cell setTitle: name];
      [cell setFont: boldFont];
      [cell setLeaf: NO];
    }

  e = [files objectEnumerator];
  while ((name = [e nextObject]) != nil)
    {
      NSBrowserCell * cell;

      [matrix addRow];
      cell = [matrix cellAtRow: [matrix numberOfRows] - 1 column: 0];
      [cell setTitle: name];
      [cell setLeaf: YES];
    }
}

- (NSString *) browser: (NSBrowser *)sender titleOfColumn: (int)column
{
  if (column == 0)
    {
      return [document projectName];
    }
  else
    {
      return [[sender selectedCellInColumn: column - 1] title];
    }
}

- (void) selectFile: sender
{
  // this needs special treatment
  if ([[browser path] isEqualToString: @"/"])
    {
      [fileIcon setImage: [NSImage imageNamed: @"pmproj"]];
      [fileIcon setShowsLinkIndicator: NO];
      [fileName setStringValue: [document projectName]];
      [filePath setStringValue: [document projectDirectory]];
      [fileSize setStringValue: MakeSizeStringFromValue([self
        measureSizeOfCategory: @"/"])];
      [fileType setStringValue: _(@"Project")];
      [lastModified setStringValue: nil];
    }
  else
    {
      if ([[browser selectedCells] count] > 1)
        {
          unsigned long long size = 0;
          NSArray * selectedCategories = [self selectedCategories];
          NSArray * selectedFiles = [self selectedFiles];
          NSEnumerator * e;
          NSString * entry;
          NSString * containingCategory = [self containingCategory];

          [fileIcon setImage: [NSImage imageNamed: @"MultipleSelection"]];
          [fileIcon setShowsLinkIndicator: NO];
          [fileName setStringValue: [NSString stringWithFormat:
            _(@"%i Elements"), [[browser selectedCells] count]]];
          SetTextFieldEditable(fileName, NO, self);
          [filePath setStringValue: nil];

          if ([selectedCategories count] > 1)
            {
              e = [selectedCategories objectEnumerator];

              while ((entry = [e nextObject]) != nil)
                {
                  size += [self measureSizeOfCategory: entry];
                }
            }

          e = [selectedFiles objectEnumerator];
          while ((entry = [e nextObject]) != nil)
            {
              size += [self measureSizeOfFile: entry
                                   inCategory: containingCategory];
            }

          [fileSize setStringValue: MakeSizeStringFromValue(size)];
          [fileType setStringValue: nil];
          [lastModified setStringValue: nil];
        }
      else
        {
          if ([[browser selectedCell] isLeaf])
            {
              NSFileManager * fm = [NSFileManager defaultManager];
              NSString * path;
              NSDictionary * fattrs;

              path = [[document projectType]
                pathToFile: [[browser path] lastPathComponent]
                inCategory: [[browser path] stringByDeletingLastPathComponent]];
              fattrs = [fm fileAttributesAtPath: path traverseLink: NO];

              [fileIcon setImage: [[NSWorkspace sharedWorkspace]
                iconForFile: path]];
              SetTextFieldEditable(fileName, YES, self);
              [fileName setStringValue: [path lastPathComponent]];
              [fileSize setStringValue:
                MakeSizeStringFromValue([fattrs fileSize])];

              if ([[fattrs fileType] isEqualToString: NSFileTypeDirectory])
                {
                  [fileType setStringValue: _(@"Directory")];
                  [fileIcon setShowsLinkIndicator: NO];
                  [filePath setStringValue: path];
                }
              else if ([[fattrs fileType] isEqualToString:
                NSFileTypeSymbolicLink])
                {
                  [fileType setStringValue: [NSString stringWithFormat:
                    _(@"Link To File"), [fm pathContentOfSymbolicLinkAtPath:
                    path]]];
                  [fileIcon setShowsLinkIndicator: YES];

                   // indicate the destination path for symbolic links
                  [filePath setStringValue: [fm
                    pathContentOfSymbolicLinkAtPath: path]];
                }
              else
                {
                  [fileType setStringValue: _(@"File")];
                  [fileIcon setShowsLinkIndicator: NO];
                  [filePath setStringValue: path];
                }

              [lastModified setObjectValue: [fattrs fileModificationDate]];
            }
          else
            {
              NSImage * categoryIcon;

              categoryIcon = [document iconForCategory: [browser path]];
              if (categoryIcon == nil)
                {
                  categoryIcon = [NSImage imageNamed: @"common_Folder"];
                }

              [fileIcon setImage: categoryIcon];
              [fileIcon setShowsLinkIndicator: NO];
              [fileName setStringValue: [[browser path] lastPathComponent]];
              SetTextFieldEditable(fileName, [[document projectType]
                canDeleteCategory: [browser path]], self);
              [filePath setStringValue: [[document projectType]
                pathToFile: nil inCategory: [browser path]]];
              [fileSize setStringValue: MakeSizeStringFromValue(
                [self measureSizeOfCategory: [[self selectedCategories]
                objectAtIndex: 0]])];
              [fileType setStringValue: _(@"Project Category")];
              [lastModified setStringValue: nil];
            }
        }
    }
}

/**
 * Instructs the browser to set it's path to `aPath', selects name
 * of the entry in the fileName text field and allows the user to
 * edit it.
 */
- (void) selectAndEditNameAtPath: (NSString *) aPath
{
  [browser setPath: aPath];
  [self selectFile: nil];
  [fileName selectText: nil];
}

- (void) changeName: sender
{
  NSString * newName = [fileName stringValue];

  if (![newName isEqualToString: [[browser path] lastPathComponent]])
    {
      NSString * previousPath = [[browser path] retain];
      BOOL result;

      if ([[browser selectedCell] isLeaf])
        {
          result = [document
            renameFile: [[browser path] lastPathComponent]
            inCategory: [[browser path] stringByDeletingLastPathComponent]
                toName: newName];
        }
      else
        {
          result = [document
            renameCategory: [[browser path] lastPathComponent]
                inCategory: [[browser path] stringByDeletingLastPathComponent]
                    toName: newName];
        }

      if (result == YES)
        {
          [browser setPath: [[previousPath stringByDeletingLastPathComponent]
            stringByAppendingPathComponent: newName]];
          [self selectFile: browser];
        }
      [previousPath release];
    }
}

/**
 * Returns the name of the file that is currently selected, otherwise nil.
 */
- (NSArray *) selectedFiles
{
  NSMutableArray * array = [NSMutableArray array];
  NSEnumerator * e = [[browser selectedCells] objectEnumerator];
  NSBrowserCell * cell;

  while ((cell = [e nextObject]) != nil)
    {
      if ([cell isLeaf])
        {
          [array addObject: [cell title]];
        }
    }

  return ([array count] > 0) ? array : nil;
}

/**
 * Returns a category path to the currently selected category, or if
 * a file is selected, the category in which the file resides.
 */
- (NSArray *) selectedCategories
{
  NSString * path = [browser path];

  if ([path isEqualToString: @"/"])
    {
      return [NSArray arrayWithObject: @"/"];
    }
  else
    {
      NSMutableArray * array = [NSMutableArray array];
      NSEnumerator * e = [[browser selectedCells] objectEnumerator];
      NSBrowserCell * cell;

      path = [path stringByDeletingLastPathComponent];
      while ((cell = [e nextObject]) != nil)
        {
          if (![cell isLeaf])
            {
              [array addObject: [path stringByAppendingPathComponent:
                [cell title]]];
            }
        }

      if ([array count] == 0)
        {
          return [NSArray arrayWithObject: path];
        }
      else
        {
          return [[array copy] autorelease];
        }
    }
}

- (NSString *) containingCategory
{
  NSArray * categories;

  categories = [self selectedCategories];
  if ([categories count] == 1)
    {
      return [categories objectAtIndex: 0];
    }
  else
    {
      return [[categories objectAtIndex: 0]
        stringByDeletingLastPathComponent];
    }
}

- (void) filesChanged: (NSNotification *) notif
{
  NSString * path = [[browser path] stringByStandardizingPath];

  if ([path isEqualToString: @"/"])
    {
      [browser reloadColumn: 0];
    }
  else
    {
      [browser reloadColumn: [[path pathComponents] count] - 2];
      [browser setPath: path];
      [self selectFile: browser];
    }
}

/**
 * Instructs the file manager to perform a drag operation. The drag
 * operation is specified by `sender'. The operation source is fully
 * specified by the `sender' argument, the destination is the current
 * file browser path.
 */
- (BOOL) performDragOperation: (id <NSDraggingInfo>) sender
{
  NSString * destCategory = [self containingCategory];
  NSPasteboard * pb;
  int operation;
  NSUserDefaults * df = [NSUserDefaults standardUserDefaults];
  BOOL ask = ![df boolForKey: @"DontAskWhenMoving"];

  NSDictionary * projectFilesData;

  if ([sender draggingSourceOperationMask] & NSDragOperationMove)
    {
      operation = NSDragOperationMove;
    }
  else if ([sender draggingSourceOperationMask] & NSDragOperationLink)
    {
      operation = NSDragOperationLink;
    }
  else
    {
      operation = NSDragOperationCopy;
    }

  pb = [sender draggingPasteboard];
  // is it one of our project files?
  projectFilesData = [pb propertyListForType: ProjectFilesPboardType];
  if (projectFilesData != nil && [[projectFilesData objectForKey: @"Project"]
    isEqualToString: [document fileName]])
    {
      NSString * srcCategory;
      NSArray * filenames;
      NSEnumerator * e;
      NSString * filename;

      if (ask)
        {
          NSString * title, * message;

          switch (operation)
            {
            case NSDragOperationMove:
              title = _(@"Really move files?");
              message = _(@"Really move the selected files to category %@?");
              break;
            case NSDragOperationLink:
              title = _(@"Really link files?");
              message = _(@"Really link the selected files from category %@?");
              break;
            default:
              title = _(@"Really copy files?");
              message = _(@"Really copy the selected files to category %@?");
              break;
            }

          if (NSRunAlertPanel(title, message, _(@"Yes"), _(@"Cancel"), nil,
            [destCategory lastPathComponent]) != NSAlertDefaultReturn)
            {
              return NO;
            }
        }

      srcCategory = [projectFilesData objectForKey: @"Category"];
      filenames = [projectFilesData objectForKey: @"Contents"];

      e = [filenames objectEnumerator];
      while ((filename = [e nextObject]) != nil)
        {
          switch (operation)
            {
            case NSDragOperationMove:
              if (![document moveFile: filename
                           inCategory: srcCategory
                           toCategory: destCategory])
                {
                  return NO;
                }
              break;
            case NSDragOperationLink:
              if (![document linkFile: filename
                           inCategory: srcCategory
                           toCategory: destCategory])
                {
                  return NO;
                }
              break;
            default:
              if (![document copyFile: filename
                           inCategory: srcCategory
                           toCategory: destCategory])
                {
                  return NO;
                }
              break;
            }
        }
    }
  // no, the file originates from somewhere outside - import it
  else
    {
      NSEnumerator * e;
      NSString * filepath;

      if (ask)
        {
          if (NSRunAlertPanel(_(@"Really import files?"),
            _(@"Really copy the selected files into the project?"),
            _(@"Yes"), _(@"Cancel"), nil,
            [destCategory lastPathComponent]) != NSAlertDefaultReturn)
            {
              return NO;
            }
        }

      e = [[pb propertyListForType: NSFilenamesPboardType] objectEnumerator];
      while ((filepath = [e nextObject]) != nil)
        {
          /* N.B. when importing we ignore the NSDragOperationMove
           * possibility - we always only copy or link the files.
           * This is for safety reasons: in case the user dragged
           * a file from the file viewer and forgot to hit `Command'
           * we won't delete the original files. */
          if (![document addFile: filepath
                      toCategory: destCategory
                            link: operation == NSDragOperationLink])
            {
              return NO;
            }
        }
    }

  return YES;
}

@end
