
#import "GDBMIParser.h"

#import <Foundation/NSString.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSBundle.h>
#import <Foundation/NSException.h>

NSString * const GDBMIParsingException = @"GDBMIParsingException";

#define RCHECK(__index__, __length__) \
  if ((__index__) >= (__length__)) \
    [NSException raise: GDBMIParsingException \
                format: _(@"Unexpected end of GDB MI reply.")]

static id
ParseGDBMIElement(NSString * aString, unsigned int * index);

static inline BOOL
IsPlainStringTerminator(unichar c)
{
  return (c == '=' || c == '{' || c == '}' || c == '[' ||
          c == ']' || c == ',');
}

static void
ParseGDBMIStringQuotedStringElement(NSString * aString,
                                    unsigned int * index,
                                    NSString ** stringPtr)
{
  unsigned int length = [aString length];
  NSRange r;
  unichar c;
  NSMutableString * str;

  r = NSMakeRange((*index) + 1, 0);
  RCHECK(NSMaxRange(r), length);

  while ((c = [aString characterAtIndex: NSMaxRange(r)]) != '"')
    {
      switch (c)
        {
        // eat escapes
        case '\\':
          r.length += 2;
          break;
        default:
          r.length++;
          break;
        }

      RCHECK(NSMaxRange(r), length);
    }

  *index = NSMaxRange(r);

  *stringPtr = [aString substringWithRange: r];
}

static void
ParseGDBMIStringPlainStringElement(NSString * aString,
                                   unsigned int * index,
                                   NSString ** stringPtr)
{
  NSRange r;

  for (r = NSMakeRange(*index, 0);
       !IsPlainStringTerminator([aString characterAtIndex: NSMaxRange(r)]);
       r.length++);

  *index = NSMaxRange(r);
  *stringPtr = [aString substringWithRange: r];
}

static inline void
ParseGDBMIStringElement(NSString * aString,
                        unsigned int * index,
                        NSString ** stringPtr)
{
  if ([aString characterAtIndex: *index] == '"')
    {
      ParseGDBMIStringQuotedStringElement(aString, index, stringPtr);
    }
  else
    {
      ParseGDBMIStringPlainStringElement(aString, index, stringPtr);
    }
}

static void
ParseGDBMIArrayElement(NSString * aString, unsigned int * index, id * anObject)
{
  id object;
  unsigned int i = *index;

  object = ParseGDBMIElement(aString, &i);
  // skip the stupid '=' construct - how can array elements be keyed??
  if ([aString characterAtIndex: i] == '=')
    {
      object = ParseGDBMIElement(aString, &i);
    }
  i++;
  RCHECK(i, [aString length]);

  *index = i;
  *anObject = object;
}

static void
ParseGDBMIDictionaryElement(NSString * aString, unsigned int * index,
                                  id * aKey, id * anObject)
{
  id key, object;
  unsigned int i = *index;

  key = ParseGDBMIElement(aString, &i);
  if ([aString characterAtIndex: i] != '=')
    {
      [NSException raise: GDBMIParsingException
                  format: _(@"Parse error in MI reply at index %i: "
        @"expected \"=\", got \"%c\"."), i, [aString characterAtIndex: i]];
    }
  i++;
  RCHECK(i, [aString length]);
  object = ParseGDBMIElement(aString, &i);

  *index = i;
  *aKey = key;
  *anObject = object;
}

static id
ParseGDBMIElement(NSString * aString, unsigned int * index)
{
  unsigned int i = *index;
  unsigned int length = [aString length];

  switch ([aString characterAtIndex: i])
    {
    // a dictionary
    case '{':
      {
        NSMutableDictionary * dict = [NSMutableDictionary dictionary];
        id key, object;

        i++;
        RCHECK(i, length);

        if ([aString characterAtIndex: i] == '}')
          {
            i++;
            *index = i;

            return dict;
          }

        ParseGDBMIDictionaryElement(aString, &i, &key, &object);
        [dict setObject: object forKey: key];

        while ([aString characterAtIndex: i] != '}')
          {
            if ([aString characterAtIndex: i] != ',')
              {
                [NSException raise: GDBMIParsingException
                            format: _(@"Error parsing GDB MI reply at %i: "
                  @"comma expected."), i];
              }
            i++;
            RCHECK(i, length);

            ParseGDBMIDictionaryElement(aString, &i, &key, &object);
            [dict setObject: object forKey: key];
          }

        i++;
        *index = i;

        return dict;
      }
      break;

    // an array
    case '[':
      {
        NSMutableArray * array = [NSMutableArray array];
        id object;

        i++;
        RCHECK(i, length);

        if ([aString characterAtIndex: i] == ']')
          {
            i++;
            *index = i;

            return array;
          }

        ParseGDBMIArrayElement(aString, &i, &object);
        [array addObject: object];

        while ([aString characterAtIndex: i] != ']')
          {
            if ([aString characterAtIndex: i] != ',')
              {
                [NSException raise: GDBMIParsingException
                            format: _(@"Error parsing GDB MI reply at %i: "
                  @"comma expected."), i];
              }
            i++;
            RCHECK(i, length);

            ParseGDBMIArrayElement(aString, &i, &object);
            [array addObject: object];
          }

        return array;
      }
      break;

    // a string
    default:
      {
        NSString * string;

        ParseGDBMIStringElement(aString, &i, &string);

        i++;
        *index = i;

        return string;
      }
    }
}

/**
 * Parses a GDB MI reply and returns it as an object. GDB MI replies
 * correspond roughly to OpenStep property lists, so the possible
 * returned objects are a string, an array and a dictionary, composed
 * of simmilar objects nested to arbitrary depth.
 */
id
ParseGDBMI(NSString * aString)
{
  unsigned int i = 0;

  return ParseGDBMIElement(aString, &i);
}
