/*
    ProjectDocument.m

    Implementation of the ProjectDocument class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "ProjectDocument.h"

#import <Foundation/NSString.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSUserDefaults.h>
#import <Foundation/NSProcessInfo.h>

#import <AppKit/NSWorkspace.h>
#import <AppKit/NSDocumentController.h>
#import <AppKit/NSPanel.h>
#import <AppKit/NSOpenPanel.h>
#import <AppKit/NSButton.h>

#import "SourceEditorDocument.h"
#import "ProjectWindowController.h"
#import "ProjectType.h"
#import "ProjectTypeLoader.h"

#import "FileManager.h"
#import "ProjectAttributes.h"
#import "Builder.h"
#import "Launcher.h"
#import "Subprojects.h"
#import "Frameworks.h"
#import "Debugger.h"

#import "FileTemplateChooser.h"
#import "ProjectCreation.h"

#import "FileTypeDetermining.h"

#import "NSArrayAdditions.h"

NSString * const ProjectFilesDidChangeNotification =
  @"ProjectFilesDidChangeNotification",
         * const ProjectSubprojectsDidChangeNotification =
  @"ProjectSubprojectsDidChangeNotification",
         * const ProjectFrameworksDidChangeNotification =
  @"ProjectFrameworksDidChangeNotification";

/**
 * Checks whether the version given as the first argument is equal to
 * or higher than the second version argument. The arguments are arrays
 * of NSStrings or NSNumbers (or something that accepts -intValue),
 * where each element identifies a component of the version string.
 * The elements are ordered from most significant number to least
 * significant.
 *
 * @return YES if the first version is equal to or higher than the latter
 * version.
 */
BOOL
CheckVersion (NSArray * myVersion, NSArray * otherVersion)
{
  int i, n1, n2;

  for (i = 0, n1 = [myVersion count], n2 = [otherVersion count];
    i < n1 || i < n2;
    i++)
    {
      int firstNumber, secondNumber;

      if (i < n1)
        {
          firstNumber = [[myVersion objectAtIndex: i] intValue];
        }
      else
        {
          firstNumber = 0;
        }

      if (i < n2)
        {
          secondNumber = [[otherVersion objectAtIndex: i] intValue];
        }
      else
        {
          secondNumber = 0;
        }

      // if it's greater we don't have to compare the rest anymore
      if (firstNumber > secondNumber)
        {
          return YES;
        }
      // simmilarily, if it's lower the rest doesn't matter anymore
      else if (firstNumber < secondNumber)
        {
          return NO;
        }
    }

  // versions are equal
  return YES;
}

/**
 * Checks whether the provided string is a valid framework (i.e
 * doesn't contain spaces and a couple of other limitations).
 * If the name isn't valid, the user is informed of the fact with
 * an alert panel.
 *
 * @return YES if the string is a valid framework name, NO otherwise.
 */
BOOL
IsValidFrameworkName (NSString * frameworkName)
{
  if ([frameworkName length] != [[frameworkName
    stringByTrimmingCharactersInSet: [NSCharacterSet
    whitespaceAndNewlineCharacterSet]] length])
    {
      NSRunAlertPanel(_(@"Invalid framework name"),
        _(@"A framework name may not contain whitespace characters."),
        nil, nil, nil);

      return NO;
    }

  return YES;
}

/**
 * Gets the category contents array of a category description
 * contained in `supercategoryContentsArray' for a category named
 * `categoryName'.
 *
 * @return The category contents array if the category is found,
 * otherwise nil.
 */
static NSMutableArray *
GetCategoryContentsArray(NSArray * supercategoryContentsArray,
                         NSString * categoryName)
{
  NSEnumerator * e = [supercategoryContentsArray objectEnumerator];
  NSDictionary * entry;
  Class dictionaryClass = [NSDictionary class];

  while ((entry = [e nextObject]) != nil)
    {
      if ([entry isKindOfClass: dictionaryClass] &&
        [[entry objectForKey: @"Name"] isEqualToString: categoryName])
        {
          return [entry objectForKey: @"Contents"];
        }
    }

  return nil;
}

@interface ProjectDocument (Private)

- (NSArray *) contentsOfCategory: (NSString *) category
                     searchFiles: (BOOL) filesFlag;

- (NSMutableArray *) contentsArrayOfCategory: (NSString *) category;

- (BOOL) deleteContentsOfCategory: (NSString *) category;

- (BOOL) existsEntryNamed: (NSString *) entryName
               inCategory: (NSString *) category;

/**
 * Makes a new name from `basename' so that it is unique in `category'
 * and `directory'. E.g. basename = @"New File", then the method will
 * check whether
 *  @"New File"
 *  @"New File 1"
 *  @"New File 2"
 *   ...
 * is unique in the provided category and directory. The first of these
 * names which already is unique will be returned.
 */
- (NSString *) makeNewUniqueNameFromBasename: (NSString *) basename
                               pathExtension: (NSString *) ext
                                  inCategory: (NSString *) category
                                andDirectory: (NSString *) directory;

@end

@implementation ProjectDocument (Private)


/**
 * Gets the entries in category named by `category' (a category path).
 * If `filesFlag' is YES, then the names of the files in the category
 * are returned, otherwise names of sub-categories are returned.
 *
 * @return An array of names of sub-categories (if filesFlag is NO),
 * or files (if filesFlag is YES).
 */
- (NSArray *) contentsOfCategory: (NSString *) category
                     searchFiles: (BOOL) filesFlag
{
  NSMutableArray * array;
  NSEnumerator * e;
  id object;
  Class stringClass = [NSString class];

  e = [[self contentsArrayOfCategory: category] objectEnumerator];
  array = [NSMutableArray array];
  while ((object = [e nextObject]) != nil)
    {
      if (filesFlag == YES)
        {
          if ([object isKindOfClass: stringClass])
            {
              [array addObject: object];
            }
        }
      else
        {
          if (![object isKindOfClass: stringClass])
            {
              [array addObject: [object objectForKey: @"Name"]];
            }
        }
    }

  return [[array copy] autorelease];
}

/**
 * Gets the contents array of category named by `category' (a category
 * path).
 *
 * @return The contents array of the named category or nil if the
 * category cannot be found.
 */
- (NSMutableArray *) contentsArrayOfCategory: (NSString *) category
{
  NSArray * pathComponents;
  NSEnumerator * e;
  NSString * component;
  NSMutableArray * contentsArray;

  category = [category stringByStandardizingPath];
  pathComponents = [category pathComponents];
  e = [pathComponents objectEnumerator];

  for (contentsArray = files, [e nextObject];
       (component = [e nextObject]) != nil;
       contentsArray = GetCategoryContentsArray(contentsArray, component));

  return contentsArray;
}

/**
 * Recursively deletes the files in `category'.
 *
 * @return YES if the operation suceeded, otherwise NO.
 */
- (BOOL) deleteContentsOfCategory: (NSString *) category
{
  NSMutableArray * contentsArray = [self contentsArrayOfCategory: category];
  Class stringClass = [NSString class];
  NSFileManager * fm = [NSFileManager defaultManager];
  int i;

  for (i = [contentsArray count] - 1; i >= 0; i--)
    {
      id object = [contentsArray objectAtIndex: i];

      if ([object isKindOfClass: stringClass])
        {
          NSString * path = [projectType pathToFile: object
                                         inCategory: category];

          if (![fm removeFileAtPath: path handler: nil])
            {
              return NO;
            }
        }
      else
        {
          if (![self deleteContentsOfCategory: [category
            stringByAppendingPathComponent: [object objectForKey: @"Name"]]])
            {
              return NO;
            }
        }

      [contentsArray removeObjectAtIndex: i];
    }

  return YES;
}

- (BOOL) existsEntryNamed: (NSString *) entryName
               inCategory: (NSString *) category
{
  NSEnumerator * e = [[self contentsArrayOfCategory: category]
    objectEnumerator];
  id object;
  Class dictionaryClass = [NSDictionary class];

  while ((object = [e nextObject]) != nil)
    {
      if ([object isKindOfClass: dictionaryClass])
        {
          if ([[object objectForKey: @"Name"] isEqualToString: entryName])
            {
              return YES;
            }
        }
      else
        {
          if ([object isEqualToString: entryName])
            {
              return YES;
            }
        }
    }

  return NO;
}

- (NSString *) makeNewUniqueNameFromBasename: (NSString *) basename
                               pathExtension: (NSString *) ext
                                  inCategory: (NSString *) category
                                andDirectory: (NSString *) directory
{
  NSString * newName;
  unsigned int i;
  NSArray * dirContents;
  NSFileManager * fm = [NSFileManager defaultManager];

  // make @"" behave as if ext = nil
  if ([ext length] == 0)
    {
      ext = nil;
    }

  if (ext != nil)
    {
      newName = [basename stringByAppendingPathExtension: ext];
    }
  else
    {
      newName = basename;
    }

  if ([self existsEntryNamed: newName inCategory: category] == NO &&
      [fm fileExistsAtPath: [directory stringByAppendingPathComponent: newName]])
    {
      return newName;
    }

  dirContents = [fm directoryContentsAtPath: directory];
  i = 1;

  do
    {
      if (ext != nil)
        {
          newName = [NSString stringWithFormat: @"%@ %i.%@", basename, i, ext];
        }
      else
        {
          newName = [NSString stringWithFormat: @"%@ %i", basename, i];
        }

      i++;
    }
  while ([self existsEntryNamed: newName inCategory: category] ||
         [dirContents containsObject: newName]);

  return newName;
}

@end

@implementation ProjectDocument

static NSString * const ProjectManagerVersion = @"0.2";
/**
 * The version numbers of the current ProjectManager release ordered
 * from most significant to least significant.
 */
static NSArray * versionNumbers = nil;

+ (void) initialize
{
  if (versionNumbers == nil)
    {
      ASSIGN(versionNumbers,
             [ProjectManagerVersion componentsSeparatedByString: @"."]);
    }
}

- (void) dealloc
{
  TEST_RELEASE(projectDirectory);
  TEST_RELEASE(projectName);
  TEST_RELEASE(projectTypeID);
  TEST_RELEASE(projectType);

  TEST_RELEASE(files);
  TEST_RELEASE(subprojects);
  TEST_RELEASE(frameworks);

  TEST_RELEASE(builder);
  TEST_RELEASE(launcher);
  TEST_RELEASE(frameworksManager);
  TEST_RELEASE(fileManager);
  TEST_RELEASE(projectAttributes);
  TEST_RELEASE(subprojectsManager);

  TEST_RELEASE(wc);

  [super dealloc];
}

- (BOOL) readFromFile: (NSString *) fileName ofType: (NSString *) fileType
{
  NSDictionary * projectFile;
  NSDictionary * projectData;
  NSString * versionString;

  ASSIGN(projectDirectory, [fileName stringByDeletingLastPathComponent]);

  projectFile = [NSDictionary dictionaryWithContentsOfFile: fileName];
  if (projectFile == nil)
    {
      return NO;
    }

  versionString = [projectFile objectForKey: @"Version"];
  if (versionString != nil)
    {
      if (!CheckVersion(versionNumbers,
                        [versionString componentsSeparatedByString: @"."]))
        {
          if (NSRunAlertPanel(_(@"Newer version of project"),
            _(@"This project was created with a newer version (%@)\n"
              @"of ProjectManager than is this one (%@).\n"
              @"Do you want me to try open the project anyway?"),
            _(@"Yes"), _(@"Cancel"), nil,
            versionString, ProjectManagerVersion) == NSAlertAlternateReturn)
            {
              return NO;
            }
        }
    }

  ASSIGN(projectTypeID, [projectFile objectForKey: @"ProjectType"]);
  if (projectTypeID == nil)
    {
      return NO;
    }

  ASSIGN(projectName, [projectFile objectForKey: @"ProjectName"]);
  if (projectName == nil)
    {
      return NO;
    }

  ASSIGN(files, [[NSArray arrayWithArray:
    [projectFile objectForKey: @"ProjectFiles"]] makeDeeplyMutableEquivalent]);

  ASSIGN(subprojects, [NSMutableDictionary dictionaryWithDictionary:
    [projectFile objectForKey: @"Subprojects"]]);
  ASSIGN(frameworks, [NSMutableArray arrayWithArray:
    [projectFile objectForKey: @"Frameworks"]]);

  projectData = [projectFile objectForKey: @"ProjectData"];

  // get the project type object
  ASSIGN(projectType, [[ProjectTypeLoader shared]
    projectTypeForTypeID: projectTypeID
                 project: self
          infoDictionary: projectData]);
  if (projectType == nil)
    {
      return NO;
    }

  if (![projectType prepareForBuild])
    {
      return NO;
    }

  return YES;
}

- (BOOL) writeToFile: (NSString *) fileName ofType: (NSString *) fileType
{
  NSMutableDictionary * dictionary;
  NSDictionary * projectData;

  dictionary = [NSMutableDictionary dictionary];

  projectData = [projectType infoDictionary];
  if (projectData != nil)
    {
      [dictionary setObject: projectData
                     forKey: @"ProjectData"];
    }
  [dictionary setObject: ProjectManagerVersion forKey: @"Version"];
  [dictionary setObject: projectTypeID forKey: @"ProjectType"];
  [dictionary setObject: projectName forKey: @"ProjectName"];
  [dictionary setObject: files forKey: @"ProjectFiles"];
  [dictionary setObject: frameworks forKey: @"Frameworks"];
  [dictionary setObject: subprojects forKey: @"Subprojects"];

  if (![projectType prepareForBuild])
    {
      return NO;
    }

  return [dictionary writeToFile: fileName
                      atomically: YES];
}

- (void) makeWindowControllers
{
  wc = [[ProjectWindowController alloc]
    initWithWindowNibName: @"Project" ownerDocument: self];

  [self addWindowController: wc];
}

- (NSString *) projectName
{
  return projectName;
}

- (NSString *) projectDirectory
{
  return projectDirectory;
}

- (void) setProjectName: (NSString *) aName
{
  ASSIGN(projectName, aName);
}

- (NSString *) projectTypeID
{
  return projectTypeID;
}

- (id <ProjectType>) projectType
{
  return projectType;
}

- (ProjectModule *) projectAttributes
{
  if (projectAttributes == nil)
    {
      projectAttributes = [[ProjectAttributes alloc] initWithDocument: self];
    }

  return projectAttributes;
}

- (ProjectModule *) fileManager
{
  if (fileManager == nil)
    {
      fileManager = [[FileManager alloc] initWithDocument: self];
    }

  return fileManager;
}

- (ProjectModule *) frameworksManager
{
  if (frameworksManager == nil)
    {
      frameworksManager = [[Frameworks alloc] initWithDocument: self];
    }

  return frameworksManager;
}

- (ProjectModule *) builder
{
  if (builder == nil)
    {
      builder = [[Builder alloc] initWithDocument: self];
    }

  return builder;
}

- (ProjectModule *) launcher
{
  if (launcher == nil)
    {
      launcher = [[Launcher alloc] initWithDocument: self];
    }

  return launcher;
}

- (ProjectModule *) debugger
{
  if (debugger == nil)
    {
      debugger = [[Debugger alloc]
        initWithDocument: self debuggerType: [projectType debuggerType]];
    }

  return debugger;
}

- (ProjectModule *) subprojectsManager
{
  if (subprojectsManager == nil)
    {
      subprojectsManager = [[Subprojects alloc] initWithDocument: self];
    }

  return subprojectsManager;
}

- (BOOL) openFile: (NSString *) filename
       inCategory: (NSString *) category
{
  NSString * path = [projectType pathToFile: filename inCategory: category];

  if ([[[NSUserDefaults standardUserDefaults]
    objectForKey: @"InternalEditorExtensions"] containsObject:
    [path pathExtension]])
    {
      return [[NSDocumentController sharedDocumentController]
        openDocumentWithContentsOfFile: path display: YES] != nil;
    }
  else
    {
      if (![[NSWorkspace sharedWorkspace] openFile: path])
        {
          // Unregistered file extension. See if it is a regular text file,
          // if YES use the internal editor to open it, otherwise
          // report an error
          if ([[[[NSFileManager defaultManager]
            fileAttributesAtPath: path traverseLink: YES]
            fileType]
            isEqualToString: NSFileTypeRegular] &&
            IsTextFile(path))
            {
              SourceEditorDocument * doc;
              NSDocumentController * dc = [NSDocumentController
                sharedDocumentController];

              doc = [dc documentForFileName: path];
              if (doc == nil)
                {
                  // open it and act as if it were a plain text file
                  doc = [[[SourceEditorDocument alloc]
                    initWithContentsOfFile: path ofType: @"txt"]
                    autorelease];
                }
              else
                {
                  [doc showWindows];

                  return YES;
                }

              if (doc == nil)
                {
                  return NO;
                }

              [dc addDocument: doc];
              [doc makeWindowControllers];
              [doc showWindows];

              return YES;
            }
          else
            {
              return NO;
            }
        }
      else
        {
          return YES;
        }
    }
}

- (NSString *) displayName
{
  return projectName;
}

/**
 * Returns all subcategories contained in `category'. Categories
 * are hierarchically organized "folders" of files and other categories
 * and are very simmilar to filesystem paths (e.g. a category name can
 * "/Localized Resource Files/English"), but they need not represent
 * physical organization of files on disk (e.g. the above mentioned
 * category example path maps to "/English.lproj" in a standard
 * application project). If `category' is "/" it means "root category".
 */
- (NSArray *) subcategoriesInCategory: (NSString *) category
{
  return [self contentsOfCategory: category searchFiles: NO];
}

/**
 * Returns all the files contained in `category'. Theoretically,
 * ProjectManager's layout allows for mixing files and further
 * subcategories in a single category (simmilar to mixing files and
 * subfolders in a single folder), but it is not recommend for most
 * setups, as it often creates confusion.
 */
- (NSArray *) filesInCategory: (NSString *) category
{
  return [self contentsOfCategory: category searchFiles: YES];
}

- (NSArray *) allFilesUnderCategory: (NSString *) category
{
  NSMutableArray * array;
  NSEnumerator * e;
  NSString * subcategoryName;

  array = [NSMutableArray array];
  [array addObjectsFromArray: [self filesInCategory: category]];
  e = [[self subcategoriesInCategory: category] objectEnumerator];
  while ((subcategoryName = [e nextObject]) != nil)
    {
      [array addObjectsFromArray: [self allFilesUnderCategory:
        [category stringByAppendingPathComponent: subcategoryName]]];
    }

  return [[array copy] autorelease];
}

- (NSArray *) allCategoriesUnderCategory: (NSString *) category
{
  NSEnumerator * e = [[self subcategoriesInCategory: category]
    objectEnumerator];
  NSString * subcategoryName;
  NSMutableArray * array = [NSMutableArray array];

  while ((subcategoryName = [e nextObject]) != nil)
    {
      NSString * subcategory = [category stringByAppendingPathComponent:
        subcategoryName];

      [array addObject: subcategory];
      [array addObjectsFromArray: [self allCategoriesUnderCategory:
        subcategory]];
    }

  return [[array copy] autorelease];
}

/**
 * Tells the project to add file at `filePath' to the project category
 * `categoryName'. If `linkFlag' is YES, then a symbolic link to the
 * file is created instead. This allows to reference files which are
 * external to the project.
 *
 * @return YES if the file has been successfuly added, NO otherwise.
 */
- (BOOL) addFile: (NSString *) filePath
      toCategory: (NSString *) category
            link: (BOOL) linkFlag
{
  NSString * fileName = [filePath lastPathComponent];
  NSString * destPath = [projectType pathToFile: fileName
                                     inCategory: category];
  NSMutableArray * contentsArray;
  NSFileManager * fm = [NSFileManager defaultManager];

  if ([self existsEntryNamed: fileName
                  inCategory: category])
    {
      if ([[projectType pathToFile: fileName inCategory: category]
        isEqualToString: filePath])
        {
          NSRunAlertPanel(_(@"File already in project"),
            _(@"The file %@ already is in the project."),
            nil, nil, nil, fileName);
          return NO;
        }
      else
        {
          NSRunAlertPanel(_(@"Entry already present"),
            _(@"There already is a file or category in the selected\n"
              @"category named %@. Please rename the file first."),
            nil, nil, nil, fileName);
          return NO;
        }
    }

  if (![destPath isEqualToString: filePath])
    {
      if ([fm fileExistsAtPath: destPath])
        {
          NSRunAlertPanel(_(@"Cannot add file"),
            _(@"A file named %@ already is in the project."),
            nil, nil, nil, fileName);
          return NO;
        }

      if (!CreateDirectoryAndIntermediateDirectories([destPath
        stringByDeletingLastPathComponent]))
        {
          NSRunAlertPanel(_(@"Cannot move file"),
            _(@"Couldn't create a directory for the file."),
            nil, nil, nil);

          return NO;
        }

      // create a copy
      if (linkFlag == NO)
        {
          if ([fm copyPath: filePath toPath: destPath handler: nil] == NO)
            {
              NSRunAlertPanel(_(@"Cannot add file"),
                _(@"Failed to copy file %@ into the project."),
                nil, nil, nil, fileName);

              return NO;
            }
        }
      else
        {
          if ([fm createSymbolicLinkAtPath: destPath
                               pathContent: filePath] == NO)
            {
              NSRunAlertPanel(_(@"Cannot add file"),
                _(@"Failed to create a link to the file %@ from the project."),
                nil, nil, nil, fileName);

              return NO;
            }
        }
    }
  // The file already is in the correct location, but make sure it
  // isn't already included in some other project category - this could
  // create inconsistencies with copying/moving files around categories.
  else
    {
      NSEnumerator * e = [[self allCategoriesUnderCategory: @"/"]
        objectEnumerator];
      NSString * category;
      NSString * filename = [filePath lastPathComponent];

      while ((category = [e nextObject]) != nil)
        {
          if ([[projectType pathToFile: filename inCategory: category]
            isEqualToString: filePath] &&
            [[self contentsArrayOfCategory: category] containsObject: filename])
            {
              NSRunAlertPanel(_(@"File already in project"),
                _(@"The file already exists in the project, in category %@."),
                nil, nil, nil, category);

              return NO;
            }
        }
    }

  contentsArray = [self contentsArrayOfCategory: category];
  [contentsArray addObject: fileName];

  [self updateChangeCount: NSChangeDone];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

- (BOOL) addCategory: (NSString *) categoryName
          toCategory: (NSString *) category
{
  NSMutableArray * contentsArray = [self contentsArrayOfCategory: category];
  NSEnumerator * e = [contentsArray objectEnumerator];
  NSDictionary * subcategory;

  while ((subcategory = [e nextObject]) != nil)
    {
      if ([subcategory isKindOfClass: [NSDictionary class]] &&
          [[subcategory objectForKey: @"Name"] isEqualToString: categoryName])
        {
          NSRunAlertPanel(_(@"Category already exists"),
            _(@"A category of the name already exists."),
            nil, nil, nil);

          return NO;
        }
    }

  [contentsArray addObject: [NSMutableDictionary dictionaryWithObjectsAndKeys:
    categoryName, @"Name",
    [NSMutableArray array], @"Contents",
    nil]];

  [self updateChangeCount: NSChangeDone];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

/**
 * Tells the project type to remove the file named by `fileName'
 * from category `categoryName'. If `flag' is YES, the file
 * should be deleted from disk as well.
 */
- (BOOL) removeFile: (NSString *) fileName
       fromCategory: (NSString *) categoryName
             delete: (BOOL) flag;
{
  NSMutableArray * array;

  if (flag)
    {
      NSFileManager * fm = [NSFileManager defaultManager];
      NSString * path = [projectType pathToFile: fileName
                                     inCategory: categoryName];

      if ([fm removeFileAtPath: path handler: nil] == NO)
        {
          NSRunAlertPanel(_(@"Couldn't delete file"),
            _(@"I failed to delete file %@. Aborting delete operation."),
            nil, nil, nil, fileName);
          return NO;
        }
    }

  array = [self contentsArrayOfCategory: categoryName];
  NSAssert(array != nil, _(@"Nil category contents array!"));
  [array removeObject: fileName];

  [self updateChangeCount: NSChangeDone];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

/**
 * Removes `categoryName' from `category'.
 *
 * @return YES if the operation succeeded, otherwise NO.
 */
- (BOOL) removeCategory: (NSString *) categoryName
           fromCategory: (NSString *) category
            deleteFiles: (BOOL) flag
{
  NSMutableArray * array;
  NSEnumerator * e;
  NSDictionary * categoryDescription;

  if (flag)
    {
      [self updateChangeCount: NSChangeDone];

      if (![self deleteContentsOfCategory: [category
        stringByAppendingPathComponent: categoryName]])
        {
          NSRunAlertPanel(_(@"Failed to delete category"),
            _(@"Couldn't delete category %@ -- a file could not be\n"
              @"deleted. Please make sure you have write permissions\n"
              @"to the project."),
            nil, nil, nil, categoryName);

          return NO;
        }
    }

  array = [self contentsArrayOfCategory: category];
  e = [array objectEnumerator];

  while ((categoryDescription = [e nextObject]) != nil)
    {
      if ([categoryDescription isKindOfClass: [NSDictionary class]] &&
        [[categoryDescription objectForKey: @"Name"]
        isEqualToString: categoryName])
        {
          [array removeObject: categoryDescription];

          break;
        }
    }

  [self updateChangeCount: NSChangeDone];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

/**
 * Tells the project type to rename file called `fileName' which is
 * in `category' to `newName'.
 */
- (BOOL) renameFile: (NSString *) fileName
         inCategory: (NSString *) category
             toName: (NSString *) newName
{
  NSFileManager * fm = [NSFileManager defaultManager];
  NSString * srcPath = [projectType pathToFile: fileName inCategory: category];
  NSString * destPath = [projectType pathToFile: newName
                                     inCategory: category];
  NSMutableArray * contentsArray;
  id object;
  NSEnumerator * e;

  // check for already present names
  if ([fm fileExistsAtPath: destPath])
    {
      NSRunAlertPanel(_(@"File already exists"),
        _(@"A file named like that already exists in the project."),
        nil, nil, nil);

      return NO;
    }

  contentsArray = [self contentsArrayOfCategory: category];

  e = [contentsArray objectEnumerator];
  while ((object = [e nextObject]) != nil)
    {
      NSString * name;

      if ([object isKindOfClass: [NSDictionary class]])
        {
          name = [object objectForKey: @"Name"];
        }
      else
        {
          name = object;
        }

      if ([name isEqualToString: newName])
        {
          NSRunAlertPanel(_(@"Name already exists"),
            _(@"A category or file named like that\n"
              @"already exists in the current category."),
            nil, nil, nil);

          return NO;
        }
    }

  if ([fm movePath: srcPath toPath: destPath handler: nil] == NO)
    {
      NSRunAlertPanel(_(@"Failed to rename file"),
        _(@"Couldn't rename the file."),
        nil, nil, nil);

      return NO;
    }

  [contentsArray addObject: newName];
  [contentsArray removeObject: fileName];

  [self updateChangeCount: NSChangeDone];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

/**
 * Tells the project to rename `oldCategoryName' to `newCategoryName'.
 * The named `oldCategoryName' and `newCategoryName' must be direct
 * names of the category, not paths to it. The category is assumed to
 * exist in `category' (which is a category path).
 */
- (BOOL) renameCategory: (NSString *) oldCategoryName
             inCategory: (NSString *) category
                 toName: (NSString *) newCategoryName
{
  NSEnumerator * e;
  id object;
  NSMutableDictionary * description;
  NSMutableArray * contentsArray = [self contentsArrayOfCategory: category];

  // check for already present names
  e = [contentsArray objectEnumerator];
  while ((object = [e nextObject]) != nil)
    {
      NSString * name;

      if ([object isKindOfClass: [NSDictionary class]])
        name = [object objectForKey: @"Name"];
      else
        name = object;

      if ([name isEqualToString: newCategoryName])
        {
          NSRunAlertPanel(_(@"Name already exists"),
            _(@"A category or file named like that\n"
              @"already exists in the current category."),
            nil, nil, nil);

          return NO;
        }
    }

  e = [contentsArray objectEnumerator];
  while ((description = [e nextObject]) != nil)
    {
      if ([description isKindOfClass: [NSDictionary class]] &&
        [[description objectForKey: @"Name"] isEqualToString: oldCategoryName])
        {
          [description setObject: newCategoryName forKey: @"Name"];
          break;
        }
    }

  [self updateChangeCount: NSChangeDone];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

// N.B. Copying/Moving entire categories isn't supported because of the
// following reasons:
//  - it adds significant complexity - which isn't all that much of a
//    problem, as I already had the feature implemented, however:
//  - it messes quite around with project layout. Because categories
//    aren't directories, several categories can share the same file
//    namespace and the user could be confused with alert panels like
//    "can't move/copy category, because of namespace clashes ..." etc.
//  - if you want to move/copy a category, create a destination category
//    of the same name and copy/move the files of the original category
//    there by simply dragging them. This creates a lot less confusion.

/**
 * Copies `fileName' from `sourceCategory' to `destinationCategory',
 * duplicating the underlying disk file. If a file already exists at
 * the destination location, the operation fails.
 *
 * @return YES if the operation succeeded, NO otherwise.
 */
- (BOOL) copyFile: (NSString *) fileName
       inCategory: (NSString *) sourceCategory
       toCategory: (NSString *) destinationCategory
{
  NSString * srcPath, * destPath;
  NSFileManager * fm = [NSFileManager defaultManager];

  if ([sourceCategory isEqualToString: destinationCategory])
    {
      return YES;
    }

  srcPath = [projectType pathToFile: fileName inCategory: sourceCategory];
  destPath = [projectType pathToFile: fileName inCategory: destinationCategory];

  if ([srcPath isEqualToString: destPath])
    {
      NSRunAlertPanel(_(@"Can't copy file"),
        _(@"Can't copy file %@: the destination category's files reside\n"
          @"in the same directory as the source category's."),
        nil, nil, nil, fileName);

      return NO;
    }

  if ([fm fileExistsAtPath: destPath])
    {
      NSRunAlertPanel(_(@"Can't copy file"),
        _(@"A file named %@ already exists in the\n"
          @"destination category's directory."),
        nil, nil, nil, fileName);

      return NO;
    }

  if ([self existsEntryNamed: fileName inCategory: destinationCategory])
    {
      NSRunAlertPanel(_(@"Can't copy file"),
        _(@"A file or category named %@ already exists\n"
          @"in the destination category."), 
        nil, nil, nil, fileName);

      return NO;
    }

  if (!CreateDirectoryAndIntermediateDirectories([destPath
    stringByDeletingLastPathComponent]))
    {
      NSRunAlertPanel(_(@"Can't copy file"),
        _(@"Couldn't create a directory for the file."),
        nil, nil, nil);

      return NO;
    }

  if (![fm copyPath: srcPath toPath: destPath handler: nil])
    {
      NSRunAlertPanel(_(@"Can't copy file"),
        _(@"Failed to copy the file %@."),
        nil, nil, nil, fileName);

      return NO;
    }

  [[self contentsArrayOfCategory: destinationCategory] addObject: fileName];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

/**
 * Tells the project to move `fileName' from `sourceCategory' to
 * `destinationCategory'. If a file already exists at the destination
 * location, the operation fails.
 *
 * @return YES if the operation succeeds, NO otherwise.
 */
- (BOOL) moveFile: (NSString *) fileName
       inCategory: (NSString *) sourceCategory
       toCategory: (NSString *) destinationCategory
{
  NSString * srcPath, * destPath;
  NSFileManager * fm;
  NSMutableArray * srcArray, * destArray;

  if ([sourceCategory isEqualToString: destinationCategory])
    {
      return YES;
    }

  srcPath = [projectType pathToFile: fileName inCategory: sourceCategory];
  destPath = [projectType pathToFile: fileName inCategory: destinationCategory];
  fm = [NSFileManager defaultManager];
  srcArray = [self contentsArrayOfCategory: sourceCategory];
  destArray = [self contentsArrayOfCategory: destinationCategory];

  if (![srcPath isEqualToString: destPath])
    {
      if ([fm fileExistsAtPath: destPath])
        {
          NSRunAlertPanel(_(@"Cannot move file"),
            _(@"A file named %@ already exists\n"
              @"in the destination category's directory."),
            nil, nil, nil, fileName);

          return NO;
        }

      if (!CreateDirectoryAndIntermediateDirectories([destPath
        stringByDeletingLastPathComponent]))
        {
          NSRunAlertPanel(_(@"Cannot move file"),
            _(@"Couldn't create a directory for the file."),
            nil, nil, nil);

          return NO;
        }

      if (![fm movePath: srcPath toPath: destPath handler: nil])
        {
          NSRunAlertPanel(_(@"Cannot move file"),
            _(@"Failed to move the file to the destination directory."),
            nil, nil, nil);

          return NO;
        }
    }

  [destArray addObject: fileName];
  [srcArray removeObject: fileName];

  [self updateChangeCount: NSChangeDone];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

/**
 * Creates a symbolic link to `fileName' contained in `sourceCategory',
 * from `destinationCategory'. If a file already exists at the destination
 * location where the link is supposed to be, the operation fails.
 *
 * @return YES if the operation succeeded, NO otherwise.
 */
- (BOOL) linkFile: (NSString *) fileName
       inCategory: (NSString *) sourceCategory
       toCategory: (NSString *) destinationCategory
{
  NSString * srcPath, * destPath;
  NSFileManager * fm = [NSFileManager defaultManager];
  BOOL result;

  if ([sourceCategory isEqualToString: destinationCategory])
    {
      return YES;
    }

  srcPath = [projectType pathToFile: fileName inCategory: sourceCategory];
  destPath = [projectType pathToFile: fileName inCategory: destinationCategory];

  if ([srcPath isEqualToString: destPath])
    {
      NSRunAlertPanel(_(@"Can't link file"),
        _(@"Can't link file %@: the destination category's files reside\n"
          @"in the same directory as the source category's."),
        nil, nil, nil, fileName);

      return NO;
    }

  if ([fm fileExistsAtPath: destPath])
    {
      NSRunAlertPanel(_(@"Can't link file"),
        _(@"A file named %@ already exists in the\n"
          @"destination category's directory."),
        nil, nil, nil, fileName);

      return NO;
    }

  if ([self existsEntryNamed: fileName inCategory: destinationCategory])
    {
      NSRunAlertPanel(_(@"Can't link file"),
        _(@"A file or category named %@ already exists\n"
          @"in the destination category."), 
        nil, nil, nil, fileName);

      return NO;
    }

  if (!CreateDirectoryAndIntermediateDirectories([destPath
    stringByDeletingLastPathComponent]))
    {
      NSRunAlertPanel(_(@"Can't link file"),
        _(@"Couldn't create a directory for the link."),
        nil, nil, nil);

      return NO;
    }

  // link to the original file if the source is a symbolic link
  if ([[[fm fileAttributesAtPath: srcPath traverseLink: NO] fileType]
    isEqualToString: NSFileTypeSymbolicLink])
    {
      result = [fm createSymbolicLinkAtPath: destPath
        pathContent: [fm pathContentOfSymbolicLinkAtPath: srcPath]];
    }
  else
    {
      result = [fm createSymbolicLinkAtPath: destPath
                                pathContent: srcPath];
    }

  if (result == NO)
    {
      NSRunAlertPanel(_(@"Can't link file"),
        _(@"Failed to link the file %@."),
        nil, nil, nil, fileName);

      return NO;
    }

  [[self contentsArrayOfCategory: destinationCategory] addObject: fileName];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFilesDidChangeNotification object: self];

  return YES;
}

- (NSImage *) iconForCategory: (NSString *) category
{
  return [projectType iconForCategory: category];
}

- (BOOL) validateMenuItem: (id <NSMenuItem>) anItem
{
  SEL action = [anItem action];

  if (sel_eq(action, @selector(newEmptyFile:)) ||
      sel_eq(action, @selector(newFileFromTemplate:)) ||
      sel_eq(action, @selector(addFiles:)) ||
      sel_eq(action, @selector(deleteFiles:)) ||
      sel_eq(action, @selector(newCategory:)) ||
      sel_eq(action, @selector(deleteCategory:)))
    {
      NSString * selectedCategory;
      NSArray * selectedFiles;

      if (!([[projectType class] projectCapabilities] & FilesProjectCapability)
        || [wc currentTab] != FilesTab)
        {
          return NO;
        }

      selectedCategory = [(FileManager *) [self fileManager]
        containingCategory];
      selectedFiles = [(FileManager *) [self fileManager]
        selectedFiles];

      if (sel_eq(action, @selector(newEmptyFile:)) ||
          sel_eq(action, @selector(newFileFromTemplate:)) ||
          sel_eq(action, @selector(addFiles:)))
        {
          return [projectType canAddFilesToCategory: selectedCategory];
        }
      else if (sel_eq(action, @selector(deleteFiles:)))
        {
          return (selectedFiles != nil);
        }
      else if (sel_eq(action, @selector(newCategory:)))
        {
          return [projectType canAddCategoriesToCategory: selectedCategory];
        }
        // @selector(deleteCategory:)
      else
        {
          NSArray * selectedCategories = [(FileManager *) [self fileManager]
            selectedCategories];

          if ([selectedCategories count] > 1)
            {
              return NO;
            }
          else if (![[selectedCategories objectAtIndex: 0]
            isEqualToString: @"/"])
            {
              return [projectType canDeleteCategory: [selectedCategories
                objectAtIndex: 0]];
            }
          else
            {
              return NO;
            }
        }
    }
  else if (sel_eq(action, @selector(openSubprojectAction:)) ||
           sel_eq(action, @selector(addSubprojectAction:)) ||
           sel_eq(action, @selector(removeSubprojectAction:)) ||
           sel_eq(action, @selector(newSubprojectAction:)))
    {
      NSString * selectedSubproject;

      if (!([[projectType class] projectCapabilities] &
        SubprojectsProjectCapability) ||
        [wc currentTab] != SubprojectsTab)
        {
          return NO;
        }

      selectedSubproject = [(Subprojects *) [self subprojectsManager]
        selectedSubproject];

      if (sel_eq(action, @selector(openSubprojectAction:)) ||
          sel_eq(action, @selector(removeSubprojectAction:)))
        {
          return (selectedSubproject != nil);
        }
      else
        {
          return YES;
        }
    }
  else if (sel_eq(action, @selector(saveDocumentTo:)) ||
           sel_eq(action, @selector(saveDocumentAs:)))
    {
      return NO;
    }
  else if (sel_eq(action, @selector(viewFiles:)))
    {
      return (projectCapabilities & FilesProjectCapability);
    }
  else if (sel_eq(action, @selector(viewFrameworks:)))
    {
      return (projectCapabilities & FrameworksProjectCapability);
    }
  else if (sel_eq(action, @selector(viewBuild:)))
    {
      return (projectCapabilities & BuildProjectCapability);
    }
  else if (sel_eq(action, @selector(viewLaunch:)))
    {
      return (projectCapabilities & RunProjectCapability);
    }
  else if (sel_eq(action, @selector(viewDebug:)))
    {
      return (projectCapabilities & DebugProjectCapability);
    }
  else if (sel_eq(action, @selector(viewSubprojects:)))
    {
      return (projectCapabilities & SubprojectsProjectCapability);
    }
  else
    {
      return [super validateMenuItem: (NSMenuItem *) anItem];
    }
}

- (BOOL) addSubproject: (NSString *) aProject
{
  NSString * projectFilePath;
  NSString * sourceDirectory = [aProject stringByDeletingLastPathComponent];
  NSString * destinationDirectory;
  NSString * subprojectName = [[aProject lastPathComponent]
    stringByDeletingPathExtension];
  ProjectDocument * doc;

  if ([subprojects objectForKey: subprojectName] != nil)
    {
      NSRunAlertPanel(_(@"Subproject already exists"),
        _(@"The project already contains a subproject named %@."),
        nil, nil, nil, subprojectName);

      return NO;
    }

  destinationDirectory = [[projectType pathToSubprojectsDirectory]
    stringByAppendingPathComponent: [sourceDirectory lastPathComponent]];
  if (![destinationDirectory isEqualToString: sourceDirectory])
    {
      NSFileManager * fm = [NSFileManager defaultManager];

      if (!CreateDirectoryAndIntermediateDirectories(destinationDirectory) ||
        ![fm copyPath: sourceDirectory
               toPath: destinationDirectory
              handler: nil])
        {
          NSRunAlertPanel(_(@"Failed to add subproject"),
            _(@"Couldn't copy %@ to the project's subproject directory."),
            nil, nil, nil, sourceDirectory);

          return NO;
        }
    }

  // open the subproject's document invisibly, save it and close it again,
  // in order to make the project regenerate it's makefiles
  projectFilePath = [destinationDirectory stringByAppendingPathComponent:
    [aProject lastPathComponent]];
  doc = [[NSDocumentController sharedDocumentController]
    makeDocumentWithContentsOfFile: projectFilePath ofType: @"pmproj"];
  if (doc == nil)
    {
      NSRunAlertPanel(_(@"Failed to add subproject"),
        _(@"Couldn't open the subproject - perhaps it is corrupt?"),
        nil, nil, nil);

      return NO;
    }
  [doc saveDocument: nil];
  [doc close];

  [subprojects setObject: projectFilePath
                  forKey: [destinationDirectory lastPathComponent]];

  [self updateChangeCount: NSChangeDone];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectSubprojectsDidChangeNotification
                  object: self];

  return YES;
}

- (BOOL) openSubproject: (NSString *) subprojectName
{
  NSString * subprojectPath;

  subprojectPath = [subprojects objectForKey: subprojectName];

  return ([[NSDocumentController sharedDocumentController]
    openDocumentWithContentsOfFile: subprojectPath display: YES] != nil);
}

- (BOOL) removeSubproject: (NSString *) subprojectName
{
  ProjectDocument * doc;
  NSString * subprojectDirectory,
           * subprojectFileName;

  subprojectFileName = [subprojects objectForKey: subprojectName];
  subprojectDirectory = [subprojectFileName stringByDeletingLastPathComponent];

  switch (NSRunAlertPanel(_(@"Really remove subproject?"),
    _(@"Are you really sure you want to remove %@ from the subprojects?"),
    _(@"Yes, but keep files in disk"),
    _(@"Yes and DELETE it from disk"),
    _(@"Cancel"), subprojectName))
    {
    case NSAlertAlternateReturn:
      {
        NSFileManager * fm = [NSFileManager defaultManager];

        if (![fm removeFileAtPath: subprojectDirectory
                          handler: nil])
          {
            NSRunAlertPanel(_(@"Cannot remove subproject"),
              _(@"Couldn't delete the subproject's files from disk."),
              nil, nil, nil);

            return NO;
          }
      }
      break;
    case NSAlertOtherReturn:
      return NO;
    default:
      break;
    }

  doc = [[NSDocumentController sharedDocumentController]
    documentForFileName: subprojectFileName];
  if (doc != nil)
    {
      [doc close];
    }

  [subprojects removeObjectForKey: subprojectName];
  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectSubprojectsDidChangeNotification
                  object: self];
  [self updateChangeCount: NSChangeDone];

  return YES;
}

/**
 * Retrieves subproject information.
 *
 * @return A dictionary where keys are subproject names and values are
 *      paths to the respective subproject's project file.
 */
- (NSDictionary *) subprojects
{
  return [[subprojects copy] autorelease];
}

/**
 * Adds files in `filenames' to the project category `category', linking
 * them if `linkFlag' is YES, otherwise copying them.
 *
 * @return YES if the operation succeeded and all files have been
 *      added, otherwise NO.
 */
- (BOOL) addFiles: (NSArray *) filenames
       toCategory: (NSString *) category
             link: (BOOL) linkFlag
{
  NSEnumerator * e = [filenames objectEnumerator];
  NSString * filename;

  while ((filename = [e nextObject]) != nil)
    {
      if (![self addFile: filename toCategory: category link: linkFlag])
        {
          return NO;
        }
    }

  return YES;
}

- (BOOL) addFramework: (NSString *) aFramework
{
  if (!IsValidFrameworkName(aFramework))
    {
      return NO;
    }

  // don't allow for duplicate frameworks
  if ([frameworks containsObject: aFramework] ||
    [[projectType fixedFrameworks] containsObject: aFramework])
    {
      NSRunAlertPanel(_(@"Framework already present"),
        _(@"There already is a framework named %@ in the frameworks list."),
        nil, nil, nil, aFramework);

      return NO;
    }
  [frameworks addObject: aFramework];
  [frameworks sortUsingSelector: @selector(caseInsensitiveCompare:)];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFrameworksDidChangeNotification
                  object: self];
  [self updateChangeCount: NSChangeDone];

  return YES;
}

- (void) removeFramework: (NSString *) aFramework
{
  [frameworks removeObject: aFramework];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFrameworksDidChangeNotification
                  object: self];
  [self updateChangeCount: NSChangeDone];
}

- (BOOL) renameFramework: (NSString *) oldName toName: (NSString *) newName
{
  if (!IsValidFrameworkName(newName))
    {
      return NO;
    }
  // don't allow for duplicate frameworks
  if ([frameworks containsObject: newName] ||
    [[projectType fixedFrameworks] containsObject: newName])
    {
      NSRunAlertPanel(_(@"Framework already present"),
        _(@"There already is a framework named %@ in the frameworks list."),
        nil, nil, nil, newName);

      return NO;
    }

  [frameworks replaceObjectAtIndex: [frameworks indexOfObject: oldName]
                        withObject: newName];
  [frameworks sortUsingSelector: @selector(caseInsensitiveCompare:)];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: ProjectFrameworksDidChangeNotification
                  object: self];
  [self updateChangeCount: NSChangeDone];

  return YES;
}

- (NSArray *) frameworks
{
  return [[frameworks copy] autorelease];
}

/**
 * Returns a list of ``fixed'' project frameworks, i.e. the ones that
 * are always present and can never be removed or renamed.
 */
- (NSArray *) fixedFrameworks
{
  return [projectType fixedFrameworks];
}

- (void) addFiles: sender
{
  NSString * category = [(FileManager *)[self fileManager] containingCategory];
  NSOpenPanel * op = [NSOpenPanel openPanel];
  static NSButton * linkButton = nil;

  if (linkButton == nil)
    {
      linkButton = [[NSButton alloc] initWithFrame: NSMakeRect(0, 0, 100, 18)];
      [linkButton setButtonType: NSSwitchButton];
      [linkButton setTitle: _(@"Link Files")];
      [linkButton sizeToFit];
    }

  [linkButton setState: NO];

  [op setTitle: _(@"Choose file(s) to add to the project")];
  [op setAllowsMultipleSelection: YES];
  [op setAccessoryView: linkButton];

  if ([op runModalForTypes: [projectType permissibleFileTypesInCategory:
    category]] == NSOKButton)
    {
      [self addFiles: [op filenames]
          toCategory: category
                link: [linkButton state]];
    }

  [op setAccessoryView: nil];
}

- (void) newEmptyFile: sender
{
  NSString * selectedCategory = [(FileManager *) [self fileManager]
    containingCategory];
  NSString * newFileName = [self
    makeNewUniqueNameFromBasename: _(@"New File")
                    pathExtension: nil
                       inCategory: selectedCategory
                     andDirectory: [projectType pathToFile: nil
                                                inCategory: selectedCategory]];
  NSString * filePath = [projectType pathToFile: newFileName
                                     inCategory: selectedCategory];

  if (![[NSFileManager defaultManager]
    createFileAtPath: filePath contents: nil attributes: nil])
    {
      NSRunAlertPanel(_(@"Can't create new file"),
        _(@"Couldn't create a new file in the project. Perhaps\n"
          @"you don't have write permissions for the project."),
        nil, nil, nil);

      return;
    }

  [self addFile: filePath toCategory: selectedCategory link: NO];
  [(FileManager *) [self fileManager]
    selectAndEditNameAtPath: [selectedCategory
    stringByAppendingPathComponent: newFileName]];
}

- (void) newFileFromTemplate: sender
{
  NSString * selectedCategory = [(FileManager *) [self fileManager]
    containingCategory];
  FileTemplateChooser * ftc = [FileTemplateChooser shared];
  NSString * templatesDirectory = [projectType
    pathToFileTemplatesDirectoryForCategory: selectedCategory];

chooseTemplate:
  if ([ftc runModalWithTemplatesDirectory: templatesDirectory] == NSOKButton)
    {
      NSString * templatePath = [ftc templatePath];
      NSString * newName = [ftc fileName];
      NSString * destinationLocation;
      NSFileManager * fm = [NSFileManager defaultManager];

      destinationLocation = [projectType
        pathToFile: newName inCategory: selectedCategory];

      if ([fm fileExistsAtPath: destinationLocation])
        {
          NSRunAlertPanel(_(@"File already present"),
            _(@"There already is a file named %@ in the selected "
              @"category's directory."),
            nil, nil, nil, newName);

          goto chooseTemplate;
        }

      if ([self existsEntryNamed: newName inCategory: selectedCategory])
        {
          NSRunAlertPanel(_(@"Category already present"),
            _(@"There already is a category named %@ in the selected "
              @"category."),
            nil, nil, nil, newName);

          goto chooseTemplate;
        }

      if (ImportProjectFile(templatePath,
                            destinationLocation,
                            projectName))
        {
          [[self contentsArrayOfCategory: selectedCategory]
            addObject: newName];
          [self updateChangeCount: NSChangeDone];
          [[NSNotificationCenter defaultCenter]
            postNotificationName: ProjectFilesDidChangeNotification
                          object: self];

          [(FileManager *) [self fileManager]
            selectAndEditNameAtPath: [selectedCategory
            stringByAppendingPathComponent: newName]];
        }
      else
        {
          NSRunAlertPanel(_(@"Couldn't add file"),
            _(@"Failed to copy the file into the project."),
            nil, nil, nil);
        }
    }
}

- (void) deleteFiles: sender
{
  NSArray * filenames = [(FileManager *) [self fileManager] selectedFiles];
  NSString * category = [(FileManager *) [self fileManager] containingCategory];
  BOOL delete = NO;
  NSEnumerator * e = [filenames objectEnumerator];
  NSString * filename;

  switch (NSRunAlertPanel(_(@"Delete file(s)?"),
    _(@"Do you really want to delete the selected file(s) from category %@?"),
    _(@"Yes, from project AND disk"),
    _(@"Yes, from project only"),
    _(@"Cancel"), category))
    {
    case NSAlertDefaultReturn:
      delete = YES;
      break;
    case NSAlertOtherReturn:
      return;
    }

  while ((filename = [e nextObject]) != nil)
    {
      if (![self removeFile: filename
               fromCategory: category
                     delete: delete])
        {
          return;
        }
    }
}

- (void) newCategory: sender
{
  NSString * selectedCategory = [(FileManager *) [self fileManager]
    containingCategory];
  NSString * newCategoryName = [self
    makeNewUniqueNameFromBasename: _(@"New Category")
                    pathExtension: nil
                       inCategory: selectedCategory
                     andDirectory: nil];

  [self addCategory: newCategoryName toCategory: selectedCategory];
  [(FileManager *) [self fileManager]
    selectAndEditNameAtPath: [selectedCategory
    stringByAppendingPathComponent: newCategoryName]];
}

- (void) deleteCategory: sender
{
  NSArray * selectedCategories;
  NSString * selectedCategory;
  NSString * parentCategory;

  selectedCategories = [(FileManager *) [self fileManager] selectedCategories];
  if ([selectedCategories count] == 1)
    {
      selectedCategory = [selectedCategories objectAtIndex: 0];
    }
  else
    {
      NSRunAlertPanel(_(@"Several categories are selected"),
        _(@"Cowardly refusing to delete several categories at once."),
        nil, nil, nil);

      return;
    }

  parentCategory = [selectedCategory stringByDeletingLastPathComponent];
  selectedCategory = [selectedCategory lastPathComponent];

  switch (NSRunAlertPanel(_(@"Delete category?"),
    _(@"Are you sure you want to delete category %@?"),
    _(@"Yes and delete files from disk"),
    _(@"Yes, but keep files on disk"),
    _(@"Cancel"), selectedCategory))
    {
    case NSAlertDefaultReturn:
      [self removeCategory: selectedCategory
              fromCategory: parentCategory
               deleteFiles: YES];
      break;
    case NSAlertAlternateReturn:
      [self removeCategory: selectedCategory
              fromCategory: parentCategory
               deleteFiles: NO];
      break;
    }
}

- (void) openSubprojectAction: sender
{
  NSString * selectedSubproject;

  selectedSubproject = [(Subprojects *) [self subprojectsManager]
    selectedSubproject];
  if (selectedSubproject != nil)
    {
      [self openSubproject: selectedSubproject];
    }
}

- (void) addSubprojectAction: sender
{
  NSOpenPanel * op = [NSOpenPanel openPanel];

  [op setCanChooseDirectories: NO];
  [op setCanChooseFiles: YES];
  [op setAllowsMultipleSelection: NO];
  if ([op runModalForTypes: [NSArray arrayWithObject: @"pmproj"]] ==
    NSOKButton)
    {
      [self addSubproject: [op filename]];
    }
}

- (void) removeSubprojectAction: sender
{
  NSString * selectedSubproject;

  selectedSubproject = [(Subprojects *) [self subprojectsManager]
    selectedSubproject];
  if (selectedSubproject != nil)
    {
      [self removeSubproject: selectedSubproject];
    }
}

- (void) newSubprojectAction: sender
{
  NSDictionary * setup;
  NSString * path, * subprojectName, * templatePath;

  setup = GetNewProjectSetup(NO);
  if (setup == nil)
    {
      return;
    }

  subprojectName = [setup objectForKey: @"ProjectName"];
  templatePath = [setup objectForKey: @"ProjectTemplate"];
  path = [[projectType pathToSubprojectsDirectory]
    stringByAppendingPathComponent: subprojectName];

  if ([subprojects objectForKey: subprojectName] != nil)
    {
      NSRunAlertPanel(_(@"Subproject already exists"),
        _(@"The project already contains a subproject named %@."),
        nil, nil, nil, subprojectName);

      return;
    }

  if (!CreateNewProject(path, subprojectName, templatePath))
    {
      return;
    }

  [self addSubproject: [path stringByAppendingPathComponent:
    [subprojectName stringByAppendingPathExtension: @"pmproj"]]];
}

@end
